# WARP.md

This file provides guidance to WARP (warp.dev) when working with code in this repository.

## Project Overview

This is a Laravel package (`bongo/estimate`) that provides an estimation/quoting system. The package manages estimates with geographic service areas, tiered pricing plans, and automated calculations for recurring service costs. It includes map integration (Google Maps) and supports multi-step estimate workflows.

## Requirements

- PHP 8.2+
- Laravel 10+ or Laravel 11+
- Bongo Framework 3.0+
- Google Maps API key (for map features)

## Common Commands

### Testing
```bash
# Run all tests (no coverage)
composer test

# Run tests with coverage (requires Xdebug)
composer test:coverage

# Run parallel tests
composer test:parallel

# Run a single test
vendor/bin/phpunit --filter TestMethodName
```

### Code Quality
```bash
# Run static analysis (PHPStan)
composer analyse

# Format code (Laravel Pint)
composer format

# Build the package
composer build

# Start test server
composer start
```

### Development Setup
```bash
# Clear testbench cache
composer clear

# Prepare testbench environment
composer prepare
```

## Code Architecture

### Core Domain Models

The package revolves around a hierarchical structure:

**EstimateService** (top-level category, e.g., "Lawn Care")
  └─ **EstimatePlan** (pricing tiers, e.g., "Basic", "Enhanced", "Premier")
      └─ **EstimatePlanItem** (area-based pricing rules, e.g., "50m² = £28", "100m² = £35")

**Estimate** (customer quote)
  └─ **EstimateItem** (specific service/plan selection with location)
      └─ **EstimateItemPrice** (calculated pricing for selected plan)

**EstimateLocation** (service coverage areas with radius)

### Key Relationships

- `Estimate` has many `EstimateItem`s (a customer can get quotes for multiple properties/services)
- `EstimateItem` belongs to an `EstimateService` and `EstimatePlan`, and has many `EstimateItemPrice`s
- `EstimateService` has many `EstimatePlan`s (different pricing tiers)
- `EstimatePlan` has many `EstimatePlanItem`s (area-based pricing breakpoints)
- All major models use UUIDs, soft deletes, and status tracking

### Directory Structure

- `src/Actions/` - Discrete business logic operations (e.g., `CalculateDistance`, `FindEstimate`, `GetPrice`)
- `src/Calculators/` - Complex calculation logic (e.g., `EstimateItemPriceCalculator` handles tax, cost per m², monthly pricing)
- `src/Models/` - Eloquent models with traits for common functionality
- `src/Traits/` - Reusable model behaviors (`HasAddress`, `HasStatus`, `HasAreas`, `HasContact`, etc.)
- `src/Events/` - Domain events fired on model changes (Created/Updated/Deleted for each entity)
- `src/Http/Controllers/` - Backend (admin) and frontend controllers
- `src/Routes/` - Route definitions (`backend.php`, `frontend.php`)
- `src/Maps/` - Google Maps integration (static maps, radius maps, polyline encoding)
- `src/Services/` - External service integrations (Google Geocoder)
- `src/Views/` - Blade views
- `src/Migrations/` - Database migrations

### Important Traits

Models extensively use traits for composition:
- `HasStatus` - Status tracking (draft/pending/published/archived)
- `HasUUID` - UUID primary keys
- `HasNumber` - Auto-generated reference numbers (e.g., "KLC-0001")
- `HasAddress` - Address fields (street, city, postcode, lat/long)
- `HasContact` - Contact info (name, email, phone)
- `HasAreas` - JSON storage of geographic areas/polygons
- `HasDate` - Date tracking with accessors
- `CanBeExported` - Export tracking (exported_at, export_error)

### Price Calculation Flow

1. Customer draws areas on map → stored in `EstimateItem.areas` (JSON with coordinates)
2. Total area calculated → `EstimateItem.total_area_m2`
3. `EstimatePlan.findItemByAreaM2()` finds appropriate pricing tier
4. `EstimateItemPriceCalculator` calculates:
   - Cost per m² (respecting minimum prices)
   - Subtotal
   - Tax (configurable rate, default 20%)
   - Total
   - Monthly price (based on chargeable treatments per year)

### Configuration

The package uses `src/Config/estimate.php` for settings:
- Route prefixes
- Email notifications
- Number format prefix
- Map center coordinates and zoom levels
- Tax configuration (`prices_include_tax`)
- Voucher codes
- Treatment schedules (treatments per year, chargeable treatments)
- Plan features mapping
- Pricing table display options
- ReCAPTCHA settings

## Testing

- Uses Orchestra Testbench for Laravel package testing
- Test base class: `tests/TestCase.php`
- DB connection set to "testing" in phpunit.xml
- Structure: `tests/Unit/` and `tests/Feature/`

## Code Style

- **Strict types**: All files use `declare(strict_types=1)`
- **Laravel Pint** with Laravel preset (config in `pint.json`)
- Custom rules:
  - `new_with_parentheses: false`
  - `nullable_type_declaration_for_default_null_value: false`
  - `declare_strict_types: true`

## Static Analysis

- **PHPStan** level 1 (see `phpstan.neon.dist`)
- Baseline file: `phpstan-baseline.neon`
- Ignores undefined property/method errors in Models, Resources, and Feature tests (expected due to Laravel magic)

## Package Integration

- Service provider: `EstimateServiceProvider`
- Auto-discovery enabled via `extra.laravel.providers` in composer.json
- Console command: `UpgradeToV4Command` (for data migration)
- Published assets to `public/` directory
- Blade components:
  - `<x-radius-map>` - Backend radius map
  - `<x-estimate-service-dropdown>` - Service selector

## Important Notes

- The package supports upgrading from v3 to v4 architecture (see deprecated config keys)
- Prices can be configured as tax-inclusive or tax-exclusive via config
- Map features require Google Maps API key stored in settings
- Models use the `AbstractModel` base class from `bongo/framework`
- Events are fired for all major model operations (create/update/delete)
- Frontend uses honeypot spam protection (Spatie Laravel Honeypot)
