<?php

declare(strict_types=1);

namespace Bongo\Estimate\Components\Backend;

use Bongo\Estimate\Maps\Circle;
use Bongo\Estimate\Maps\Path;
use Bongo\Estimate\Models\EstimateLocation;
use Exception;
use Illuminate\Contracts\Foundation\Application;
use Illuminate\Contracts\View\Factory;
use Illuminate\Contracts\View\View;
use Illuminate\View\Component;

class RadiusMap extends Component
{
    public string $radiusMapUrl;

    /** @throws Exception */
    public function __construct()
    {
        if (empty(setting('system::credentials.google_maps_api_key'))) {
            throw new Exception('Google Maps API key is required to use the radius map.');
        }

        // Create the map
        $radiusMap = (new \Bongo\Estimate\Maps\RadiusMap())
            ->setKey(setting('system::credentials.google_maps_api_key'))
            ->setCenter(config('estimate.map_center', [52.600850, 1.175440]))
            ->setZoom(config('estimate.radius_map.zoom_level', 8));

        $locations = EstimateLocation::query()
            ->select([
                'name', 'latitude', 'longitude',
                'radius', 'fill', 'stroke', 'status',
            ])
            ->where('status', EstimateLocation::ACTIVE)
            ->get();

        // Add the paths (Overlays)
        foreach ($locations as $location) {

            // Create a circle based on the area
            $coordinates = (new Circle())
                ->setLatitude($location->latitude)
                ->setLongitude($location->longitude)
                ->setZoom($radiusMap->getZoom())
                ->setRadius($location->radius)
                ->getCoordinates();

            // Convert the coordinates to a path
            $path = (new Path())
                ->setCoordinates($coordinates)
                ->setFillColor($location->fill)
                ->setStrokeColor($location->stroke);

            $radiusMap->addPath($path);
        }

        // Generate the map URL
        $this->radiusMapUrl = $radiusMap->generateUrl();
    }

    public function render(): Application|Factory|View
    {
        return view('estimate::backend.components.radius_map');
    }
}
