<?php

declare(strict_types=1);

namespace Bongo\Estimate\Http\Controllers\Backend;

use Bongo\Estimate\Events\PlanCreated;
use Bongo\Estimate\Events\PlanDeleted;
use Bongo\Estimate\Events\PlanUpdated;
use Bongo\Estimate\Http\Requests\Backend\StorePlanRequest;
use Bongo\Estimate\Http\Requests\Backend\UpdatePlanRequest;
use Bongo\Estimate\Models\EstimatePlan;
use Bongo\Estimate\Models\EstimatePlanItem;
use Bongo\Framework\Http\Controllers\AbstractController;
use Illuminate\Http\RedirectResponse;
use Illuminate\Http\Request;
use Illuminate\View\View;

class PlanController extends AbstractController
{
    protected EstimatePlan $estimatePlan;

    protected EstimatePlanItem $estimatePlanItem;

    public function __construct(EstimatePlan $estimatePlan, EstimatePlanItem $estimatePlanItem)
    {
        $this->estimatePlan = $estimatePlan;
        $this->estimatePlanItem = $estimatePlanItem;
    }

    public function index(): View
    {
        return view('estimate::backend.plan.index');
    }

    public function create(): View
    {
        $estimatePlan = $this->estimatePlan;

        return view('estimate::backend.plan.create', compact('estimatePlan'));
    }

    public function store(StorePlanRequest $request): RedirectResponse
    {
        $estimatePlan = $this->estimatePlan->create($request->all());
        $this->handlePlanItems($request, $estimatePlan);

        event(new PlanCreated($estimatePlan));

        return redirect()
            ->route('backend.estimate_plan.show', $estimatePlan->id)
            ->success(trans('estimate::backend.plan_store_success'));
    }

    public function show(EstimatePlan $estimatePlan): View
    {
        $estimatePlan->loadMissing('items');

        return view('estimate::backend.plan.show', compact('estimatePlan'));
    }

    public function edit(EstimatePlan $estimatePlan): View
    {
        $estimatePlan->loadMissing('items');

        return view('estimate::backend.plan.edit', compact('estimatePlan'));
    }

    public function update(UpdatePlanRequest $request, EstimatePlan $estimatePlan): RedirectResponse
    {
        $estimatePlan->update($request->all());
        $this->handlePlanItems($request, $estimatePlan);

        event(new PlanUpdated($estimatePlan));

        return redirect()
            ->route('backend.estimate_plan.show', $estimatePlan->id)
            ->success(trans('estimate::backend.plan_update_success'));
    }

    public function destroy(EstimatePlan $estimatePlan)
    {
        if ($estimatePlan->delete()) {
            event(new PlanDeleted($estimatePlan));
            $redirect = back();

            // If the entity was deleted from their detail view,
            // we'll redirect back to the entity list view.
            if (route('backend.estimate_plan.show', $estimatePlan) == $redirect->getTargetUrl()) {
                $redirect = redirect()->route('backend.estimate_plan.index');
            }

            return $redirect
                ->success(trans('estimate::backend.plan_delete_success'));
        }

        return back()
            ->error(trans('estimate::backend.plan_delete_failed'));
    }

    public function duplicate(EstimatePlan $estimatePlan)
    {
        $newEstimatePlan = $estimatePlan->duplicate();

        return redirect()
            ->route('backend.estimate_plan.edit', $newEstimatePlan->id)
            ->success(trans('estimate::backend.plan_update_success'));
    }

    private function handlePlanItems(Request $request, $estimatePlan)
    {
        $items = $request->get('items');

        // If the user has not passed any items through then delete all
        if (empty($items) || count($items) < 1) {
            $estimatePlan->items()->delete();

            return;
        }

        // Delete the removed items
        $estimatePlan->items()
            ->whereNotIn('id', array_keys($items))
            ->delete();

        // Create/Update the items
        foreach ($items as $item) {
            $item = $this->setItemDetails($estimatePlan, $item);

            if ($estimatePlanItem = $this->estimatePlanItem->find($item['id'])) {
                $estimatePlanItem->update($item);
            } else {
                $this->estimatePlanItem->create($item);
            }
        }
    }

    private function setItemDetails($estimatePlan, $item)
    {
        $item['estimate_plan_id'] = $estimatePlan->id;

        return $item;
    }
}
