<?php

declare(strict_types=1);

namespace Bongo\Estimate\Maps;

class PolylineEncoder
{
    protected array $points;

    protected int $numLevels;

    protected int $zoomFactor;

    protected mixed $verySmall;

    protected bool $forceEndpoints;

    protected array $zoomLevelBreaks;

    public function __construct(
        array $points = [],
        $numLevels = 18,
        $zoomFactor = 2,
        $verySmall = 0.00001,
        $forceEndpoints = true
    ) {
        $this->points = $points;
        $this->numLevels = $numLevels;
        $this->zoomFactor = $zoomFactor;
        $this->verySmall = $verySmall;
        $this->forceEndpoints = $forceEndpoints;

        for ($i = 0; $i < $this->numLevels; $i++) {
            $this->zoomLevelBreaks[$i] = $this->verySmall * pow($this->zoomFactor, $this->numLevels - $i - 1);
        }
    }

    public function getPoints(): array
    {
        return $this->points;
    }

    /**
     * The main method which is called to perform the encoding
     * Returns an associative array containing the encoded points, levels,
     * an escaped string literal containing the encoded points
     * It also returns the zoomFactor and numLevels
     */
    public function dpEncode(): array
    {
        if (count($this->points) > 2) {
            $stack[] = [0, count($this->points) - 1];
            while (count($stack) > 0) {
                $current = array_pop($stack);
                $maxDist = 0;
                $absMaxDist = 0;

                for ($i = $current[0] + 1; $i < $current[1]; $i++) {
                    $temp = self::distance(
                        $this->points[$i],
                        $this->points[$current[0]],
                        $this->points[$current[1]]
                    );
                    if ($temp > $maxDist) {
                        $maxDist = $temp;
                        $maxLoc = $i;
                        if ($maxDist > $absMaxDist) {
                            $absMaxDist = $maxDist;
                        }
                    }
                }

                if ($maxDist > $this->verySmall) {
                    $dists[$maxLoc] = $maxDist;
                    $stack[] = [$current[0], $maxLoc];
                    $stack[] = [$maxLoc, $current[1]];
                }
            }
        }

        $encodedPoints = self::createEncodings($this->points, $dists);
        $encodedLevels = self::encodeLevels($this->points, $dists, $absMaxDist);
        $encodedPointsLiteral = str_replace('\\', '\\\\', $encodedPoints);

        $polyline['points'] = $encodedPoints;
        $polyline['levels'] = $encodedLevels;
        $polyline['points_literal'] = $encodedPointsLiteral;
        $polyline['zoom_factor'] = $this->zoomFactor;
        $polyline['num_levels'] = $this->numLevels;

        return $polyline;
    }

    protected function computeLevel($dd): int
    {
        $level = 0;

        if ($dd > $this->verySmall) {
            while ($dd < $this->zoomLevelBreaks[$level]) {
                $level++;
            }
        }

        return $level;
    }

    protected function distance($p0, $p1, $p2): float
    {
        if ($p1[0] == $p2[0] && $p1[1] == $p2[1]) {
            return sqrt(pow($p2[0] - $p0[0], 2) + pow($p2[1] - $p0[1], 2));
        }

        $out = 0;
        $u = (($p0[0] - $p1[0]) * ($p2[0] - $p1[0]) + ($p0[1] - $p1[1]) * ($p2[1] - $p1[1]))
            / (pow($p2[0] - $p1[0], 2) + pow($p2[1] - $p1[1], 2));

        if ($u <= 0) {
            $out = sqrt(pow($p0[0] - $p1[0], 2) + pow($p0[1] - $p1[1], 2));
        }

        if ($u >= 1) {
            $out = sqrt(pow($p0[0] - $p2[0], 2) + pow($p0[1] - $p2[1], 2));
        }

        if ($u > 0 && $u < 1) {
            $out = sqrt(
                pow($p0[0] - $p1[0] - $u * ($p2[0] - $p1[0]), 2)
                + pow($p0[1] - $p1[1] - $u * ($p2[1] - $p1[1]), 2)
            );
        }

        return $out;
    }

    protected static function encodeSignedNumber($num): string
    {
        $sgn_num = $num << 1;

        if ($num < 0) {
            $sgn_num = ~($sgn_num);
        }

        return self::encodeNumber($sgn_num);
    }

    protected static function createEncodings($points, $dists): string
    {
        $plat = 0;
        $plng = 0;
        $encodedPoints = '';

        for ($i = 0; $i < count($points); $i++) {
            if (isset($dists[$i]) || $i == 0 || $i == count($points) - 1) {
                $point = $points[$i];
                $lat = $point[0];
                $lng = $point[1];
                $late5 = floor($lat * 1e5);
                $lnge5 = floor($lng * 1e5);
                $dlat = $late5 - $plat;
                $dlng = $lnge5 - $plng;
                $plat = $late5;
                $plng = $lnge5;
                $encodedPoints .= self::encodeSignedNumber($dlat).self::encodeSignedNumber($dlng);
            }
        }

        return $encodedPoints;
    }

    protected function encodeLevels($points, $dists, $absMaxDist): string
    {
        $encodedLevels = '';

        if ($this->forceEndpoints) {
            $encodedLevels .= self::encodeNumber($this->numLevels - 1);
        } else {
            $encodedLevels .= self::encodeNumber($this->numLevels - self::computeLevel($absMaxDist) - 1);
        }

        for ($i = 1; $i < count($points) - 1; $i++) {
            if (isset($dists[$i])) {
                $encodedLevels .= self::encodeNumber($this->numLevels - self::computeLevel($dists[$i]) - 1);
            }
        }

        if ($this->forceEndpoints) {
            $encodedLevels .= self::encodeNumber($this->numLevels - 1);
        } else {
            $encodedLevels .= self::encodeNumber($this->numLevels - self::computeLevel($absMaxDist) - 1);
        }

        return $encodedLevels;
    }

    protected static function encodeNumber($num): string
    {
        $encodeString = '';

        while ($num >= 0x20) {
            $nextValue = (0x20 | ($num & 0x1F)) + 63;
            $encodeString .= chr($nextValue);
            $num >>= 5;
        }

        $finalValue = $num + 63;
        $encodeString .= chr($finalValue);

        return $encodeString;
    }
}
