<?php

declare(strict_types=1);

namespace Bongo\Estimate\Maps;

use RecursiveArrayIterator;
use RecursiveIteratorIterator;

class StaticMapUrl extends URLGenerator
{
    public function generate(): string
    {
        $url = self::BASE_URL;
        $url .= $this->getMapParams();
        $url .= $this->getMarkerParams();
        $url .= $this->getPathParams();

        return $url;
    }

    protected function getMarkerParams(): string
    {
        $markers = $this->map->getMarkers();
        if (empty($markers)) {
            return '';
        }

        $markerParams = '';
        foreach ($markers as $marker) {
            $markerParams .= '&markers='.$this->buildMarkerParam($marker);
        }

        return $markerParams;
    }

    protected function buildMarkerParam(Marker $marker): string
    {
        $params = [];
        $params['color'] = $marker->getColor();
        $params['size'] = $marker->getSize();
        $params['anchor'] = $marker->getAnchor();
        if (! empty($marker->getLabel())) {
            $params['label'] = $marker->getLabel();
        }
        if (! empty($marker->getIcon())) {
            $params['icon'] = $marker->getIcon();
        }

        $markerParam = '';
        foreach ($params as $key => $value) {
            $markerParam .= "{$key}:{$value}|";
        }
        $markerParam = rtrim($markerParam, '|');
        unset($params);

        if (! empty($marker->getLocations())) {
            foreach ($marker->getLocations() as $point) {
                $markerParam .= "|{$point->getLat()},{$point->getLng()}";
            }
        }

        return $markerParam;
    }

    /**
     * Convert array of coordinates into compressed ANSI string
     *
     * @link https://developers.google.com/maps/documentation/utilities/polylinealgorithm
     */
    public function encodeCoordinates(array $coordinates): string
    {
        assert(is_array($coordinates));

        $tuple = 2;
        $precision = 5;

        // Zero fill previous point placeholder
        $previous = array_fill(0, $tuple, 0);

        // Flatten given coordinates
        $tmp = [];
        foreach (new RecursiveIteratorIterator(new RecursiveArrayIterator($coordinates)) as $value) {
            $tmp[] = $value;
        }
        $coordinates = $tmp;

        $index = 0;
        $encodedString = '';

        // Each cord (lat or long) is converted to float, then multiplied by 10^5 and rounded
        foreach ($coordinates as $cord) {

            $cord = (float) ($cord);
            $cord = (int) round($cord * pow(10, $precision));

            $diff = $cord - $previous[$index % $tuple];
            $previous[$index % $tuple] = $cord;
            $cord = $diff;

            $index++;
            $cord = ($cord < 0) ? ~($cord << 1) : ($cord << 1);

            $chunk = '';
            while ($cord >= 0x20) {
                $chunk .= chr((0x20 | ($cord & 0x1F)) + 63);
                $cord >>= 5;
            }
            $chunk .= chr($cord + 63);

            $encodedString .= $chunk;
        }

        return urlencode($encodedString);
    }
}
