<?php

declare(strict_types=1);

namespace Bongo\Estimate\Models;

use Bongo\Estimate\Database\Factories\EstimateServiceFactory;
use Bongo\Estimate\Traits\HasDefault;
use Bongo\Framework\Interfaces\StatusInterface;
use Bongo\Framework\Models\AbstractModel;
use Bongo\Framework\Traits\HasKey;
use Bongo\Framework\Traits\HasStatus;
use Bongo\Framework\Traits\HasUUID;
use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Relations\HasMany;
use Illuminate\Database\Eloquent\SoftDeletes;

class EstimateService extends AbstractModel implements StatusInterface
{
    use HasDefault;
    use HasFactory;
    use HasKey;
    use HasStatus;
    use HasUUID;
    use SoftDeletes;

    /** @var array */
    protected $fillable = [
        'name',
    ];

    public function plans(): HasMany
    {
        return $this->hasMany(EstimatePlan::class, 'estimate_service_id');
    }

    public function hasPlans(): bool
    {
        $this->loadMissing('plans');

        return ! is_null($this->plans) && count($this->plans);
    }

    /**
     * Check if the highest item by area_m2 is greater
     * than or equal to the total_area_m2.
     */
    public function hasPlanItemByAreaM2(int|float $totalAreaM2): bool
    {
        $this->loadMissing('plans.items');

        foreach ($this->plans as $plan) {
            if ($plan->hasItems() && $plan->hasItemByAreaM2($totalAreaM2)) {
                return true;
            }
        }

        return false;
    }

    protected static function newFactory(): EstimateServiceFactory
    {
        return EstimateServiceFactory::new();
    }
}
