// Global variables
let mapElement = document.getElementById('map');
let latitude = mapElement.dataset.latitude;
let longitude = mapElement.dataset.longitude;

// Marked area calculations
let areas = [];
let selectedArea;
let markerMoved = 0;
let totalAreaM2 = 0;

// Map objects
let map;
let marker;
let latLng;
let drawingManager;

// Custom map controls
let addAreaBtn;
let deleteSelectedAreaBtn;
let deleteAllAreasBtn;

// Initialize the map
let gMaps;
let gDrawing;
let gGeometry;
window.initMap = initMap;

// Latitude
function getLatitude() {
    return parseFloat(document.getElementById('latitude-input').value);
}

function setLatitude(latitude) {
    document.getElementById('latitude-input').value = latitude;
}

function updateLatitudeUI(latitude) {
    document
        .getElementById('latitude')
        .querySelector('.value')
        .innerHTML = latitude.toFixed(7);
}

// Longitude
function getLongitude() {
    return parseFloat(document.getElementById('longitude-input').value);
}

function setLongitude(longitude) {
    document.getElementById('longitude-input').value = longitude;
}

function updateLongitudeUI(longitude) {
    document
        .getElementById('longitude')
        .querySelector('.value')
        .innerHTML = longitude.toFixed(7);
}

// Areas
function getAreas() {
    return document.getElementById('areas-input').value
        ? JSON.parse(document.getElementById('areas-input').value)
        : [];
}

function setAreas(areas) {
    if (! areas) {
        document.getElementById('areas-input').value = null;
        return;
    }
    document.getElementById('areas-input').value = convertAreasToJSON(areas);
}

function convertAreasToJSON(areas) {

    // Make a copy areas without the overlay property and convert to json
    let areasJSON = [];

    for (let i = 0; i < areas.length; i++) {
        let clone = Object.assign({}, areas[i]);
        delete clone.overlay;
        areasJSON.push(clone);
    }

    return JSON.stringify(areasJSON);
}

function resetAreas() {
    for (let i = 0; i < areas.length; i++) {
        areas[i].overlay.setMap(null);
    }
    areas = [];
    setAreas(areas);
}

// Map marker
function getMarkerMoved() {
    return parseInt(document.getElementById('marker-moved-input').value);
}

function setMarkerMoved(markerMoved) {
    document.getElementById('marker-moved-input').value = markerMoved;
}

function createMarker(map, latLng) {
    return new gMaps.Marker({
        position: latLng,
        map: map,
        draggable: true,
        title: 'This is your property.',
    });
}

function registerMarkerEvents(map, marker) {
    gMaps.event.addListener(marker, "dragend", function () {

        let position = marker.getPosition();

        // Update the latitude to the new position
        latitude = position.lat();
        setLatitude(latitude);
        updateLatitudeUI(latitude);

        // update the longitude to the new position
        longitude = position.lng();
        setLongitude(longitude);
        updateLongitudeUI(longitude);

        // Update the marker moved flag
        markerMoved = 1;
        setMarkerMoved(markerMoved);

        // Re-center the map
        latLng = createLatLng(latitude, longitude);
        map.setCenter(latLng);
    });
}

// Selected area
function setSelectedArea(area) {
    resetSelectedArea();

    selectedArea = area;
    selectedArea.overlay.setOptions({
        editable: true,
        fillColor: "#add8e6",
    });
}

function resetSelectedArea() {
    selectedArea = null;
    for (let i = 0; i < areas.length; i++) {
        areas[i].overlay.setOptions({
            editable: false,
            fillColor: "#00FF00",
        });
    }
}

function resetSelectedAreaM2() {
    updateSelectedAreaM2UI(0);
}

function updateSelectedAreaM2UI(selectedAreaM2) {
    document
        .getElementById('selected-area')
        .querySelector('.value')
        .innerHTML = selectedAreaM2.toFixed(2);
}

// Total area
function getTotalAreaM2() {
    return parseFloat(document.getElementById('total-area-input').value);
}

function setTotalAreaM2(totalAreaM2) {
    document.getElementById('total-area-input').value = totalAreaM2;
}

function resetTotalAreaM2() {
    totalAreaM2 = 0;
    setTotalAreaM2(totalAreaM2);
    updateTotalAreaM2UI(totalAreaM2);
}

function calculateTotalAreaM2() {
    totalAreaM2 = 0;
    for (let i = 0; i < areas.length; i++) {
        areas[i].no_of_m2 = calculateAreaFromOverlay(areas[i].overlay)
        totalAreaM2 += areas[i].no_of_m2;
    }
    return totalAreaM2;
}

function updateTotalAreaM2UI(totalAreaM2) {
    document
        .getElementById('total-area')
        .querySelector('.value')
        .innerHTML = totalAreaM2.toFixed(2);
}

function updateMeasurements() {

    // Calculate and update the total area
    totalAreaM2 = calculateTotalAreaM2();
    setTotalAreaM2(totalAreaM2);
    updateTotalAreaM2UI(totalAreaM2);

    // Update the areas
    setAreas(areas);
    handleSubmitBtn();
    // debug();
}

// Map
function createMap(latLng, mapElement) {
    return new gMaps.Map(mapElement, {
        center: latLng,
        zoom: 19,
        tilt: 0,
        mapTypeId: gMaps.MapTypeId.SATELLITE,
        zoomControl: true,
        disableDoubleClickZoom: true,
        mapTypeControl: false,
        scaleControl: false,
        scrollwheel: false,
        streetViewControl: false,
        rotateControl: true,
        fullscreenControl: true,
        overviewMapControl: false,
        noClear: false,
    });
}

function restoreMap(map, areas) {
    if (areas.length > 0) {
        map.setZoom(20);

        for (let i = 0; i < areas.length; i++) {
            areas[i].overlay = new gMaps.Polygon(
                Object.assign(getOverlayOptions(), {
                    map: map,
                    editable: false,
                    paths: areas[i].coordinates
                })
            );
            registerOverlayEvents(areas[i]);
        }
    }
}

function registerMapEvents(map) {
    gMaps.event.addListener(map, 'click', function (e) {
        resetSelectedArea();
    });
}

// Custom controls
function addCustomControlsToMap(map, addAreaBtn, deleteSelectedAreaBtn, deleteAllAreasBtn) {

    // Create the custom controls toolbar
    const customControls = document.createElement("div");

    // Add the buttons to the toolbar
    customControls.appendChild(deleteAllAreasBtn);
    customControls.appendChild(deleteSelectedAreaBtn);
    customControls.appendChild(addAreaBtn);

    // Add the toolbar to the map
    map.controls[gMaps.ControlPosition.LEFT_TOP].push(customControls);
}

function abstractMapBtn() {

    const btn = document.createElement("button");
    btn.type = "button";
    btn.style.padding = "0.25rem 0.75rem";
    btn.style.margin = "0.5rem 0 0.5rem 0.5rem";
    btn.style.borderRadius = "0.125rem";
    btn.style.cursor = "pointer";
    btn.style.textAlign = "center";
    btn.style.lineHeight = "1.5";
    btn.style.fontSize = "0.825rem";
    btn.style.fontWeight = "bold";

    return btn;
}

function createAddAreaBtn() {

    const addAreaBtn = abstractMapBtn();
    addAreaBtn.textContent = "Add Area";
    addAreaBtn.title = "Click to add a new area to the map";
    addAreaBtn.style.color = "#fff";
    addAreaBtn.style.backgroundColor = "#28a745";
    addAreaBtn.style.border = "1px solid #28a745";

    return addAreaBtn;
}

function createDeleteSelectedAreaBtn() {

    const deleteSelectedAreaBtn = abstractMapBtn();
    deleteSelectedAreaBtn.textContent = "Delete Selected Area";
    deleteSelectedAreaBtn.title = "Click to delete the selected area on the map";
    deleteSelectedAreaBtn.style.color = "#fff";
    deleteSelectedAreaBtn.style.backgroundColor = "#fd7e14";
    deleteSelectedAreaBtn.style.border = "1px solid #fd7e14";

    return deleteSelectedAreaBtn;
}

function createDeleteAllAreasBtn() {

    const deleteAllAreasBtn = abstractMapBtn();
    deleteAllAreasBtn.textContent = "Delete All Areas";
    deleteAllAreasBtn.title = "Click to delete all areas on the map";
    deleteAllAreasBtn.style.color = "#fff";
    deleteAllAreasBtn.style.backgroundColor = "#dc3545";
    deleteAllAreasBtn.style.border = "1px solid #dc3545";

    return deleteAllAreasBtn;
}

function registerAddAreaEvent(map, addAreaBtn, drawingManager) {
    addAreaBtn.addEventListener("click", () => {
        if (markerMoved !== 1) {
            alert('Please drag the "Red Pin" on the map and place on your property first.');
            return;
        }
        enableDrawingManager(drawingManager);
    });
}

function registerDeleteSelectedAreaEvent(map, deleteSelectedAreaBtn) {
    deleteSelectedAreaBtn.addEventListener("click", () => {
        if (areas.length < 1) {
            alert('There are no areas to delete, please add an area using the "Add Area" button.');
            return;
        }
        if (! selectedArea) {
            alert("No area selected, please click an area on the map in order to delete it.");
            return;
        }

        // Remove the selected area
        selectedArea.overlay.setMap(null);
        areas.splice(selectedArea.index, 1);
        resetSelectedAreaM2();
        updateMeasurements()
    });
}

function registerDeleteAllAreasEvent(map, deleteAllAreasBtn) {
    deleteAllAreasBtn.addEventListener("click", () => {
        if (areas.length < 1) {
            alert('There are no areas to delete, please add an area using the "Add Area" button.');
            return;
        }

        resetSelectedArea();
        resetSelectedAreaM2();
        resetTotalAreaM2();
        resetAreas();
        handleSubmitBtn();
    });
}

// Overlay (Polygon)
function calculateAreaFromOverlay(overlay) {
    return convertPathToSquareMeters(overlay.getPath())
}

function getCordsFromOverlay(overlay) {

    let cords = [];
    let paths = overlay.getPath().getArray();

    // Get the coordinates of the overlay
    if (paths.length > 0) {
        for (let i in paths) {
            cords.push({
                lat: paths[i].lat(),
                lng: paths[i].lng(),
            });
        }
    }

    return cords;
}

function getOverlayOptions() {
    return {
        clickable: true,
        draggable: false,
        editable: true,
        fillColor: "#00FF00",
        fillOpacity: 0.5,
        geodesic: false,
        strokeColor: "#ff0000",
        strokeWeight: 2,
        zIndex: 1,
    };
}

function registerOverlayEvents(area) {

    // Polygon events
    gMaps.event.addListener(area.overlay, "click", function () {
        setSelectedArea(area);
        updateSelectedAreaM2UI(area.no_of_m2);
    });

    // Polygon path events
    gMaps.event.addListener(area.overlay.getPath(), "set_at", function () {
        areas[area.index].coordinates = getCordsFromOverlay(area.overlay);
        updateSelectedAreaM2UI(area.no_of_m2);
        updateMeasurements();
    });
    gMaps.event.addListener(area.overlay.getPath(), "insert_at", function () {
        updateSelectedAreaM2UI(area.no_of_m2);
        updateMeasurements();
    });
    gMaps.event.addListener(area.overlay.getPath(), "remove_at", function () {
        updateSelectedAreaM2UI(area.no_of_m2);
        updateMeasurements();
    });
}

// Drawing manager
function createDrawingManager() {
    return new gDrawing.DrawingManager({
        drawingMode: gDrawing.OverlayType.POLYGON,
        drawingControl: false,
        drawingControlOptions: {
            position: gMaps.ControlPosition.TOP_CENTER,
            drawingModes: ['polygon']
        },
        polygonOptions: getOverlayOptions(),
    });
}

function enableDrawingManager(drawingManager) {
    drawingManager.setDrawingMode(gDrawing.OverlayType.POLYGON);
}

function disableDrawingManager(drawingManager) {
    drawingManager.setDrawingMode(null);
}

function registerDrawingManagerEvents(drawingManager) {
    document.onkeydown = function (event) {
        event = event || window.event;
        if (event.keyCode === 27) {
            disableDrawingManager(drawingManager);
        }
    };
    gMaps.event.addListener(drawingManager, 'overlaycomplete', function (event) {

        // Turn off drawing mode
        disableDrawingManager(drawingManager);
        resetSelectedArea();

        // Create the area object
        let area = {
            index: areas.length,
            id: areas.length + 1,
            label: 'A' + (areas.length + 1),
            type: event.type,
            overlay: event.overlay,
            coordinates: getCordsFromOverlay(event.overlay),
            no_of_m2: calculateAreaFromOverlay(event.overlay),
        };

        // Add the area object to the array
        areas.push(area);

        // Calculate and update the selected area
        selectedArea = area;
        updateSelectedAreaM2UI(area.no_of_m2);
        updateMeasurements();

        // Register the overlay events
        registerOverlayEvents(area);
    });
}

// Helpers
function createLatLng(latitude, longitude) {
    return new gMaps.LatLng(latitude, longitude)
}

function convertPathToSquareMeters(path) {
    return gGeometry.spherical.computeArea(path);
}

function registerSubmitBtnEvents() {
    document
        .getElementById('get-my-quote-btn')
        .addEventListener("click", function (e) {
            e.preventDefault();

            this.disabled = true;
            this.innerText = 'Please wait...';
            document.getElementById('step-2-form').submit();

            return false;
        });
}

function enableSubmitBtn() {
    document
        .getElementById('get-my-quote-btn')
        .disabled = false;
}

function disableSubmitBtn() {
    document
        .getElementById('get-my-quote-btn')
        .disabled = true;
}

function handleSubmitBtn() {
    if (areas.length > 0) {
        enableSubmitBtn();
    } else {
        disableSubmitBtn();
    }
}

function debug() {
    console.log('-------------------');
    console.log('Areas: ', areas);
    console.log('Areas JSON: ', convertAreasToJSON(areas));
    console.log('Marker Moved: ' + markerMoved);
    console.log('Selected Area: ', selectedArea);
    console.log('Totals Area M2: ' + totalAreaM2);
}

// Main initialise function
function initMap() {

    gMaps = google.maps;
    gDrawing = gMaps.drawing;
    gGeometry = gMaps.geometry;

    // Restore state from the hidden fields
    latitude = getLatitude();
    longitude = getLongitude();
    areas = getAreas();
    markerMoved = getMarkerMoved();
    totalAreaM2 = getTotalAreaM2();

    // Update the UI
    updateLatitudeUI(latitude);
    updateLongitudeUI(longitude);
    updateTotalAreaM2UI(totalAreaM2);

    // Create the Google map
    latLng = createLatLng(latitude, longitude);
    map = createMap(latLng, mapElement);
    restoreMap(map, areas);

    // Create the address marker
    marker = createMarker(map, latLng);

    // Create the drawing manager in disabled mode
    drawingManager = createDrawingManager();
    drawingManager.setMap(map);
    disableDrawingManager(drawingManager);

    // Create the custom controls & add them to the map
    addAreaBtn = createAddAreaBtn();
    deleteSelectedAreaBtn = createDeleteSelectedAreaBtn();
    deleteAllAreasBtn = createDeleteAllAreasBtn();
    addCustomControlsToMap(map, addAreaBtn, deleteSelectedAreaBtn, deleteAllAreasBtn);

    // Register the event handlers
    registerMapEvents(map);
    registerMarkerEvents(map, marker);
    registerDrawingManagerEvents(drawingManager);
    registerAddAreaEvent(map, addAreaBtn, drawingManager);
    registerDeleteSelectedAreaEvent(map, deleteSelectedAreaBtn);
    registerDeleteAllAreasEvent(map, deleteAllAreasBtn);

    registerSubmitBtnEvents();
    handleSubmitBtn();
}
