<?php

namespace Bongo\Estimate\Http\Controllers\Frontend;

use Bongo\Estimate\Actions\FindEstimate;
use Bongo\Estimate\Exceptions\GoogleGeoCodeException;
use Bongo\Estimate\Http\Requests\SearchRequest;
use Bongo\Estimate\Services\GoogleGeoCode;
use Bongo\Framework\Http\Controllers\AbstractController;
use Illuminate\Contracts\Container\BindingResolutionException;
use Illuminate\Http\RedirectResponse;
use Spatie\Geocoder\Exceptions\CouldNotGeocode;

class SearchController extends AbstractController
{
    /**
     * @throws BindingResolutionException
     * @throws CouldNotGeocode
     * @throws GoogleGeoCodeException
     */
    public function index(SearchRequest $request): RedirectResponse
    {
        // Find the estimate or fail
        if (! $estimate = FindEstimate::byUuid(session('estimate_id'))) {
            session()->forget('estimate_id');

            return redirect()->route('frontend.estimate.step_1.show');
        }

        // Validate the postcode
        $validated = $request->validated();

        // Call the Google locations api to get the lat and long
        $result = (new GoogleGeoCode())->search($validated['line_1'], $validated['postcode']);

        // Check if the lat/long is within the coverage area
        if (! $result || ! $estimate->isWithinCoverageArea($result['latitude'], $result['longitude'])) {
            $estimate->resetAddress();
            return redirect()->to(setting('package::estimate.more_assistance_url'));
        }

        // Store the address details and continue
        $estimate->line_1 = $result['line_1'];
        $estimate->line_2 = $result['line_2'];
        $estimate->line_3 = $result['line_3'];
        $estimate->city = $result['city'];
        $estimate->county = $result['county'];
        $estimate->postcode = $result['postcode'];
        $estimate->latitude = $result['latitude'];
        $estimate->longitude = $result['longitude'];
        $estimate->save();

        return redirect()->route('frontend.estimate.step_1.show');
    }

    public function reset(): RedirectResponse
    {
        // Find the estimate or fail
        if (! $estimate = FindEstimate::byUuid(session('estimate_id'))) {
            session()->forget('estimate_id');

            return redirect()->route('frontend.estimate.step_1.show');
        }

        // Reset the address details and continue
        $estimate->resetAddress();

        return redirect()->route('frontend.estimate.step_1.show');
    }
}
