<?php

namespace Bongo\Estimate\Http\Controllers\Frontend;

use Bongo\Estimate\Actions\FindEstimate;
use Bongo\Estimate\Events\EstimateUpdated;
use Bongo\Estimate\Mailables\EstimateMailable;
use Bongo\Framework\Helpers\Tax;
use Bongo\Framework\Http\Controllers\AbstractController;
use Illuminate\Support\Facades\Mail;

class Step3Controller extends AbstractController
{
    public function show()
    {
        // Find the estimate or fail
        $estimate = FindEstimate::byUuid(session('estimate_id'));
        if (! $estimate || intval($estimate->step) < 3) {
            return redirect()->route('frontend.estimate.step_1.show');
        }

        // Destroy the session so they have to start again
        session()->forget('estimate_id');

        // Calculate the estimate
        $priceBasedOnArea = $this->getPriceBasedOnArea($estimate->total_area_m2);

        if (config('estimate.prices_include_tax')) {
            $priceBasedOnArea = Tax::excluding(
                $priceBasedOnArea,
                setting('package::estimate.tax_rate', 20)
            );
        }
        $estimate->cost_per_m2 = ($priceBasedOnArea / $estimate->total_area_m2);

        $estimate->subtotal = $priceBasedOnArea;
        $estimate->tax_rate = setting('package::estimate.tax_rate', 20);
        $estimate->tax = Tax::calculate($estimate->subtotal, $estimate->tax_rate);
        $estimate->total = ($estimate->subtotal + $estimate->tax);
        $estimate->save();

        // Send the estimate to the customer
        if (empty($estimate->sent_at)) {
            Mail::to($estimate->email)->send(new EstimateMailable($estimate));
            $estimate->sent_at = now();
            $estimate->save();
        }

        // Fire the event for anyone who is listening
        event(new EstimateUpdated($estimate));

        return view('estimate::frontend.step_3', [
            'estimate' => $estimate,
        ]);
    }

    private function getPriceBasedOnArea($totalAreaM2): int
    {
        $subtotal = config('estimate.min_price');

        foreach (config('estimate.prices') as $noOfMeters => $price) {
            $subtotal = $price;
            if ($totalAreaM2 <= intval($noOfMeters)) {
                break;
            }
        }

        return $subtotal;
    }
}
