<?php

namespace Bongo\Estimate\Maps;

use RecursiveIteratorIterator;

class URLGenerator
{
    const BASE_URL = "https://maps.googleapis.com/maps/api/staticmap?";

    protected StaticMap $staticMap;

    protected array $params = [];

    function __construct($staticMap)
    {
        $this->staticMap = $staticMap;
    }

    public function generate(): string
    {
        $url = self::BASE_URL;
        $url .= $this->getMapParams();
        $url .= $this->getMarkerParams();
        $url .= $this->getPathParams();

        return $url;
    }

    protected function getMapParams(): string
    {
        $this->params['key'] = $this->staticMap->getKey();
        $this->params['maptype'] = $this->staticMap->getMaptype();
        $this->params['zoom'] = $this->staticMap->getZoom();
        $this->params['size'] = $this->staticMap->getSize();
        $this->params['scale'] = $this->staticMap->getScale();

        $center = $this->staticMap->getCenter();
        $this->params['center'] = $center->getLat().','.$center->getLng();

        return http_build_query($this->params, '', '&');
    }

    protected function getMarkerParams(): string
    {
        $markers = $this->staticMap->getMarkers();
        if (empty($markers)) {
            return '';
        }

        $markerParams = "";
        foreach ($markers as $marker) {
            $markerParams .= "&markers=".$this->buildMarkerParam($marker);
        }

        return $markerParams;
    }

    protected function buildMarkerParam(Marker $marker): string
    {
        $params = [];
        $params['color'] = $marker->getColor();
        $params['size'] = $marker->getSize();
        $params['anchor'] = $marker->getAnchor();
        if (! empty($marker->getLabel())) {
            $params['label'] = $marker->getLabel();
        }
        if (! empty($marker->getIcon())) {
            $params['icon'] = $marker->getIcon();
        }

        $markerParam = "";
        foreach ($params as $key => $value) {
            $markerParam .= "{$key}:{$value}|";
        }
        $markerParam = rtrim($markerParam, '|');
        unset($params);

        if (! empty($marker->getLocations())) {
            foreach ($marker->getLocations() as $point) {
                $markerParam .= "|{$point->getLat()},{$point->getLng()}";
            }
        }

        return $markerParam;
    }

    protected function getPathParams(): string
    {
        $paths = $this->staticMap->getPaths();
        if (empty($paths)) {
            return '';
        }

        $pathParams = "";
        foreach ($paths as $path) {
            $pathParams .= "&".$this->buildPathParam($path);
        }

        return $pathParams;
    }

    public function buildPathParam(Path $path): string
    {
        $coordinates = $path->getCoordinates();
        if (empty($coordinates)) {
            return '';
        }

        $fillColor = $path->getFillColor();
        $strokeColor = $path->getStrokeColor();
        $strokeWeight = $path->getStrokeWeight();
        $encodedCoordinates = $this->encodeCoordinates($coordinates);

        return "path=fillcolor:{$fillColor}|color:{$strokeColor}|weight:{$strokeWeight}|enc:{$encodedCoordinates}";
    }

    /**
     * Convert array of coordinates into compressed ANSI string
     *
     * @link https://developers.google.com/maps/documentation/utilities/polylinealgorithm
     */
    public function encodeCoordinates(array $coordinates): string
    {
        assert(is_array($coordinates));

        $tuple = 2;
        $precision = 5;

        // Zero fill previous point placeholder
        $previous = array_fill(0, $tuple, 0);

        // Flatten given coordinates
        $tmp = [];
        foreach (new RecursiveIteratorIterator(new \RecursiveArrayIterator($coordinates)) as $value) {
            $tmp[] = $value;
        }
        $coordinates = $tmp;

        $index = 0;
        $encodedString = '';

        // Each cord (lat or long) is converted to float, then multiplied by 10^5 and rounded
        foreach ($coordinates as $cord) {

            $cord = (float) ($cord);
            $cord = (int) round($cord * pow(10, $precision));

            $diff = $cord - $previous[$index % $tuple];
            $previous[$index % $tuple] = $cord;
            $cord = $diff;

            $index++;
            $cord = ($cord < 0) ? ~($cord << 1) : ($cord << 1);

            $chunk = '';
            while ($cord >= 0x20) {
                $chunk .= chr((0x20 | ($cord & 0x1f)) + 63);
                $cord >>= 5;
            }
            $chunk .= chr($cord + 63);

            $encodedString .= $chunk;
        }

        return urlencode($encodedString);
    }
}
