<?php

namespace Bongo\Estimate\Models;

use Bongo\Estimate\Actions\CalculateDistance;
use Bongo\Estimate\Maps\Marker;
use Bongo\Estimate\Maps\Path;
use Bongo\Estimate\Maps\StaticMap;
use Bongo\Estimate\Traits\HasDate;
use Bongo\Estimate\Traits\HasNumber;
use Bongo\Estimate\Traits\HasStatus;
use Bongo\Framework\Models\AbstractModel;
use Bongo\Framework\Traits\HasUUID;
use Exception;
use Illuminate\Contracts\Container\BindingResolutionException;
use Illuminate\Database\Eloquent\SoftDeletes;

class Estimate extends AbstractModel
{
    public const DRAFT = 'draft';
    public const PENDING = 'pending';
    public const ACCEPTED = 'accepted';
    public const REJECTED = 'rejected';

    use HasDate;
    use HasNumber;
    use SoftDeletes;
    use HasStatus;
    use HasUUID;

    /** @var array */
    protected $fillable = [
        'number',
        'date',
        'step',
        'status',

        'line_1',
        'line_2',
        'line_3',
        'city',
        'county',
        'postcode',
        'latitude',
        'longitude',

        'first_name',
        'last_name',
        'email',
        'phone',

        'accepted_terms',
        'marketing_emails',

        'areas',
        'marker_moved',
        'total_area_m2',
        'cost_per_m2',

        'subtotal',
        'tax_rate',
        'tax',
        'total',
    ];

    /** @var array */
    protected $dates = [
        'created_at',
        'updated_at',
        'date',
        'exported_at',
    ];

    public function getNameAttribute(): string
    {
        return implode(' ', array_filter([
            $this->first_name,
            $this->last_name,
        ]));
    }

    public function getAddressAttribute(): string
    {
        return implode(', ', array_filter([
            $this->line_1,
            $this->line_2,
            $this->line_3,
            $this->city,
            $this->county,
            $this->postcode,
        ]));
    }

    public function getCostPerM2Attribute()
    {
        return ! empty($this->attributes['cost_per_m2'])
            ? $this->attributes['cost_per_m2'] / 100
            : 0;
    }

    public function setCostPerM2Attribute($val)
    {
        $this->attributes['cost_per_m2'] = $val * 100;
    }

    public function getSubtotalAttribute()
    {
        return ! empty($this->attributes['subtotal'])
            ? $this->attributes['subtotal'] / 100
            : 0;
    }

    public function setSubtotalAttribute($val)
    {
        $this->attributes['subtotal'] = $val * 100;
    }

    public function getTaxAttribute()
    {
        return ! empty($this->attributes['tax'])
            ? $this->attributes['tax'] / 100
            : 0;
    }

    public function setTaxAttribute($val)
    {
        $this->attributes['tax'] = $val * 100;
    }

    public function getTotalAttribute()
    {
        return ! empty($this->attributes['total'])
            ? $this->attributes['total'] / 100
            : 0;
    }

    public function setTotalAttribute($val)
    {
        $this->attributes['total'] = $val * 100;
    }

    public function getPerMonthAttribute()
    {
        $total = $this->attributes['total'] / 100;
        $monthsInYear = 12;
        $noOfTreatments = config('estimate.chargeable_treatments_per_year');

        return ! empty($total) ? ($total * $noOfTreatments) / $monthsInYear : 0;
    }

    public function stepIs(int $step): bool
    {
        return $this->step === $step;
    }

    public function stepIsGt(int $step): bool
    {
        return $this->step > $step;
    }

    public function stepIsGte(int $step): bool
    {
        return $this->step >= $step;
    }

    public function setStepAs(int $step): void
    {
        $this->update(['step' => $step]);
    }

    public function isWithinCoverageArea(?float $toLatitude, ?float $toLongitude): bool
    {
        if (is_null($toLatitude) || is_null($toLongitude)) {
            return false;
        }

        $radiusInMiles = floatval(setting('package::estimate.radius_in_miles'));
        $fromLatitude = floatval(setting('package::estimate.from_latitude'));
        $fromLongitude = floatval(setting('package::estimate.from_longitude'));

        $distance = CalculateDistance::inMiles($fromLatitude, $fromLongitude, $toLatitude, $toLongitude);

        return $distance <= $radiusInMiles;
    }

    public function resetAddress(): void
    {
        $this->line_1 = null;
        $this->line_2 = null;
        $this->line_3 = null;
        $this->city = null;
        $this->county = null;
        $this->postcode = null;
        $this->latitude = null;
        $this->longitude = null;
        $this->save();
    }

    /**
     * @throws BindingResolutionException
     * @throws Exception
     */
    public function getStaticMap($asImage = false): string
    {
        // Create the map
        $staticMap = (new StaticMap())
            ->setKey(setting('system::credentials.google_maps_api_key'))
            ->setCenter([$this->latitude, $this->longitude]);

        // Add the marker
        $marker = (new Marker())->addLocation([$this->latitude, $this->longitude]);
        $staticMap->addMarker($marker);

        // Add the areas (Overlays)
        if (! empty($this->areas)) {
            foreach (json_decode($this->areas, true) as $area) {
                $path = (new Path())->setCoordinates($area['coordinates']);
                $staticMap->addPath($path);
            }
        }

        // Generate the map
        return $asImage
            ? $staticMap->generateImage("{$this->uuid}.png", 'estimates')
            : $staticMap->generateUrl();
    }

    public function getAreas(): array
    {
        return ! empty($this->areas)
            ? json_decode($this->areas)
            : [];
    }

    public function hasExportedAt(): bool
    {
        return ! empty($this->exported_at);
    }

    public function hasExportError(): bool
    {
        return ! empty($this->export_error);
    }
}
