<?php

namespace Bongo\Estimate\Services;

use Bongo\Estimate\Exceptions\GoogleGeoCodeException;
use Bongo\Estimate\Interfaces\GeoLookup;
use GuzzleHttp\Client;
use Illuminate\Contracts\Container\BindingResolutionException;
use Illuminate\Support\Str;
use Spatie\Geocoder\Exceptions\CouldNotGeocode;

class GoogleGeoCode implements GeoLookup
{
    /**
     * @throws GoogleGeoCodeException
     * @throws BindingResolutionException
     * @throws CouldNotGeocode
     */
    public function search(string $line1, string $postcode): array
    {
        if (empty(setting('system::credentials.google_geocoding_api_key'))) {
            throw new GoogleGeoCodeException('Google Geocoding API key is not set');
        }

        // Instantiate the geocoder
        $client = new Client();
        $geocoder = new GoogleGeoCoder($client);
        $geocoder->setApiKey(setting('system::credentials.google_geocoding_api_key'));
        $geocoder->setCountry('UK');

        // Geocode the address
        $address = $this->formatPostcode($postcode).', '.$this->formatSearchLine1($line1);
        $coordinates = $geocoder->getCoordinatesForAddress($address);

        // Create the array
        $result = [
            'line_1' => null,
            'line_2' => null,
            'line_3' => null,
            'city' => null,
            'county' => null,
            'postcode' => null,
            'latitude' => null,
            'longitude' => null,
        ];

        // Get the components
        if (! empty($coordinates['address_components'])) {
            foreach ($coordinates['address_components'] as $component) {
                if (in_array('street_number', $component->types)) {
                    $result['line_1'] = $component->long_name;
                }
                if (in_array('route', $component->types)) {
                    $result['line_2'] = $component->long_name;
                }
                if (in_array('locality', $component->types)) {
                    $result['line_3'] = $component->long_name;
                }
                if (in_array('postal_town', $component->types)) {
                    $result['city'] = $component->long_name;
                }
                if (in_array('administrative_area_level_2', $component->types)) {
                    $result['county'] = $component->long_name;
                }
            }
            $result['line_1'] = $this->formatLine1($line1);
            $result['postcode'] = $this->formatPostcode($postcode);
            $result['latitude'] = $this->setLatitude($coordinates);
            $result['longitude'] = $this->setLongitude($coordinates);
        }

        return $result;
    }

    protected function formatSearchLine1(string $line1): string
    {
        // If the line contains a comma, reverse the segments
        if (Str::contains($line1, ',')) {
            $segments = explode(',', $line1);
            if (count($segments) > 0) {
                foreach ($segments as $key => $segment) {
                    $segments[$key] = ucwords(strtolower(trim($segment)));
                }
                $line1 = implode(', ', array_reverse($segments));
            }
        }

        return trim($line1);
    }

    protected function formatLine1(string $line1): string
    {
        // If the line contains a comma, reverse the segments
        if (Str::contains($line1, ',')) {
            $segments = explode(',', $line1);
            if (count($segments) > 0) {
                $line1 = ucfirst($segments[0]);
            }
        }

        return trim($line1);
    }

    protected function formatPostcode(string $postcode): string
    {
        $beforeLastThreeDigits = substr($postcode, 0, strlen($postcode) - 3);
        $lastThreeDigits = substr($postcode, -3);

        // Adds a space before the last three digits
        $postcode = strtoupper($beforeLastThreeDigits.' '.$lastThreeDigits);

        return preg_replace('/\s\s+/', ' ', $postcode);
    }

    public function setLatitude(array $coordinates): ?float
    {
        return (float) $coordinates['lat'] ?? null;
    }

    public function setLongitude(array $coordinates): ?float
    {
        return (float) $coordinates['lng'] ?? null;
    }
}
