<?php

namespace Bongo\Form\Http\Controllers\Backend;

use Bongo\Form\Models\FormItem;
use Bongo\Framework\Http\Controllers\AbstractController;
use Bongo\Form\Events\FormCreated;
use Bongo\Form\Events\FormDeleted;
use Bongo\Form\Events\FormUpdated;
use Bongo\Form\Http\Requests\StoreFormRequest;
use Bongo\Form\Http\Requests\UpdateFormRequest;
use Bongo\Form\Models\Form;

use Exception;
use Illuminate\Http\RedirectResponse;
use Illuminate\View\View;

/**
 * Class FormController.
 *
 * @category  Bespoke_Software
 * @author    Bespoke.ws Ltd <support@bespokeuk.com>
 * @copyright 2015-2020 Bespoke.ws Ltd, All Rights Reserved
 * @license   Proprietary and confidential
 * Unauthorized copying of this file, via any medium is strictly prohibited
 * @link      https://bespokeuk.com
 * @package   Bongo\Form\Http\Controllers\Backend
 */
class FormController extends AbstractController
{
    /**
     * @var Form
     */
    protected $form;

    /**
     * FormController constructor.
     *
     * @param Form $form
     */
    public function __construct(Form $form)
    {
        $this->form = $form;
    }

    /**
     * @return View
     */
    public function index(): View
    {
        return view('form::backend.index');
    }

    /**
     * @return View
     */
    public function create(): View
    {
        $form = $this->form;

        return view('form::backend.create', compact('form'));
    }

    /**
     * @param StoreFormRequest $request
     *
     * @return RedirectResponse
     * @throws Exception
     */
    public function store(StoreFormRequest $request): RedirectResponse
    {
        $form = $this->form->create($request->all());
        $this->handleFormItems($form);
        event(new FormCreated($form));

        return redirect()
            ->route('backend.form.show', $form->id)
            ->success(trans('form::backend.store_success'));
    }

    /**
     * @param Form $form
     *
     * @return View
     */
    public function show(Form $form): View
    {
        $form->loadMissing('items');

        return view('form::backend.show', compact('form'));
    }

    /**
     * @param Form $form
     *
     * @return View
     */
    public function edit(Form $form): View
    {
        $form->loadMissing('items');

        return view('form::backend.edit', compact('form'));
    }

    /**
     * @param UpdateFormRequest $request
     * @param Form              $form
     *
     * @return RedirectResponse
     * @throws Exception
     */
    public function update(UpdateFormRequest $request, Form $form): RedirectResponse
    {
        $form->update($request->all());
        $this->handleFormItems($form);
        event(new FormUpdated($form));

        return redirect()
            ->route('backend.form.show', $form->id)
            ->success(trans('form::backend.update_success'));
    }

    /**
     * @param Form $form
     *
     * @return mixed
     * @throws Exception
     */
    public function destroy(Form $form)
    {
        if ($form->delete()) {
            event(new FormDeleted($form));
            $redirect = back();

            // If the form was deleted from their detail view,
            // we'll redirect back to the form list view.
            if (route('backend.form.show', $form) == $redirect->getTargetUrl()) {
                $redirect = redirect()->route('backend.form.index');
            }

            return $redirect->success(trans('form::backend.delete_success'));
        }

        return back()->error(trans('form::backend.delete_failed'));
    }

    /**
     * @param $form
     */
    private function handleFormItems($form)
    {
        $items = request('items');

        // If the user has not passed any items through then delete all
        if (!isset($items) || empty($items)) {
            $form->items()->delete();
            return;
        }

        // Delete the removed items
        $form->items()->whereNotIn('id', array_keys($items))->delete();

        // Create/Update the items
        foreach ($items as  $key => $item) {

            $item['form_id'] = $form->id;
            if ($item['type'] !== FormItem::SELECT) {
                $item['options'] = null;
            }
            if (empty($item['sort_order'])) {
                $item['sort_order'] = $key;
            }

            if ($formItem = FormItem::find($item['id'])) {
                $formItem->update($item);
            } else {
                FormItem::create($item);
            }
        }
    }
}
