<?php

namespace Bongo\Form\Services;

use Bongo\Form\Models\Form;

use Bongo\Form\Models\FormItem;
use Exception;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Storage;
use Illuminate\Support\Facades\Validator;
use Illuminate\Support\Str;
use Illuminate\Validation\Rule;
use Illuminate\Validation\ValidationException;

/**
 * Class FormHandler
 *
 * @category  Bespoke_Software
 * @author    Bespoke.ws Ltd <support@bespokeuk.com>
 * @copyright 2015-2020 Bespoke.ws Ltd, All Rights Reserved
 * @license   Proprietary and confidential
 * Unauthorized copying of this file, via any medium is strictly prohibited
 * @link      https://bespokeuk.com
 * @package   Bongo\Form\Models
 */
class FormHandler
{
    /**
     * @var Form
     */
    protected Form $form;

    /**
     * @var Request
     */
    protected Request $request;

    /**
     * @var array
     */
    protected array $fields = [];

    /**
     * @param  Form  $form
     * @param  Request  $request
     *
     * @throws Exception
     */
    public function __construct(Form $form, Request $request)
    {
        $this->form = $form;
        $this->request = $request;

        $this->setFieldsArray();
        $this->addValuesToFieldsArray();
    }

    /**
     * @throws Exception
     */
    protected function setFieldsArray()
    {
        // Make sure this form has items
        if (! $this->form->hasItems()) {
            throw new Exception("{$this->form->id} does not have any items");
        }

        // Create the initial data structure
        $formItems = $this->form->items()->notText()->get();
        foreach ($formItems as $item) {
            $this->fields[$item->name]['label'] = $item->label;
            $this->fields[$item->name]['name'] = $item->name;
            $this->fields[$item->name]['type'] = $item->type;
            $this->fields[$item->name]['rules'] = $item->rules;
            $this->fields[$item->name]['value'] = null;
        }
    }

    /**
     * @throws Exception
     */
    protected function addValuesToFieldsArray()
    {
        // Make sure we have some data
        if (! $this->request->has('fields')) {
            throw new Exception("{$this->form->id} does not have submitted fields");
        }

        // Populate the array with the standard data
        foreach ($this->request->get('fields') as $key => $value) {
            if (isset($this->fields[$key])) {
                $this->fields[$key]['value'] = $value;
            }
        }

        // Populate the array with the file data
        if ($this->request->file('fields') && ! empty($this->request->file('fields'))) {
            foreach ($this->request->file('fields') as $key => $value) {
                if (isset($this->fields[$key])) {
                    $this->fields[$key]['value'] = $value;
                }
            }
        }
    }

    /**
     * @throws ValidationException
     */
    public function validateFields()
    {
        // Create an array of key/values
        $data = [];
        foreach ($this->fields as $key => $field) {
            $data[$field['name']] = $field['value'];
        }

        // Create an array of key/rules
        $rules = [];
        foreach ($this->fields as $key => $field) {
            $fieldName = is_array($field['value']) ? $field['name'].'.*' : $field['name'];
            $rules[$fieldName] = implode('|', $field['rules']);
        }

        // Add the reCaptcha rule if enabled
        if (setting()->reCaptchaEnabled() && config('form.recaptcha.enabled')) {
            $recaptchaKey = make_key($this->form->name);
            $recaptchaMinScore = config('form.recaptcha.min_score', 0.5);

            $data['g-recaptcha-response'] = $this->request->get('g-recaptcha-response');
            $rules['g-recaptcha-response'] = [
                "recaptchav3:{$recaptchaKey},{$recaptchaMinScore},",
                'required',
            ];
        }

        // Pass the data/rules to the validator and validate
        Validator::make($data, $rules)->validate();

        unset($data);
        unset($rules);
    }

    /**
     * Store the files for the email
     */
    public function storeFiles()
    {
        // Loop through the fields
        foreach ($this->fields as $key => $field) {

            // if the field holds images or files then save them
            if (($field['type'] === FormItem::IMAGE || $field['type'] === FormItem::FILE)
                && ! empty($field['value'])
            ) {
                // Get the files
                foreach ($field['value'] as $fKey => $file) {
                    // Set the path and name
                    $tmpPath = config('document.tmp_path');
                    $fileName = $this->generateFileName($file);
                    $filePath = $tmpPath.$fileName;

                    // If the file saved, then replace the value with the file location
                    if ($file->storePubliclyAs(rtrim($tmpPath, '/'), $fileName)) {
                        $this->fields[$key]['value'][$fKey] = Storage::url($filePath);;
                    }
                }
            }
        }
    }

    /**
     * @return array
     */
    public function getFields(): array
    {
        // Remove the empty fields
        foreach ($this->fields as $key => $field) {
            if (empty($field['value'])) {
                unset($this->fields[$key]);
            }
        }

        return $this->fields;
    }

    /**
     * @param $file
     *
     * @return string
     */
    protected function generateFileName($file): string
    {
        $fileExt = $file->getClientOriginalExtension();
        $fileName = rtrim($file->getClientOriginalName(), $fileExt);
        $fileName = strtolower(Str::slug($fileName));
        $fileName = $fileName.'-'.time().'.'.$fileExt;

        return $fileName;
    }
}
