# GitHub Copilot Instructions - Bongo Form Package

## Project Context

This is the `bongo/form` Laravel package for creating and managing dynamic forms with multiple field types, validation, file uploads, and email notifications.

**Namespace**: `Bongo\Form`
**Dependencies**: `bongo/framework` ^3.0, `bongo/captcha`
**PHP**: 8.2+ | **Laravel**: 10+

## Key Classes & Relationships

### Models

```php
// Form.php - Main form model
namespace Bongo\Form\Models;

class Form extends AbstractModel
{
    use SoftDeletes, HasKey, HasUUID;

    // Relationships
    public function items(): HasMany;

    // Key methods
    public function hasItems(): bool;
    public function getRecipientsAsArray(): ?array;
}

// FormItem.php - Form field/item model
class FormItem extends AbstractModel
{
    use HasUUID, HasType;

    // Constants for types
    const CHECKBOX = 'checkbox';
    const DATE = 'date';
    const EMAIL = 'email';
    const FILE = 'file';
    const HTML = 'html';
    const IMAGE = 'image';
    const INPUT = 'input';
    const SELECT = 'select';
    const TEXT = 'text';
    const TEXTAREA = 'textarea';

    // Key methods
    public function getRulesAttribute(): array;
    public function isRequired(): bool;
    public function hasOptions(): bool;
    public function getOptionsAsArray(): ?array;
}
```

### Service Classes

```php
// FormHandler.php - Processes form submissions
namespace Bongo\Form\Services;

class FormHandler
{
    public function __construct(Form $form, Request $request);
    public function validateFields(): void;
    public function storeFiles(): void;
    public function purgeOldFiles(): void;
    public function getFields(): array;
}
```

### Controllers

```php
// Backend/FormController.php - Admin CRUD
namespace Bongo\Form\Http\Controllers\Backend;

class FormController extends AbstractController
{
    public function index(): View;
    public function create(): View;
    public function store(StoreFormRequest $request): RedirectResponse;
    public function show(Form $form): View;
    public function edit(Form $form): View;
    public function update(UpdateFormRequest $request, Form $form): RedirectResponse;
    public function destroy(Form $form);
    private function handleFormItems(Form $form);
}

// Frontend/FormController.php - Public submission
namespace Bongo\Form\Http\Controllers\Frontend;

class FormController extends AbstractController
{
    public function store(Request $request): RedirectResponse;
    public function preview(string $uuid): View;
}
```

### Events

```php
namespace Bongo\Form\Events;

class FormCreated { public Form $form; }
class FormUpdated { public Form $form; }
class FormDeleted { public Form $form; }
```

### Mailables

```php
namespace Bongo\Form\Mailables;

// Sent to submitter
class FormMailable extends Mailable
{
    public function __construct(Form $form, array $fields);
}

// Sent to admin
class AdminFormMailable extends Mailable
{
    public function __construct(Form $form, array $fields, array $emailFields);
}
```

## Code Style Templates

### Adding a New Form Field Type

```php
// 1. Add constant to FormItem model
public const NEWTYPE = 'newtype';

// 2. Add to HasType trait
public function scopeNewType($query)
{
    return $query->where('type', self::NEWTYPE);
}

public function isNewType(): bool
{
    return $this->type === self::NEWTYPE;
}

// 3. Update validation if needed
public function getRulesAttribute(): array
{
    $rules = $this->isRequired() ? ['required'] : ['nullable'];

    if ($this->isNewType()) {
        $rules = array_merge($rules, ['custom_rule']);
    }

    return $rules;
}

// 4. Create blade partial: src/Views/frontend/partials/newtype.blade.php
```

### Working with Form Submissions

```php
use Bongo\Form\Services\FormHandler;
use Bongo\Form\Models\Form;

// Get form by UUID
$form = Form::where('uuid', $request->get('ref'))->first();

// Process submission
$formHandler = new FormHandler($form, $request);
$formHandler->validateFields();
$formHandler->storeFiles();

// Get processed fields
$fields = $formHandler->getFields();

// Extract email addresses from email fields
$emailFields = collect($fields)
    ->filter(fn($field) => $field['type'] == FormItem::EMAIL)
    ->pluck('value')
    ->toArray();
```

### Creating/Updating Form Items

```php
// In backend controller
private function handleFormItems(Form $form)
{
    $items = request('items');

    if (empty($items)) {
        $form->items()->delete();
        return;
    }

    // Delete removed items
    $form->items()->whereNotIn('id', array_keys($items))->delete();

    // Create/update items
    foreach ($items as $key => $item) {
        $item['form_id'] = $form->id;

        // Clean HTML fields
        if ($item['type'] === FormItem::HTML && !empty($item['label'])) {
            $item['label'] = strip_tags($item['label'], config('form.allowed_html_tags'));
        }

        // Remove options for non-select fields
        if ($item['type'] !== FormItem::SELECT) {
            $item['options'] = null;
        }

        // Update or create
        if ($formItem = FormItem::find($item['id'])) {
            $formItem->update($item);
        } else {
            FormItem::create($item);
        }
    }
}
```

### Sending Form Emails

```php
use Bongo\Form\Mailables\FormMailable;
use Bongo\Form\Mailables\AdminFormMailable;
use Illuminate\Support\Facades\Mail;

// Send to submitter (if email fields exist)
if (!empty($emailFields)) {
    Mail::to($emailFields)
        ->send(new FormMailable($form, $fields));
}

// Send to admin
Mail::to($form->getRecipientsAsArray())
    ->send(new AdminFormMailable($form, $fields, $emailFields));
```

### Query Scopes for FormItem Types

```php
// Get all email fields for a form
$emailFields = FormItem::where('form_id', $form->id)
    ->email()
    ->get();

// Get all non-text fields
$inputFields = FormItem::where('form_id', $form->id)
    ->notText()
    ->orderBy('sort_order')
    ->get();

// Check field type
if ($item->isImage()) {
    // Handle image upload
}

if ($item->isRequired()) {
    // Mark as required in form
}
```

## Common Patterns

### File Upload Handling

```php
// In FormHandler service
public function storeFiles(): void
{
    foreach ($this->fields as $key => $field) {
        if (($field['type'] === FormItem::IMAGE || $field['type'] === FormItem::FILE)
            && !empty($field['value'])
        ) {
            foreach ($field['value'] as $fKey => $file) {
                $tmpPath = config('document.tmp_path');
                $fileName = $this->generateFileName($file);
                $filePath = $tmpPath . $fileName;

                if ($file->storePubliclyAs(rtrim($tmpPath, '/'), $fileName)) {
                    $this->fields[$key]['value'][$fKey] = Storage::url($filePath);
                }
            }
        }
    }
}
```

### Dynamic Validation Rules

```php
// Build validation rules from FormItem
$rules = [];
foreach ($fields as $field) {
    $fieldName = is_array($field['value'])
        ? $field['name'] . '.*'
        : $field['name'];
    $rules[$fieldName] = implode('|', $field['rules']);
}

// Add reCAPTCHA if enabled
if (setting()->reCaptchaEnabled()) {
    $data['captcha-response'] = $request->get('captcha-response');
    $rules['captcha-response'] = new Captcha(
        action: make_key($form->name),
        minScore: config('form.recaptcha.min_score', 0.5)
    );
}

Validator::make($data, $rules)->validate();
```

### Purging Old Files

```php
public function purgeOldFiles(): void
{
    $allFiles = Storage::files(config('document.tmp_path'));
    $purgeFilesAfter = config('form.purge_files_after', 60);
    $nowMinusPurgeDays = strtotime("-{$purgeFilesAfter} days");

    $oldFiles = array_filter($allFiles, function($file) use ($nowMinusPurgeDays) {
        return Storage::lastModified($file) < $nowMinusPurgeDays;
    });

    foreach ($oldFiles as $file) {
        Storage::delete($file);
    }
}
```

## Configuration Access

```php
// Route prefix
config('form.prefix') // 'forms'

// File types
config('form.allowed_image_types') // 'png,jpg,jpeg,gif'
config('form.allowed_document_types') // 'png,jpg,jpeg,gif,pdf'

// File purging
config('form.purge_files') // true
config('form.purge_files_after') // 60 days

// HTML sanitization
config('form.allowed_html_tags') // '<p><h1><h2>...'

// reCAPTCHA
config('form.recaptcha.enabled') // true
config('form.recaptcha.min_score') // 0.5
```

## Route Naming Convention

```php
// Backend routes (auth + employee middleware)
route('backend.form.index')
route('backend.form.create')
route('backend.form.store')
route('backend.form.show', $form)
route('backend.form.edit', $form)
route('backend.form.update', $form)
route('backend.form.destroy', $form)
route('backend.form.datatable')

// Frontend routes
route('form.preview', $uuid) // Developer only
route('form.store') // Honeypot protected
```

## View Naming Convention

```php
// Backend views
view('form::backend.index')
view('form::backend.create', compact('form'))
view('form::backend.edit', compact('form'))
view('form::backend.show', compact('form'))

// Frontend views
view('form::frontend.preview', compact('form'))

// Frontend partials (field types)
view('form::frontend.partials.input')
view('form::frontend.partials.email')
view('form::frontend.partials.select')
// etc.

// Mail views
view('form::mail.response', ['form' => $form, 'fields' => $fields])
view('form::mail.admin_response', ['form' => $form, 'fields' => $fields])
```

## Translation Keys

```php
// Backend
trans('form::backend.store_success')
trans('form::backend.update_success')
trans('form::backend.delete_success')
trans('form::backend.delete_failed')

// Frontend
trans('form::frontend.store_success')
```

## Database Schema Reference

### forms Table
- `id`, `uuid`, `name`, `key`
- `success_url`, `recipients`, `subject`
- `created_by`, `updated_by`, `deleted_by`
- `created_at`, `updated_at`, `deleted_at`

### form_items Table
- `id`, `uuid`, `form_id`
- `label`, `display_label`, `name`, `class`, `width`, `placeholder`
- `options` (text), `required` (enum), `type` (enum)
- `sort_order`
- `created_by`, `updated_by`
- `created_at`, `updated_at`

## Security Best Practices

1. **Always sanitize HTML fields**: Use `strip_tags()` with allowed tags
2. **Validate file uploads**: Check mime types against config whitelist
3. **Use reCAPTCHA v3**: Score-based validation for spam prevention
4. **Honeypot protection**: Frontend store route uses `ProtectAgainstSpam` middleware
5. **Purge temporary files**: Auto-delete after configured days
6. **Type validation**: Use FormItem constants, never raw strings
