<?php

namespace Bongo\Form\Http\Controllers\Frontend;

use Bongo\Form\Mailables\AdminFormMailable;
use Bongo\Form\Mailables\FormMailable;
use Bongo\Form\Models\Form;
use Bongo\Form\Models\FormItem;
use Bongo\Form\Services\FormHandler;
use Bongo\Framework\Http\Controllers\AbstractController;
use Exception;
use Illuminate\Http\RedirectResponse;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Mail;
use Illuminate\View\View;

class FormController extends AbstractController
{
    /**
     * @throws Exception
     */
    public function store(Request $request): RedirectResponse
    {
        // Get the form
        $form = Form::where('uuid', $request->get('ref'))->first();

        if (! $form) {
            abort('404', 'Form could not be found');
        }

        // Handle the form data
        $formHandler = new FormHandler($form, $request);
        $formHandler->validateFields();
        $formHandler->storeFiles();

        // If the form had any email fields then grab them & merge with the default
        $emailFields = collect($formHandler->getFields())
            ->filter(function ($field) {
                return $field['type'] == FormItem::EMAIL;
            })
            ->pluck('value')
            ->values()
            ->toArray();

        // If the form contained email fields then send email to the customer
        if (! empty($emailFields)) {
            Mail::to($emailFields)
                ->send(new FormMailable($form, $formHandler->getFields()));
        }

        // Send the email to the admin
        Mail::to($form->getRecipientsAsArray()) // to the recipients
            ->send(new AdminFormMailable($form, $formHandler->getFields(), $emailFields));

        // Clean up any old emails
        if (config('form.purge_files')) {
            $formHandler->purgeOldFiles();
        }

        // Redirect the user to the thank-you screen
        return redirect()
            ->to($form->success_url)
            ->success(trans('form::frontend.store_success'));
    }

    public function preview(string $uuid): View
    {
        $form = Form::where('uuid', '=', $uuid)->first();

        if (! $form || (! auth()->user() || ! auth()->user()->isDeveloper())) {
            abort('404', 'Form could not be found');
        }

        $form->loadMissing('items');

        return view('form::frontend.preview', compact('form'));
    }
}
