<?php

namespace Bongo\Form\Http\Controllers\Backend;

use Bongo\Form\Events\FormCreated;
use Bongo\Form\Events\FormDeleted;
use Bongo\Form\Events\FormUpdated;
use Bongo\Form\Http\Requests\StoreFormRequest;
use Bongo\Form\Http\Requests\UpdateFormRequest;
use Bongo\Form\Models\Form;
use Bongo\Form\Models\FormItem;
use Bongo\Framework\Http\Controllers\AbstractController;
use Illuminate\Http\RedirectResponse;
use Illuminate\View\View;

class FormController extends AbstractController
{
    protected Form $form;

    public function __construct(Form $form)
    {
        $this->form = $form;
    }

    public function index(): View
    {
        return view('form::backend.index');
    }

    public function create(): View
    {
        $form = $this->form;

        return view('form::backend.create', compact('form'));
    }

    public function store(StoreFormRequest $request): RedirectResponse
    {
        $form = $this->form->create($request->all());
        $this->handleFormItems($form);
        event(new FormCreated($form));

        return redirect()
            ->route('backend.form.show', $form->id)
            ->success(trans('form::backend.store_success'));
    }

    public function show(Form $form): View
    {
        $form->loadMissing('items');

        return view('form::backend.show', compact('form'));
    }

    public function edit(Form $form): View
    {
        $form->loadMissing('items');

        return view('form::backend.edit', compact('form'));
    }

    public function update(UpdateFormRequest $request, Form $form): RedirectResponse
    {
        $form->update($request->all());
        $this->handleFormItems($form);
        event(new FormUpdated($form));

        return redirect()
            ->route('backend.form.show', $form->id)
            ->success(trans('form::backend.update_success'));
    }

    public function destroy(Form $form)
    {
        if ($form->delete()) {
            event(new FormDeleted($form));
            $redirect = back();

            // If the form was deleted from their detail view,
            // we'll redirect back to the form list view.
            if (route('backend.form.show', $form) == $redirect->getTargetUrl()) {
                $redirect = redirect()->route('backend.form.index');
            }

            return $redirect->success(trans('form::backend.delete_success'));
        }

        return back()->error(trans('form::backend.delete_failed'));
    }

    private function handleFormItems(Form $form)
    {
        $items = request('items');

        // If the user has not passed any items through then delete all
        if (empty($items)) {
            $form->items()->delete();

            return;
        }

        // Delete the removed items
        $form->items()->whereNotIn('id', array_keys($items))->delete();

        // Create/Update the items
        foreach ($items as $key => $item) {

            // Set the form ID
            $item['form_id'] = $form->id;

            // If the item is HTML then clean the label
            if ($item['type'] === FormItem::HTML && ! empty($item['label'])) {
                $item['label'] = strip_tags($item['label'], config('form.allowed_html_tags'));
            }

            // If the item is not a select, remove the options
            if ($item['type'] !== FormItem::SELECT) {
                $item['options'] = null;
            }

            // If the item has no sort order, set it to the key
            if (empty($item['sort_order'])) {
                $item['sort_order'] = $key;
            }

            // Update or create the item
            if ($formItem = FormItem::find($item['id'])) {
                $formItem->update($item);
            } else {
                FormItem::create($item);
            }
        }
    }
}
