# Architecture Documentation - Bongo Form Package

## Overview

The `bongo/form` package provides a complete form builder and management system for Laravel applications. It allows administrators to create custom forms with dynamic field types through a backend interface, while providing frontend form rendering, validation, file uploads, and email notifications.

**Package**: `bongo/form`
**Namespace**: `Bongo\Form`
**Version**: ^3.0
**PHP**: 8.2+
**Laravel**: 10+

## High-Level Architecture

```
┌─────────────────────────────────────────────────────────────────┐
│                        Bongo Form Package                        │
├─────────────────────────────────────────────────────────────────┤
│                                                                   │
│  ┌────────────────────┐         ┌──────────────────────┐       │
│  │   Backend Admin    │         │  Frontend Rendering  │       │
│  │   (CRUD Forms)     │         │  (Submit Forms)      │       │
│  └─────────┬──────────┘         └──────────┬───────────┘       │
│            │                               │                    │
│            ├───────────┬───────────────────┤                    │
│            │           │                   │                    │
│  ┌─────────▼─────┐  ┌──▼────────┐  ┌──────▼───────┐           │
│  │  Form Model   │  │ FormItem  │  │ FormHandler  │           │
│  │  (HasMany)    ├──┤  Model    │  │   Service    │           │
│  │               │  │ (HasType) │  │              │           │
│  └───────────────┘  └───────────┘  └──────┬───────┘           │
│                                            │                    │
│                     ┌──────────────────────┴────────┐           │
│                     │                               │           │
│           ┌─────────▼──────────┐        ┌──────────▼────────┐  │
│           │  Email Mailables   │        │  File Storage     │  │
│           │  (Admin/Customer)  │        │  (Temp uploads)   │  │
│           └────────────────────┘        └───────────────────┘  │
│                                                                   │
└─────────────────────────────────────────────────────────────────┘
         │                                              │
         │ Extends                                      │ Uses
         ▼                                              ▼
┌──────────────────┐                          ┌─────────────────┐
│ bongo/framework  │                          │ bongo/captcha   │
│ - AbstractModel  │                          │ - Captcha Rule  │
│ - Traits (UUID,  │                          │ - reCAPTCHA v3  │
│   Key, etc.)     │                          └─────────────────┘
└──────────────────┘
```

## Directory Structure

```
src/
├── Config/
│   └── form.php                              # Package configuration
│
├── Events/
│   ├── FormCreated.php                       # Event: Form created
│   ├── FormUpdated.php                       # Event: Form updated
│   └── FormDeleted.php                       # Event: Form deleted
│
├── Http/
│   ├── Controllers/
│   │   ├── Backend/
│   │   │   ├── FormController.php            # CRUD operations
│   │   │   └── FormDatatableController.php   # DataTable integration
│   │   └── Frontend/
│   │       └── FormController.php            # Form submission & preview
│   └── Requests/
│       ├── StoreFormRequest.php              # Validation for create
│       └── UpdateFormRequest.php             # Validation for update
│
├── Mailables/
│   ├── FormMailable.php                      # Confirmation to submitter
│   └── AdminFormMailable.php                 # Notification to admin
│
├── Migrations/
│   ├── 2021_01_01_000001_create_forms_table.php
│   ├── 2021_01_01_000002_create_form_items_table.php
│   ├── 2021_01_01_000003_add_additional_types_to_form_items_table.php
│   ├── 2021_01_01_000004_add_placeholder_column_to_form_items_table.php
│   ├── 2021_01_01_000005_add_email_type_to_form_items_table.php
│   ├── 2021_01_01_000006_add_display_label_column_to_form_items_table.php
│   ├── 2021_01_01_000007_add_width_column_to_form_items_table.php
│   └── 2024_01_01_000008_add_html_type_to_form_items_table.php
│
├── Models/
│   ├── Form.php                              # Main form model
│   └── FormItem.php                          # Form field/item model
│
├── Routes/
│   ├── backend.php                           # Admin routes (backend.form.*)
│   └── frontend.php                          # Public routes (form.*)
│
├── Services/
│   └── FormHandler.php                       # Form submission processor
│
├── Traits/
│   └── HasType.php                           # Type checking for FormItem
│
├── Translations/
│   └── en/
│       ├── backend.php                       # Admin translations
│       └── frontend.php                      # Frontend translations
│
├── Views/
│   ├── backend/
│   │   ├── index.blade.php                   # List forms
│   │   ├── create.blade.php                  # Create form
│   │   ├── edit.blade.php                    # Edit form
│   │   ├── show.blade.php                    # View form
│   │   └── partials/
│   │       └── form/
│   │           └── details.blade.php         # Form details partial
│   ├── frontend/
│   │   ├── preview.blade.php                 # Preview form
│   │   └── partials/
│   │       ├── form.blade.php                # Main form wrapper
│   │       ├── checkbox.blade.php            # Checkbox field
│   │       ├── date.blade.php                # Date field
│   │       ├── email.blade.php               # Email field
│   │       ├── file.blade.php                # File upload field
│   │       ├── html.blade.php                # HTML content field
│   │       ├── image.blade.php               # Image upload field
│   │       ├── input.blade.php               # Text input field
│   │       ├── select.blade.php              # Select dropdown field
│   │       ├── text.blade.php                # Display text field
│   │       └── textarea.blade.php            # Textarea field
│   └── mail/
│       ├── response.blade.php                # Customer email (HTML)
│       ├── response_plain.blade.php          # Customer email (plain)
│       ├── admin_response.blade.php          # Admin email (HTML)
│       └── admin_response_plain.blade.php    # Admin email (plain)
│
└── FormServiceProvider.php                   # Service provider
```

## Core Components

### 1. Service Provider

**File**: `src/FormServiceProvider.php`

```php
class FormServiceProvider extends AbstractServiceProvider
{
    protected string $module = 'form';

    public function boot(): void
    {
        parent::boot();
        AliasLoader::getInstance()->alias('FormItem', FormItem::class);
    }
}
```

**Responsibilities**:
- Extends `Bongo\Framework\Providers\AbstractServiceProvider`
- Automatic bootstrapping via parent `boot()`:
  - Registers config from `src/Config/form.php`
  - Registers routes from `src/Routes/` (backend.php, frontend.php)
  - Registers views from `src/Views/form/`
  - Registers migrations from `src/Migrations/`
  - Registers translations from `src/Translations/`
- Registers global alias: `FormItem` → `Bongo\Form\Models\FormItem`

### 2. Models

#### Form Model

**File**: `src/Models/Form.php`

```
┌─────────────────────────────────────────┐
│             Form Model                  │
├─────────────────────────────────────────┤
│ Extends: AbstractModel                  │
│ Traits:                                 │
│   - SoftDeletes                         │
│   - HasKey (from bongo/framework)       │
│   - HasUUID (from bongo/framework)      │
├─────────────────────────────────────────┤
│ Fields:                                 │
│   - id (PK)                             │
│   - uuid (indexed)                      │
│   - name                                │
│   - key (indexed, auto-generated)       │
│   - recipients (comma-separated)        │
│   - subject                             │
│   - success_url                         │
│   - created_by, updated_by, deleted_by  │
│   - timestamps, soft deletes            │
├─────────────────────────────────────────┤
│ Relationships:                          │
│   items(): HasMany → FormItem           │
│                (ordered by sort_order)  │
├─────────────────────────────────────────┤
│ Key Methods:                            │
│   hasItems(): bool                      │
│   getRecipientsAsArray(): ?array        │
└─────────────────────────────────────────┘
```

**Key Implementation Details**:

```php
class Form extends AbstractModel
{
    use SoftDeletes, HasKey, HasUUID;

    protected $fillable = [
        'name',
        'recipients',
        'subject',
        'success_url',
    ];

    public function items(): HasMany
    {
        return $this->hasMany(FormItem::class, 'form_id')
            ->orderBy('sort_order', 'ASC');
    }

    public function hasItems(): bool
    {
        return !is_null($this->items) && count($this->items);
    }

    public function getRecipientsAsArray(): ?array
    {
        return explode(',', preg_replace('/\s+/', '', $this->recipients));
    }
}
```

#### FormItem Model

**File**: `src/Models/FormItem.php`

```
┌─────────────────────────────────────────────────────────────┐
│                    FormItem Model                           │
├─────────────────────────────────────────────────────────────┤
│ Extends: AbstractModel                                      │
│ Traits:                                                     │
│   - HasUUID (from bongo/framework)                          │
│   - HasType (type checking methods)                         │
├─────────────────────────────────────────────────────────────┤
│ Constants:                                                  │
│   REQUIRED = 'yes', NOT_REQUIRED = 'no'                     │
│   SHOW_LABEL = 'yes', HIDE_LABEL = 'no'                     │
│                                                             │
│   Field Types:                                              │
│   - CHECKBOX, DATE, EMAIL, FILE, HTML                       │
│   - IMAGE, INPUT, SELECT, TEXT, TEXTAREA                    │
├─────────────────────────────────────────────────────────────┤
│ Fields:                                                     │
│   - id (PK)                                                 │
│   - uuid (indexed)                                          │
│   - form_id (FK → forms, cascades)                          │
│   - label, display_label, name                              │
│   - class, width, placeholder                               │
│   - options (pipe/comma separated)                          │
│   - required (enum: yes/no)                                 │
│   - type (enum: see constants)                              │
│   - sort_order (indexed)                                    │
│   - created_by, updated_by                                  │
│   - timestamps                                              │
├─────────────────────────────────────────────────────────────┤
│ Key Methods:                                                │
│   getRulesAttribute(): array                                │
│   isRequired(): bool                                        │
│   labelIsVisible(): bool                                    │
│   labelIsHidden(): bool                                     │
│   hasWidth(): bool                                          │
│   hasOptions(): bool                                        │
│   getOptionsAsArray(): ?array                               │
│   setOptionsAttribute($value)                               │
│                                                             │
│ From HasType Trait:                                         │
│   isCheckbox(), isDate(), isEmail(),                        │
│   isFile(), isHtml(), isImage(),                            │
│   isInput(), isSelect(), isText(),                          │
│   isTextArea()                                              │
│                                                             │
│   scopeCheckbox($query), scopeEmail($query),                │
│   etc. (and their scopeNot* counterparts)                   │
└─────────────────────────────────────────────────────────────┘
```

**Dynamic Validation Rules**:

```php
public function getRulesAttribute(): array
{
    $rules = [];

    // Required/nullable
    if ($this->isRequired()) {
        $rules = array_merge($rules, ['required']);
    } else {
        $rules = array_merge($rules, ['nullable']);
    }

    // Image validation
    if ($this->isImage()) {
        $rules = array_merge($rules, [
            'max:10240',
            'mimes:' . config('form.allowed_image_types'),
        ]);
    }

    // File validation
    elseif ($this->isFile()) {
        $rules = array_merge($rules, [
            'max:10240',
            'mimes:' . config('form.allowed_document_types'),
        ]);
    }

    // Email validation
    if ($this->isEmail()) {
        $rules = array_merge($rules, ['email']);
    }

    return $rules;
}
```

### 3. Services

#### FormHandler Service

**File**: `src/Services/FormHandler.php`

```
┌────────────────────────────────────────────────────────────┐
│                     FormHandler                            │
├────────────────────────────────────────────────────────────┤
│ Responsibilities:                                          │
│   - Parse form items into fields array                     │
│   - Validate submitted data                                │
│   - Handle file uploads to temporary storage               │
│   - Purge old temporary files                              │
│   - Return processed fields for email                      │
├────────────────────────────────────────────────────────────┤
│ Constructor:                                               │
│   __construct(Form $form, Request $request)                │
│     → Calls setFieldsArray()                               │
│     → Calls addValuesToFieldsArray()                       │
├────────────────────────────────────────────────────────────┤
│ Public Methods:                                            │
│   validateFields(): void                                   │
│     → Builds validation rules from FormItem->rules         │
│     → Adds reCAPTCHA validation if enabled                 │
│     → Validates using Laravel Validator                    │
│                                                            │
│   storeFiles(): void                                       │
│     → Stores FILE/IMAGE uploads to temp storage            │
│     → Replaces file objects with public URLs               │
│                                                            │
│   purgeOldFiles(): void                                    │
│     → Removes files older than config days                 │
│     → Runs after successful submission                     │
│                                                            │
│   getFields(): array                                       │
│     → Returns fields array (excluding empty values)        │
├────────────────────────────────────────────────────────────┤
│ Protected Methods:                                         │
│   setFieldsArray(): void                                   │
│     → Creates initial fields structure from form items     │
│     → Excludes TEXT type (display only)                    │
│                                                            │
│   addValuesToFieldsArray(): void                           │
│     → Populates fields with submitted values               │
│     → Handles both regular fields and file uploads         │
│                                                            │
│   generateFileName($file): string                          │
│     → Slugifies filename + timestamp                       │
└────────────────────────────────────────────────────────────┘
```

**Form Submission Flow**:

```
┌──────────────────────────────────────────────────────────────┐
│             Frontend Form Submission Flow                     │
└──────────────────────────────────────────────────────────────┘

1. User submits form
   ↓
2. Frontend\FormController::store(Request)
   ↓
3. Find Form by UUID from request
   ↓
4. Create FormHandler($form, $request)
   │  ├─→ setFieldsArray() (from form items)
   │  └─→ addValuesToFieldsArray() (from request)
   ↓
5. $formHandler->validateFields()
   │  ├─→ Build validation rules from FormItem->rules
   │  ├─→ Add reCAPTCHA validation if enabled
   │  └─→ Validator::make()->validate()
   ↓
6. $formHandler->storeFiles()
   │  └─→ Upload FILE/IMAGE types to temporary storage
   ↓
7. Extract email addresses from email fields
   ↓
8. Send FormMailable to submitter (if email fields exist)
   ↓
9. Send AdminFormMailable to form recipients
   ↓
10. $formHandler->purgeOldFiles() (if config enabled)
    ↓
11. Redirect to success_url with success message
```

### 4. Controllers

#### Backend FormController

**File**: `src/Http/Controllers/Backend/FormController.php`

```
┌────────────────────────────────────────────────────────────┐
│          Backend\FormController                            │
├────────────────────────────────────────────────────────────┤
│ Extends: AbstractController                                │
│ Constructor: Injects Form model                            │
├────────────────────────────────────────────────────────────┤
│ Methods:                                                   │
│                                                            │
│   index(): View                                            │
│     → Returns 'form::backend.index'                        │
│     → Lists all forms (uses DataTable)                     │
│                                                            │
│   create(): View                                           │
│     → Returns 'form::backend.create'                       │
│                                                            │
│   store(StoreFormRequest): RedirectResponse               │
│     → Creates form                                         │
│     → Calls handleFormItems()                              │
│     → Dispatches FormCreated event                         │
│     → Redirects to show page                               │
│                                                            │
│   show(Form): View                                         │
│     → Eager loads items                                    │
│     → Returns 'form::backend.show'                         │
│                                                            │
│   edit(Form): View                                         │
│     → Eager loads items                                    │
│     → Returns 'form::backend.edit'                         │
│                                                            │
│   update(UpdateFormRequest, Form): RedirectResponse       │
│     → Updates form                                         │
│     → Calls handleFormItems()                              │
│     → Dispatches FormUpdated event                         │
│     → Redirects to show page                               │
│                                                            │
│   destroy(Form)                                            │
│     → Soft deletes form (cascades to items)                │
│     → Dispatches FormDeleted event                         │
│     → Smart redirect (back or to index)                    │
│                                                            │
│   handleFormItems(Form): void (private)                    │
│     → Deletes removed items                                │
│     → Creates/updates items from request                   │
│     → Sanitizes HTML fields                                │
│     → Clears options for non-SELECT types                  │
│     → Sets sort_order if missing                           │
└────────────────────────────────────────────────────────────┘
```

**handleFormItems() Logic**:

```
┌──────────────────────────────────────────────────────────┐
│              handleFormItems() Flow                       │
└──────────────────────────────────────────────────────────┘

1. Get items from request('items')
   │
   ├─ If empty → Delete all form items → RETURN
   │
   └─ If not empty → Continue
   ↓
2. Delete removed items
   │  Form->items()->whereNotIn('id', array_keys($items))->delete()
   ↓
3. Loop through each item:
   │
   ├─→ Set form_id
   │
   ├─→ If type is HTML:
   │     └─ Sanitize label: strip_tags($label, config('form.allowed_html_tags'))
   │
   ├─→ If type is NOT SELECT:
   │     └─ Clear options: $item['options'] = null
   │
   ├─→ If sort_order is empty:
   │     └─ Set to array key
   │
   └─→ Update or create:
         ├─ If FormItem::find($item['id']) exists → update()
         └─ Else → create()
```

#### Backend FormDatatableController

**File**: `src/Http/Controllers/Backend/FormDatatableController.php`

```php
class FormDatatableController extends AbstractDatatableController
{
    protected Form $form;

    public function __construct(Form $form)
    {
        $this->form = $form;
    }

    protected function getBaseQuery(): Builder
    {
        return $this->form->newQuery();
    }
}
```

**Purpose**: Provides DataTable data endpoint for `backend.form.datatable` route.

#### Frontend FormController

**File**: `src/Http/Controllers/Frontend/FormController.php`

```
┌────────────────────────────────────────────────────────────┐
│          Frontend\FormController                           │
├────────────────────────────────────────────────────────────┤
│ Extends: AbstractController                                │
├────────────────────────────────────────────────────────────┤
│ Methods:                                                   │
│                                                            │
│   store(Request): RedirectResponse                         │
│     → Get form by UUID (ref)                               │
│     → Create FormHandler                                   │
│     → Validate fields                                      │
│     → Store files                                          │
│     → Extract email fields                                 │
│     → Send FormMailable to submitters                      │
│     → Send AdminFormMailable to recipients                 │
│     → Purge old files (if config enabled)                  │
│     → Redirect to success_url                              │
│                                                            │
│   preview(string $uuid): View                              │
│     → Developer-only preview                               │
│     → Requires auth + developer middleware                 │
│     → Returns 'form::frontend.preview'                     │
└────────────────────────────────────────────────────────────┘
```

### 5. Events

**Files**: `src/Events/FormCreated.php`, `FormUpdated.php`, `FormDeleted.php`

```php
class FormCreated
{
    use Dispatchable, InteractsWithSockets, SerializesModels;

    public Form $form;

    public function __construct(Form $form)
    {
        $this->form = $form;
    }
}
```

**Usage**: Listen to these events in your application's `EventServiceProvider` to perform additional actions when forms are created, updated, or deleted.

### 6. Mailables

#### FormMailable

**File**: `src/Mailables/FormMailable.php`

```
┌────────────────────────────────────────────────────────────┐
│                    FormMailable                            │
├────────────────────────────────────────────────────────────┤
│ Purpose: Confirmation email sent to form submitter         │
├────────────────────────────────────────────────────────────┤
│ Recipients: Email addresses from email-type fields         │
│                                                            │
│ From: config('settings.mail_from_address')                 │
│       config('settings.mail_from_name')                    │
│                                                            │
│ Reply-To: Form recipients (admin addresses)                │
│                                                            │
│ Subject: $form->subject                                    │
│                                                            │
│ Views:                                                     │
│   - form::mail.response (HTML)                             │
│   - form::mail.response_plain (plain text)                 │
│                                                            │
│ Data:                                                      │
│   - $form: Form model                                      │
│   - $fields: Array of submitted field data                 │
└────────────────────────────────────────────────────────────┘
```

#### AdminFormMailable

**File**: `src/Mailables/AdminFormMailable.php`

```
┌────────────────────────────────────────────────────────────┐
│                 AdminFormMailable                          │
├────────────────────────────────────────────────────────────┤
│ Purpose: Notification email sent to form administrators    │
├────────────────────────────────────────────────────────────┤
│ Recipients: Form recipients (from $form->recipients)        │
│                                                            │
│ From: config('settings.mail_from_address')                 │
│       config('settings.mail_from_name')                    │
│                                                            │
│ Reply-To: Submitter email addresses (if provided)          │
│                                                            │
│ Subject: $form->subject                                    │
│                                                            │
│ Views:                                                     │
│   - form::mail.admin_response (HTML)                       │
│   - form::mail.admin_response_plain (plain text)           │
│                                                            │
│ Data:                                                      │
│   - $form: Form model                                      │
│   - $fields: Array of submitted field data                 │
│   - $emailFields: Array of submitter email addresses       │
└────────────────────────────────────────────────────────────┘
```

## Routing Architecture

### Backend Routes

**File**: `src/Routes/backend.php`

**Prefix**: `config('form.prefix')` (default: `/admin/forms`)
**Route Name Prefix**: `backend.form.*`
**Middleware**: `auth`, `employee` (from AbstractServiceProvider)

```
Route Structure:
└─ /admin/forms
   ├─ GET  /                   → backend.form.index (FormController@index)
   ├─ GET  /create             → backend.form.create (FormController@create)
   ├─ POST /store              → backend.form.store (FormController@store)
   ├─ GET  /datatable          → backend.form.datatable (FormDatatableController@index)
   └─ /{form}
      ├─ GET    /              → backend.form.show (FormController@show)
      ├─ GET    /edit          → backend.form.edit (FormController@edit)
      ├─ POST   /update        → backend.form.update (FormController@update)
      └─ DELETE /delete        → backend.form.destroy (FormController@destroy)
```

### Frontend Routes

**File**: `src/Routes/frontend.php`

**Prefix**: `config('form.prefix')` (default: `/forms`)
**Route Name Prefix**: `form.*`
**Middleware**: Various (see below)

```
Route Structure:
└─ /forms
   ├─ GET  /preview/{uuid}     → form.preview (FormController@preview)
   │                              Middleware: auth, developer
   │
   └─ POST /store               → form.store (FormController@store)
                                  Middleware: ProtectAgainstSpam (honeypot)
```

## Database Schema

### forms Table

```
┌────────────────────┬──────────────┬─────────────────────────┐
│ Column             │ Type         │ Description             │
├────────────────────┼──────────────┼─────────────────────────┤
│ id                 │ INT (PK)     │ Primary key             │
│ uuid               │ UUID (IDX)   │ Public identifier       │
│ name               │ VARCHAR      │ Form name               │
│ key                │ VARCHAR(IDX) │ URL-safe key (auto)     │
│ success_url        │ VARCHAR      │ Redirect after submit   │
│ recipients         │ VARCHAR      │ Comma-separated emails  │
│ subject            │ VARCHAR      │ Email subject           │
│ created_by         │ INT (IDX)    │ Audit: creator user ID  │
│ updated_by         │ INT (IDX)    │ Audit: updater user ID  │
│ deleted_by         │ INT (IDX)    │ Audit: deleter user ID  │
│ created_at         │ TIMESTAMP    │ Creation timestamp      │
│ updated_at         │ TIMESTAMP    │ Last update timestamp   │
│ deleted_at         │ TIMESTAMP    │ Soft delete timestamp   │
└────────────────────┴──────────────┴─────────────────────────┘
```

### form_items Table

```
┌────────────────────┬──────────────┬─────────────────────────────────┐
│ Column             │ Type         │ Description                     │
├────────────────────┼──────────────┼─────────────────────────────────┤
│ id                 │ INT (PK)     │ Primary key                     │
│ uuid               │ UUID (IDX)   │ Public identifier               │
│ form_id            │ INT (FK,IDX) │ FK to forms (cascade delete)    │
│ label              │ TEXT         │ Field label (or HTML content)   │
│ display_label      │ VARCHAR      │ Show/hide label (yes/no)        │
│ name               │ VARCHAR      │ Field name attribute            │
│ class              │ VARCHAR      │ CSS classes                     │
│ width              │ VARCHAR      │ Custom width (e.g. "50%")       │
│ placeholder        │ VARCHAR      │ Placeholder text                │
│ options            │ TEXT         │ Pipe/comma separated options    │
│ required           │ ENUM         │ yes/no (see constants)          │
│ type               │ ENUM         │ Field type (see constants)      │
│ sort_order         │ INT (IDX)    │ Display order                   │
│ created_by         │ INT (IDX)    │ Audit: creator user ID          │
│ updated_by         │ INT (IDX)    │ Audit: updater user ID          │
│ created_at         │ TIMESTAMP    │ Creation timestamp              │
│ updated_at         │ TIMESTAMP    │ Last update timestamp           │
└────────────────────┴──────────────┴─────────────────────────────────┘

Constraints:
  - FK form_id REFERENCES forms(id) ON DELETE CASCADE ON UPDATE CASCADE
```

## Field Type System

### Supported Field Types

| Type Constant | Value      | Description                    | Validation                  |
|---------------|------------|--------------------------------|-----------------------------|
| INPUT         | input      | Text input field               | Required/nullable           |
| EMAIL         | email      | Email input field              | Required/nullable + email   |
| TEXTAREA      | textarea   | Multi-line text area           | Required/nullable           |
| SELECT        | select     | Dropdown select (with options) | Required/nullable           |
| CHECKBOX      | checkbox   | Checkbox field                 | Required/nullable           |
| DATE          | date       | Date picker field              | Required/nullable           |
| FILE          | file       | File upload field              | Max 10MB + mime types       |
| IMAGE         | image      | Image upload field             | Max 10MB + image mime types |
| TEXT          | text       | Display-only text (no input)   | N/A (excluded from form)    |
| HTML          | html       | HTML content display           | N/A (sanitized on save)     |

### Field Type Flow

```
┌──────────────────────────────────────────────────────────────┐
│             Field Type Processing Flow                       │
└──────────────────────────────────────────────────────────────┘

Backend (Form Creation/Update):
  1. Admin selects type from enum dropdown
  2. If type is HTML:
     └─→ Sanitize label: strip_tags($label, config('form.allowed_html_tags'))
  3. If type is not SELECT:
     └─→ Clear options field
  4. Save to form_items table

Frontend (Form Rendering):
  1. Load form with items (ordered by sort_order)
  2. Loop through items:
     └─→ @include("form::frontend.partials.{$item->type}")
  3. Each partial renders appropriate field with:
     - Label (if display_label == 'yes')
     - Field attributes (name, class, placeholder, required, etc.)
     - Options (for SELECT type)

Frontend (Form Submission):
  1. FormHandler extracts items (excluding TEXT type)
  2. Build fields array with:
     - label, name, type, rules, value
  3. Validate fields:
     - Required/nullable based on FormItem->required
     - Email validation for EMAIL type
     - File/image validation with max size + mime types
  4. Store files to temporary storage
  5. Return processed fields for email
```

## Configuration System

**File**: `src/Config/form.php`

```php
return [
    // URL prefix for routes
    'prefix' => 'forms',

    // Allowed mime types for uploads
    'allowed_image_types' => 'png,jpg,jpeg,gif',
    'allowed_document_types' => 'png,jpg,jpeg,gif,pdf',

    // File purging (temporary uploads)
    'purge_files' => true,          // Enable auto-purge
    'purge_files_after' => 60,      // Days before deletion

    // HTML field sanitization
    'allowed_html_tags' => '<p><h1><h2><h3><h4><h5><h6><strong><span><br>',

    // reCAPTCHA v3
    'recaptcha' => [
        'enabled' => true,          // Enable/disable globally
        'min_score' => 0.5,         // Min score (0.1 = bot, 1.0 = human)
    ],
];
```

## Extension Points

### 1. Adding a New Field Type

**Steps**:

1. Add constant to `FormItem` model:
   ```php
   public const NEWTYPE = 'newtype';
   ```

2. Add migration to update `type` enum in `form_items` table.

3. Add type checker methods to `HasType` trait:
   ```php
   public function scopeNewType($query)
   {
       return $query->where('type', self::NEWTYPE);
   }

   public function isNewType(): bool
   {
       return $this->type === self::NEWTYPE;
   }
   ```

4. Update `getRulesAttribute()` in `FormItem` if custom validation needed:
   ```php
   if ($this->isNewType()) {
       $rules = array_merge($rules, ['custom_rule', 'min:10']);
   }
   ```

5. Create blade partial: `src/Views/frontend/partials/newtype.blade.php`

6. Update form rendering logic in `src/Views/frontend/partials/form.blade.php`

### 2. Customizing Validation

Override `getRulesAttribute()` in `FormItem` model or extend FormHandler service.

### 3. Adding Custom Email Templates

Create listeners for `FormCreated`, `FormUpdated`, `FormDeleted` events and send custom emails.

### 4. Customizing File Storage

Extend `FormHandler` service and override `storeFiles()` method to use custom storage disk/path.

### 5. Adding Custom Middleware

Modify routes in `src/Routes/backend.php` or `frontend.php` to add custom middleware.

## Security Architecture

### 1. Input Validation

```
┌──────────────────────────────────────────────────────────────┐
│                  Validation Layers                           │
└──────────────────────────────────────────────────────────────┘

Backend (Admin Forms):
  ├─→ StoreFormRequest / UpdateFormRequest
  │     └─ Validates form metadata (name, recipients, etc.)
  │
  └─→ Backend\FormController::handleFormItems()
        └─ Sanitizes HTML fields with strip_tags()

Frontend (Form Submissions):
  └─→ FormHandler::validateFields()
        ├─ Builds rules from FormItem->rules attribute
        ├─ Adds reCAPTCHA v3 validation (if enabled)
        └─ Uses Laravel Validator::make()->validate()
```

### 2. File Upload Security

- **Max file size**: 10MB (10240 KB)
- **Image mime types**: `png,jpg,jpeg,gif` (configurable)
- **Document mime types**: `png,jpg,jpeg,gif,pdf` (configurable)
- **Storage location**: Temporary storage (auto-purged)
- **Filename sanitization**: Slugified + timestamped

### 3. Spam Prevention

- **Honeypot**: `ProtectAgainstSpam` middleware on `form.store` route
- **reCAPTCHA v3**: Score-based validation (0.5 minimum by default)
  - Action key: Generated from form name via `make_key($form->name)`
  - Validates score using `Bongo\Captcha\Rules\Captcha` rule

### 4. HTML Sanitization

- **HTML field type**: Strips tags to allowed list from config
- **Allowed tags**: `<p><h1><h2><h3><h4><h5><h6><strong><span><br>` (configurable)
- **Sanitization point**: `Backend\FormController::handleFormItems()`

### 5. Authorization

- **Backend routes**: Protected by `auth` + `employee` middleware
- **Preview route**: Protected by `auth` + `developer` middleware
- **Frontend store**: Public (protected by honeypot + reCAPTCHA)

## Testing Strategy

### Unit Tests

**Location**: `tests/` directory

**Recommended Test Coverage**:

1. **Model Tests**:
   - Form: `hasItems()`, `getRecipientsAsArray()`
   - FormItem: `getRulesAttribute()`, type checkers, `getOptionsAsArray()`

2. **Service Tests**:
   - FormHandler: `validateFields()`, `storeFiles()`, `purgeOldFiles()`

3. **Trait Tests**:
   - HasType: Query scopes and boolean methods

### Feature Tests

1. **Backend CRUD**:
   - Create form with items
   - Update form and modify items
   - Delete form (cascades to items)
   - Verify events dispatched

2. **Frontend Submission**:
   - Submit form with various field types
   - Validate file uploads
   - Verify emails sent
   - Test reCAPTCHA validation

3. **Validation**:
   - Required/nullable fields
   - Email field validation
   - File/image mime type validation
   - reCAPTCHA score validation

### Running Tests

```bash
# From package root
vendor/bin/phpunit

# With coverage
vendor/bin/phpunit --coverage-html coverage
```

## Performance Considerations

### 1. Query Optimization

- **Eager loading**: Always use `$form->loadMissing('items')` before rendering
- **Indexed columns**: `uuid`, `key`, `sort_order`, `form_id`
- **DataTable**: Uses `AbstractDatatableController` for pagination

### 2. File Storage

- **Temporary storage**: Files stored in temp location (not permanent)
- **Auto-purge**: Old files deleted after configured days (default 60)
- **Storage disk**: Configurable via `config('document.tmp_path')`

### 3. Validation

- **Dynamic rules**: Built from `FormItem->rules` attribute (cached by model)
- **Conditional reCAPTCHA**: Only validated if `setting()->reCaptchaEnabled()`

## Dependencies

### bongo/framework

**Usage**:
- `AbstractModel` - Base model with audit trails
- `AbstractController` - Base controller
- `AbstractServiceProvider` - Automatic bootstrapping
- `AbstractDatatableController` - DataTable integration
- `HasKey` - Auto-generates URL-safe key from name
- `HasUUID` - Auto-generates UUID
- `make_key()` - Helper function for generating keys
- `setting()->reCaptchaEnabled()` - Check reCAPTCHA status

### bongo/captcha

**Usage**:
- `Bongo\Captcha\Rules\Captcha` - reCAPTCHA v3 validation rule
- Validates score-based submissions with action keys

### spatie/laravel-honeypot

**Usage**:
- `ProtectAgainstSpam` middleware on frontend store route
- Prevents bot submissions

## Troubleshooting

### Common Issues

1. **Forms not saving items**:
   - Check `request('items')` structure
   - Verify `form_id` is set correctly
   - Check foreign key constraint

2. **Files not uploading**:
   - Verify `config('document.tmp_path')` exists
   - Check storage permissions
   - Validate mime types against config

3. **Emails not sending**:
   - Check `$form->recipients` is set
   - Verify `config('settings.mail_from_address')` is set
   - Check mail configuration (`.env`)

4. **reCAPTCHA validation failing**:
   - Verify `config('form.recaptcha.enabled')` is true
   - Check `setting()->reCaptchaEnabled()` returns true
   - Validate `min_score` threshold (0.1-1.0)
   - Verify `make_key($form->name)` matches frontend action key

5. **Items not ordered correctly**:
   - Check `sort_order` values
   - Verify `orderBy('sort_order', 'ASC')` in relationship

## Migration History

1. `2021_01_01_000001` - Create `forms` table
2. `2021_01_01_000002` - Create `form_items` table
3. `2021_01_01_000003` - Add additional types to `form_items`
4. `2021_01_01_000004` - Add `placeholder` column
5. `2021_01_01_000005` - Add `email` type
6. `2021_01_01_000006` - Add `display_label` column
7. `2021_01_01_000007` - Add `width` column
8. `2024_01_01_000008` - Add `html` type

Each migration is idempotent (checks `Schema::hasTable()` before creating).
