# Claude Code Guidance - Bongo Form Package

## Overview

This is the `bongo/form` package, a Laravel package for creating and managing custom forms with dynamic field types, validation, file uploads, and email notifications.

**Package**: `bongo/form`
**Namespace**: `Bongo\Form`
**PHP**: 8.2+ | **Laravel**: 10+

## Quick Links

- **Detailed Architecture**: See `ARCHITECTURE.md` for comprehensive documentation
- **Cursor AI Instructions**: See `.cursorrules` for development guidelines
- **GitHub Copilot**: See `.github/copilot-instructions.md` for code patterns

## Commands

From the package root directory:

```bash
# Testing
vendor/bin/phpunit                    # Run tests

# Code Style (Laravel Pint)
vendor/bin/pint                       # Fix code style
vendor/bin/pint --test                # Check code style

# Dependencies
composer install                      # Install dependencies
composer update -W                    # Update dependencies
rm -f composer.lock && composer update -W  # Force update
```

## Package Architecture Quick Reference

### Extends Bongo Framework

This package extends `Bongo\Framework\Providers\AbstractServiceProvider`:

- **Module name**: `form`
- **Automatic bootstrapping**:
  - Config: `src/Config/form.php`
  - Routes: `src/Routes/backend.php`, `src/Routes/frontend.php`
  - Views: `src/Views/form/`
  - Migrations: `src/Migrations/`
  - Translations: `src/Translations/`
- **Route middleware**:
  - `backend.php` → `auth` + `employee` middleware
  - `frontend.php` → Various (honeypot on store)

### Core Models

#### Form (`Bongo\Form\Models\Form`)
- Extends: `AbstractModel`
- Traits: `SoftDeletes`, `HasKey`, `HasUUID`
- Relationship: `items()` → HasMany `FormItem` (ordered by `sort_order`)
- Key Methods:
  - `hasItems(): bool`
  - `getRecipientsAsArray(): ?array`

#### FormItem (`Bongo\Form\Models\FormItem`)
- Extends: `AbstractModel`
- Traits: `HasUUID`, `HasType`
- Field Types: `CHECKBOX`, `DATE`, `EMAIL`, `FILE`, `HTML`, `IMAGE`, `INPUT`, `SELECT`, `TEXT`, `TEXTAREA`
- Key Methods:
  - `getRulesAttribute(): array` - Dynamic validation rules
  - `isRequired(): bool`
  - `hasOptions(): bool`
  - `getOptionsAsArray(): ?array`
  - Type checkers: `isEmail()`, `isFile()`, `isImage()`, etc.

### Core Service

#### FormHandler (`Bongo\Form\Services\FormHandler`)
Processes form submissions:
- `validateFields(): void` - Validates with reCAPTCHA
- `storeFiles(): void` - Handles file uploads
- `purgeOldFiles(): void` - Cleans old files
- `getFields(): array` - Returns processed data

### Controllers

- **Backend\FormController** - CRUD operations (index, create, store, show, edit, update, destroy)
- **Backend\FormDatatableController** - DataTable integration
- **Frontend\FormController** - Form submission (store) and preview

### Events

- `FormCreated` - Dispatched after form creation
- `FormUpdated` - Dispatched after form update
- `FormDeleted` - Dispatched after form deletion

### Mailables

- `FormMailable` - Confirmation email to submitter
- `AdminFormMailable` - Notification email to admin

## Key Files Reference

| File Path | Purpose |
|-----------|---------|
| `src/FormServiceProvider.php` | Service provider (extends AbstractServiceProvider) |
| `src/Models/Form.php` | Main form model with SoftDeletes |
| `src/Models/FormItem.php` | Form field model with type constants |
| `src/Services/FormHandler.php` | Form submission processor |
| `src/Traits/HasType.php` | Type checking methods for FormItem |
| `src/Config/form.php` | Configuration (file types, reCAPTCHA, purging) |
| `src/Routes/backend.php` | Admin routes (prefix: /admin/forms) |
| `src/Routes/frontend.php` | Public routes (prefix: /forms) |
| `src/Http/Controllers/Backend/FormController.php` | Admin CRUD controller |
| `src/Http/Controllers/Frontend/FormController.php` | Frontend submission controller |
| `src/Events/FormCreated.php` | Form created event |
| `src/Mailables/FormMailable.php` | Customer email |
| `src/Mailables/AdminFormMailable.php` | Admin notification email |
| `src/Views/frontend/partials/*.blade.php` | Field type templates (input, email, select, etc.) |

## Common Tasks

### Creating a New Form (Backend)

```php
// Backend\FormController@store
$form = Form::create($request->all());

// Handle form items
foreach ($items as $item) {
    $item['form_id'] = $form->id;

    // Sanitize HTML fields
    if ($item['type'] === FormItem::HTML) {
        $item['label'] = strip_tags($item['label'], config('form.allowed_html_tags'));
    }

    FormItem::create($item);
}

event(new FormCreated($form));
```

### Processing Form Submission (Frontend)

```php
// Frontend\FormController@store
$form = Form::where('uuid', $request->get('ref'))->first();

$formHandler = new FormHandler($form, $request);
$formHandler->validateFields();  // Validates + reCAPTCHA
$formHandler->storeFiles();      // Uploads to temp storage

$fields = $formHandler->getFields();

// Extract email addresses
$emailFields = collect($fields)
    ->filter(fn($field) => $field['type'] == FormItem::EMAIL)
    ->pluck('value')
    ->toArray();

// Send emails
Mail::to($emailFields)->send(new FormMailable($form, $fields));
Mail::to($form->getRecipientsAsArray())->send(new AdminFormMailable($form, $fields, $emailFields));

// Cleanup
$formHandler->purgeOldFiles();

return redirect()->to($form->success_url);
```

### Querying Form Items by Type

```php
// Get all email fields
$emailFields = $form->items()->email()->get();

// Get all required fields
$requiredFields = $form->items()
    ->where('required', FormItem::REQUIRED)
    ->get();

// Get all fields except text (display-only)
$inputFields = $form->items()->notText()->get();

// Check field type
if ($item->isImage()) {
    // Handle image upload
}
```

### Adding Custom Validation

```php
// Extend FormItem->getRulesAttribute()
public function getRulesAttribute(): array
{
    $rules = $this->isRequired() ? ['required'] : ['nullable'];

    if ($this->isEmail()) {
        $rules[] = 'email';
    }

    if ($this->isCustomType()) {
        $rules[] = 'custom_rule';
        $rules[] = 'min:10';
    }

    return $rules;
}
```

## Configuration

**File**: `src/Config/form.php`

```php
'prefix' => 'forms',                           // URL prefix
'allowed_image_types' => 'png,jpg,jpeg,gif',   // Image mimes
'allowed_document_types' => 'png,jpg,jpeg,gif,pdf', // Document mimes
'purge_files' => true,                         // Auto-delete old files
'purge_files_after' => 60,                     // Days before purging
'allowed_html_tags' => '<p><h1><h2>...',       // HTML field sanitization
'recaptcha' => [
    'enabled' => true,                         // Enable reCAPTCHA
    'min_score' => 0.5,                        // Min score (0.1-1.0)
],
```

## Routes

### Backend (prefix: `/admin/forms`, middleware: `auth`, `employee`)

- `backend.form.index` - GET `/` - List forms
- `backend.form.create` - GET `/create` - Create form
- `backend.form.store` - POST `/store` - Store form
- `backend.form.datatable` - GET `/datatable` - DataTable data
- `backend.form.show` - GET `/{form}` - View form
- `backend.form.edit` - GET `/{form}/edit` - Edit form
- `backend.form.update` - POST `/{form}/update` - Update form
- `backend.form.destroy` - DELETE `/{form}/delete` - Delete form

### Frontend (prefix: `/forms`)

- `form.preview` - GET `/preview/{uuid}` - Preview (auth + developer)
- `form.store` - POST `/store` - Submit form (honeypot protected)

## Database Schema

### forms Table
- `id`, `uuid`, `name`, `key`
- `success_url`, `recipients`, `subject`
- Audit: `created_by`, `updated_by`, `deleted_by`
- Timestamps + soft deletes

### form_items Table
- `id`, `uuid`, `form_id` (FK → forms)
- `label`, `display_label`, `name`, `class`, `width`, `placeholder`
- `options` (text), `required` (enum), `type` (enum)
- `sort_order`
- Audit: `created_by`, `updated_by`
- Timestamps

## Field Types

| Type | Value | Description | Validation |
|------|-------|-------------|------------|
| INPUT | input | Text input | Required/nullable |
| EMAIL | email | Email input | Required/nullable + email |
| TEXTAREA | textarea | Multi-line text | Required/nullable |
| SELECT | select | Dropdown (with options) | Required/nullable |
| CHECKBOX | checkbox | Checkbox | Required/nullable |
| DATE | date | Date picker | Required/nullable |
| FILE | file | File upload | Max 10MB + mimes |
| IMAGE | image | Image upload | Max 10MB + image mimes |
| TEXT | text | Display text (no input) | N/A |
| HTML | html | HTML content | Sanitized on save |

## Code Style

This package follows the Laravel code style preset:

- **Method signatures**: Always include return types
- **Type hints**: Use for parameters and properties
- **Constants**: Use for enum values (e.g., `FormItem::EMAIL`)
- **Boolean methods**: Prefix with `is`, `has`, or verb (e.g., `isRequired()`, `hasItems()`)
- **Query scopes**: Follow `scope*` convention (e.g., `scopeEmail($query)`)
- **Accessor methods**: Use `get*AsArray()` pattern

### Formatting

```bash
# Check style
vendor/bin/pint --test

# Fix style
vendor/bin/pint
```

## Security Features

1. **HTML Sanitization**: Uses `strip_tags()` with allowed tags from config
2. **File Upload Validation**: Max 10MB + mime type whitelist
3. **reCAPTCHA v3**: Score-based spam prevention (default 0.5)
4. **Honeypot**: Spam protection on frontend submissions
5. **Authorization**: Backend protected by `auth` + `employee` middleware
6. **Temporary Files**: Auto-purged after configurable days

## Dependencies

### bongo/framework
- Base classes: `AbstractModel`, `AbstractController`, `AbstractServiceProvider`
- Traits: `HasKey`, `HasUUID`
- Helpers: `make_key()`, `setting()->reCaptchaEnabled()`

### bongo/captcha
- `Bongo\Captcha\Rules\Captcha` - reCAPTCHA v3 validation rule

### spatie/laravel-honeypot
- `ProtectAgainstSpam` - Honeypot middleware

## Extending the Package

### Add New Field Type

1. Add constant to `FormItem`: `const NEWTYPE = 'newtype';`
2. Add migration to update `type` enum
3. Add methods to `HasType` trait: `scopeNewType()`, `isNewType()`
4. Update `getRulesAttribute()` if custom validation needed
5. Create blade partial: `src/Views/frontend/partials/newtype.blade.php`

### Listen to Events

```php
// In EventServiceProvider
use Bongo\Form\Events\FormCreated;

protected $listen = [
    FormCreated::class => [
        YourListener::class,
    ],
];
```

### Customize Email Templates

Override views in your application:
- `form::mail.response` (customer HTML)
- `form::mail.response_plain` (customer plain)
- `form::mail.admin_response` (admin HTML)
- `form::mail.admin_response_plain` (admin plain)

## Troubleshooting

**Forms not saving items**: Check `request('items')` structure and `form_id` is set

**Files not uploading**: Verify `config('document.tmp_path')` exists and has correct permissions

**Emails not sending**: Check `$form->recipients` is set and mail config is correct

**reCAPTCHA failing**: Verify `config('form.recaptcha.enabled')` and `setting()->reCaptchaEnabled()` both return true

**Items not ordered**: Check `sort_order` values in `form_items` table

## Support

- **Repository**: https://bitbucket.org/designtec/form
- **Package Registry**: https://designtecpackages.co.uk

## Related Documentation

- `ARCHITECTURE.md` - Detailed architecture and design patterns
- `.cursorrules` - Development guidelines for Cursor AI
- `.github/copilot-instructions.md` - Code patterns for GitHub Copilot
