# Bongo Form Package

A Laravel package for creating and managing custom forms with dynamic field types, validation, file uploads, and email notifications.

## Features

- **Dynamic Form Builder**: Create forms with multiple field types through an admin interface
- **10 Field Types**: Input, Email, Textarea, Select, Checkbox, Date, File, Image, Text, HTML
- **Validation**: Dynamic validation rules based on field types
- **File Uploads**: Support for image and document uploads with automatic cleanup
- **Email Notifications**: Automatic emails to both submitters and administrators
- **Spam Prevention**: Built-in reCAPTCHA v3 and honeypot protection
- **Developer Preview**: Preview forms before publishing
- **Soft Deletes**: Safe form deletion with audit trails

## Requirements

- PHP 8.2+
- Laravel 10+
- bongo/framework ^3.0
- bongo/captcha (for reCAPTCHA support)

## Installation

### Composer

```bash
composer require bongo/form
```

### Service Provider

The package uses Laravel auto-discovery. If you need to manually register it:

```php
// config/app.php
'providers' => [
    // ...
    Bongo\Form\FormServiceProvider::class,
],
```

### Publish Assets (Optional)

```bash
# Publish configuration
php artisan vendor:publish --tag=form-config

# Publish migrations
php artisan vendor:publish --tag=form-migrations

# Publish views
php artisan vendor:publish --tag=form-views

# Publish translations
php artisan vendor:publish --tag=form-translations
```

### Run Migrations

```bash
php artisan migrate
```

## Configuration

Configuration file: `config/form.php`

```php
return [
    // URL prefix for routes
    'prefix' => 'forms',

    // Allowed mime types for uploads
    'allowed_image_types' => 'png,jpg,jpeg,gif',
    'allowed_document_types' => 'png,jpg,jpeg,gif,pdf',

    // Auto-purge old temporary files
    'purge_files' => true,
    'purge_files_after' => 60, // days

    // HTML field sanitization
    'allowed_html_tags' => '<p><h1><h2><h3><h4><h5><h6><strong><span><br>',

    // reCAPTCHA v3
    'recaptcha' => [
        'enabled' => true,
        'min_score' => 0.5, // 0.1 (bot) to 1.0 (human)
    ],
];
```

## Usage

### Creating a Form (Admin)

1. Navigate to `/admin/forms`
2. Click "Create Form"
3. Fill in form details:
   - **Name**: Form identifier
   - **Recipients**: Comma-separated email addresses for notifications
   - **Subject**: Email subject line
   - **Success URL**: Redirect URL after submission
4. Add form fields with various types
5. Save the form

### Rendering a Form (Frontend)

```blade
{{-- In your view --}}
@include('form::frontend.partials.form', ['form' => $form])
```

### Processing Submissions (Automatic)

Forms are automatically processed when submitted to `route('form.store')`. The package handles:

1. Validation (including reCAPTCHA)
2. File uploads
3. Email notifications
4. Cleanup of old files

### Listening to Events

```php
// app/Providers/EventServiceProvider.php

use Bongo\Form\Events\FormCreated;
use Bongo\Form\Events\FormUpdated;
use Bongo\Form\Events\FormDeleted;

protected $listen = [
    FormCreated::class => [
        \App\Listeners\HandleFormCreated::class,
    ],
    FormUpdated::class => [
        \App\Listeners\HandleFormUpdated::class,
    ],
    FormDeleted::class => [
        \App\Listeners\HandleFormDeleted::class,
    ],
];
```

## Field Types

| Type | Description | Validation |
|------|-------------|------------|
| **Input** | Standard text input | Required/nullable |
| **Email** | Email input with validation | Required/nullable + email |
| **Textarea** | Multi-line text input | Required/nullable |
| **Select** | Dropdown with options | Required/nullable |
| **Checkbox** | Checkbox input | Required/nullable |
| **Date** | Date picker | Required/nullable |
| **File** | File upload | Max 10MB + mime types |
| **Image** | Image upload | Max 10MB + image types |
| **Text** | Display-only text (no input) | N/A |
| **HTML** | HTML content display | Sanitized on save |

## API Reference

### Models

#### Form

```php
use Bongo\Form\Models\Form;

// Create a form
$form = Form::create([
    'name' => 'Contact Form',
    'recipients' => 'admin@example.com',
    'subject' => 'New Contact Form Submission',
    'success_url' => '/thank-you',
]);

// Get form items
$items = $form->items;

// Check if form has items
if ($form->hasItems()) {
    // ...
}

// Get recipients as array
$recipients = $form->getRecipientsAsArray();
```

#### FormItem

```php
use Bongo\Form\Models\FormItem;

// Create a form item
$item = FormItem::create([
    'form_id' => $form->id,
    'label' => 'Email Address',
    'name' => 'email',
    'type' => FormItem::EMAIL,
    'required' => FormItem::REQUIRED,
    'sort_order' => 1,
]);

// Type checking
if ($item->isEmail()) {
    // Handle email field
}

if ($item->isRequired()) {
    // Mark as required
}

// Get validation rules
$rules = $item->rules;
```

### Services

#### FormHandler

```php
use Bongo\Form\Services\FormHandler;

$formHandler = new FormHandler($form, $request);

// Validate fields (includes reCAPTCHA)
$formHandler->validateFields();

// Store uploaded files
$formHandler->storeFiles();

// Get processed fields
$fields = $formHandler->getFields();

// Purge old files
$formHandler->purgeOldFiles();
```

## Routes

### Backend (Admin)

- `backend.form.index` - List all forms
- `backend.form.create` - Create form
- `backend.form.store` - Store form
- `backend.form.show` - View form details
- `backend.form.edit` - Edit form
- `backend.form.update` - Update form
- `backend.form.destroy` - Delete form
- `backend.form.datatable` - DataTable endpoint

### Frontend

- `form.store` - Submit form (POST)
- `form.preview` - Preview form (developer only)

## Security

### Spam Prevention

1. **reCAPTCHA v3**: Score-based validation (configurable threshold)
2. **Honeypot**: Invisible honeypot field via Spatie package

### File Upload Security

- Maximum file size: 10MB
- Whitelist mime types (configurable)
- Temporary storage with automatic cleanup
- Slugified filenames with timestamps

### HTML Sanitization

- HTML field type uses `strip_tags()` with configurable allowed tags
- Default allowed tags: `<p><h1><h2><h3><h4><h5><h6><strong><span><br>`

### Authorization

- Backend routes protected by `auth` + `employee` middleware
- Preview route requires `auth` + `developer` middleware

## Customization

### Custom Email Templates

Override the email views in your application:

```
resources/views/vendor/form/mail/
├── response.blade.php              # Customer email (HTML)
├── response_plain.blade.php        # Customer email (plain)
├── admin_response.blade.php        # Admin email (HTML)
└── admin_response_plain.blade.php  # Admin email (plain)
```

### Custom Field Types

1. Add constant to `FormItem` model
2. Update database enum migration
3. Add type checker methods to `HasType` trait
4. Create blade partial in `resources/views/vendor/form/frontend/partials/`
5. Update validation rules if needed

See `ARCHITECTURE.md` for detailed instructions.

## Testing

```bash
# Run tests
vendor/bin/phpunit

# Code style check
vendor/bin/pint --test

# Fix code style
vendor/bin/pint
```

## Documentation

- **ARCHITECTURE.md** - Detailed architecture and design patterns
- **CLAUDE.md** - Claude Code guidance
- **.cursorrules** - Cursor AI development guidelines
- **.github/copilot-instructions.md** - GitHub Copilot code patterns

## Troubleshooting

### Forms not saving items

Check that `request('items')` contains the correct structure and `form_id` is set.

### Files not uploading

- Verify `config('document.tmp_path')` directory exists
- Check storage permissions
- Validate file mime types against configuration

### Emails not sending

- Ensure `recipients` field is populated
- Verify mail configuration in `.env`
- Check `config('settings.mail_from_address')` is set

### reCAPTCHA validation failing

- Verify `config('form.recaptcha.enabled')` is `true`
- Check `setting()->reCaptchaEnabled()` returns `true`
- Validate score threshold is appropriate (0.1-1.0)
- Ensure frontend action key matches backend key generation

### Items not displaying in correct order

Check `sort_order` values in the `form_items` table.

## Support & Contributing

- **Repository**: https://bitbucket.org/designtec/form
- **Package Registry**: https://designtecpackages.co.uk
- **Issues**: Submit issues via Bitbucket

## License

This package is proprietary software. See LICENSE file for details.

## Credits

Developed by [Bespoke UK](https://bespokeuk.com)

- Stuart Elliott - stuart.elliott@bespokeuk.com
