<?php

namespace Bongo\Form\Models;

use Bongo\Form\Traits\HasType;
use Bongo\Framework\Models\AbstractModel;
use Bongo\Framework\Traits\HasUUID;
use Illuminate\Support\Str;

class FormItem extends AbstractModel
{
    use HasUUID;
    use HasType;

    public const REQUIRED = 'yes';
    public const NOT_REQUIRED = 'no';

    public const SHOW_LABEL = 'yes';
    public const HIDE_LABEL = 'no';

    // Types
    public const CHECKBOX = 'checkbox';
    public const DATE = 'date';
    public const EMAIL = 'email';
    public const FILE = 'file';
    public const HTML = 'html';
    public const IMAGE = 'image';
    public const INPUT = 'input';
    public const SELECT = 'select';
    public const TEXT = 'text';
    public const TEXTAREA = 'textarea';

    /**
     * @var array
     */
    protected $fillable = [
        'form_id',
        'label',
        'display_label',
        'name',
        'class',
        'width',
        'placeholder',
        'required',
        'options',
        'type',
        'sort_order',
    ];

    /**
     * @param $value
     */
    public function setOptionsAttribute($value)
    {
        if (! empty($value)) {
            $options = Str::contains($value, '|')
                ? explode('|', $value)
                : explode(',', $value);

            foreach ($options as $key => $option) {
                $options[$key] = $option;
            }
            $this->attributes['options'] = rtrim(implode('|', $options), '|');
        }
    }

    public function getRulesAttribute(): array
    {
        $rules = [];

        if ($this->isRequired()) {
            $rules = array_merge($rules, ['required']);
        } else {
            $rules = array_merge($rules, ['nullable']);
        }

        if ($this->isImage()) {
            $rules = array_merge($rules, [
                'max:10240',
                'mimes:'.config('form.allowed_image_types'),
            ]);
        } elseif ($this->isFile()) {
            $rules = array_merge($rules, [
                'max:10240',
                'mimes:'.config('form.allowed_document_types'),
            ]);
        }

        if ($this->isEmail()) {
            $rules = array_merge($rules, ['email']);
        }

        return $rules;
    }

    public function isRequired(): bool
    {
        return $this->required === self::REQUIRED;
    }

    public function labelIsVisible(): bool
    {
        return $this->display_label === self::SHOW_LABEL;
    }

    public function labelIsHidden(): bool
    {
        return $this->display_label === self::HIDE_LABEL;
    }

    public function hasWidth(): bool
    {
        return ! empty($this->width);
    }

    public function hasOptions(): bool
    {
        return $this->type === self::SELECT && ! empty($this->options);
    }

    public function getOptionsAsArray(): ?array
    {
        $options = Str::contains($this->options, '|')
            ? explode('|', $this->options)
            : explode(',', $this->options);

        foreach ($options as $key => $option) {
            $options[$key] = trim($option);
        }

        return $options;
    }
}
