<?php

namespace Bongo\Form\Services;

use Bongo\Captcha\Rules\Captcha;
use Bongo\Form\Models\Form;
use Bongo\Form\Models\FormItem;
use Exception;
use Illuminate\Contracts\Container\BindingResolutionException;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Storage;
use Illuminate\Support\Facades\Validator;
use Illuminate\Support\Str;
use Illuminate\Validation\ValidationException;

class FormHandler
{
    protected Form $form;

    protected Request $request;

    protected array $fields = [];

    /**
     * @throws Exception
     */
    public function __construct(Form $form, Request $request)
    {
        $this->form = $form;
        $this->request = $request;

        $this->setFieldsArray();
        $this->addValuesToFieldsArray();
    }

    /**
     * @throws Exception
     */
    protected function setFieldsArray(): void
    {
        // Make sure this form has items
        if (! $this->form->hasItems()) {
            throw new Exception("{$this->form->id} does not have any items");
        }

        // Create the initial data structure
        $formItems = $this->form->items()->notText()->get();
        foreach ($formItems as $item) {
            $this->fields[$item->name]['label'] = $item->label;
            $this->fields[$item->name]['name'] = $item->name;
            $this->fields[$item->name]['type'] = $item->type;
            $this->fields[$item->name]['rules'] = $item->rules;
            $this->fields[$item->name]['value'] = null;
        }
    }

    /**
     * @throws Exception
     */
    protected function addValuesToFieldsArray(): void
    {
        // Make sure we have some data
        if (! $this->request->has('fields')) {
            throw new Exception("{$this->form->id} does not have submitted fields");
        }

        // Populate the array with the standard data
        foreach ($this->request->get('fields') as $key => $value) {
            if (isset($this->fields[$key])) {
                $this->fields[$key]['value'] = $value;
            }
        }

        // Populate the array with the file data
        if ($this->request->file('fields') && ! empty($this->request->file('fields'))) {
            foreach ($this->request->file('fields') as $key => $value) {
                if (isset($this->fields[$key])) {
                    $this->fields[$key]['value'] = $value;
                }
            }
        }
    }

    /**
     * @throws ValidationException
     * @throws BindingResolutionException
     */
    public function validateFields(): void
    {
        // Create an array of key/values
        $data = [];
        foreach ($this->fields as $field) {
            $data[$field['name']] = $field['value'];
        }

        // Create an array of key/rules
        $rules = [];
        foreach ($this->fields as $field) {
            $fieldName = is_array($field['value']) ? $field['name'].'.*' : $field['name'];
            $rules[$fieldName] = implode('|', $field['rules']);
        }

        // Add the reCaptcha rule if enabled
        if (setting()->reCaptchaEnabled()) {
            $recaptchaKey = make_key($this->form->name);
            $recaptchaMinScore = config('form.recaptcha.min_score', 0.5);

            $data['captcha-response'] = $this->request->get('captcha-response');

            $rules['captcha-response'] = new Captcha(
                action: $recaptchaKey,
                minScore: $recaptchaMinScore,
            );
        }

        // Pass the data/rules to the validator and validate
        Validator::make($data, $rules)->validate();

        unset($data);
        unset($rules);
    }

    public function storeFiles(): void
    {
        // Loop through the fields
        foreach ($this->fields as $key => $field) {

            // if the field holds images or files then save them
            if (($field['type'] === FormItem::IMAGE || $field['type'] === FormItem::FILE)
                && ! empty($field['value'])
            ) {
                // Get the files
                foreach ($field['value'] as $fKey => $file) {
                    // Set the path and name
                    $tmpPath = config('document.tmp_path');
                    $fileName = $this->generateFileName($file);
                    $filePath = $tmpPath.$fileName;

                    // If the file saved, then replace the value with the file location
                    if ($file->storePubliclyAs(rtrim($tmpPath, '/'), $fileName)) {
                        $this->fields[$key]['value'][$fKey] = Storage::url($filePath);
                    }
                }
            }
        }
    }

    public function purgeOldFiles(): void
    {
        // Loop through the temp directory and get an array of all files
        $allFiles = Storage::files(config('document.tmp_path'));

        // Using the config calculate the date to purge files after
        $purgeFilesAfter = config('form.purge_files_after', 60); // days
        $nowMinusPurgeDays = strtotime("-{$purgeFilesAfter} days"); // The timestamp XX days ago

        // Only get the files where the last modified date is older than the purge date
        $oldFiles = array_filter($allFiles, function ($file) use ($nowMinusPurgeDays) {
            return Storage::lastModified($file) < $nowMinusPurgeDays;
        });

        // If there are no old files then bail
        if (empty($oldFiles)) {
            return;
        }

        // Otherwise, delete the old files
        foreach ($oldFiles as $file) {
            Storage::delete($file);
        }
    }

    public function getFields(): array
    {
        // Remove the empty fields
        foreach ($this->fields as $key => $field) {
            if (empty($field['value'])) {
                unset($this->fields[$key]);
            }
        }

        return $this->fields;
    }

    protected function generateFileName($file): string
    {
        $fileExt = $file->getClientOriginalExtension();
        $fileName = rtrim($file->getClientOriginalName(), $fileExt);
        $fileName = strtolower(Str::slug($fileName));

        return $fileName.'-'.time().'.'.$fileExt;
    }
}
