<?php

namespace Bongo\Framework\Helpers;

use Exception;
use GuzzleHttp\Client;
use GuzzleHttp\Exception\GuzzleException;
use Illuminate\Support\Facades\Log;

class CloudFlare
{
    protected $api;

    /**
     * @throws Exception
     */
    public function __construct()
    {
        if (empty(config('cloudflare.email'))) {
            throw new Exception('CloudFlare email credentials not set');
        }
        if (empty(config('cloudflare.api_key'))) {
            throw new Exception('CloudFlare api key not set');
        }
        if (empty(config('cloudflare.api_url'))) {
            throw new Exception('CloudFlare api url not set');
        }

        $this->setApiConnection();
    }

    protected function setApiConnection(): void
    {
        $this->api = new Client([
            'debug' => false,
            'verify' => false,
            'base_uri' => config('cloudflare.api_url'),
            'headers' => [
                'X-Auth-Email' => config('cloudflare.email'),
                'X-Auth-Key' => config('cloudflare.api_key'),
                'Content-Type' => 'application/json',
            ],
        ]);
    }

    public function hasValidCredentials(): bool
    {
        try {
            $response = $this->api->get('/client/v4/user');

            return $response->getStatusCode() == 200;

        } catch (GuzzleException $e) {
            Log::error($e->getMessage());

            return false;
        }
    }

    public function getAllZones(): ?array
    {
        try {
            $response = $this->api->get('/client/v4/zones');

            return json_decode($response->getBody()->getContents(), true);

        } catch (GuzzleException $e) {
            Log::error($e->getMessage());

            return null;
        }
    }

    public function getZoneId(): ?string
    {
        try {
            $response = $this->api->get('/client/v4/zones', [
                'name' => config('cloudflare.domain'),
            ]);
            $results = json_decode($response->getBody()->getContents(), true);

            if (isset($results['result'])
                && isset($results['result'][0])
                && isset($results['result'][0]['id'])
            ) {
                return $results['result'][0]['id'];
            }

            return null;

        } catch (GuzzleException $e) {
            Log::error($e->getMessage());

            return null;
        }
    }

    public function clearCache(string $zoneId): bool
    {
        try {
            $response = $this->api->request('DELETE', "/client/v4/zones/{$zoneId}/purge_cache", [
                'json' => [
                    'purge_everything' => true,
                ],
            ]);

            return $response->getStatusCode() == 200;

        } catch (GuzzleException $e) {
            Log::error($e->getMessage());

            return false;
        }
    }
}
