<?php

namespace Bongo\Framework\Traits;

use Illuminate\Support\Facades\Cache;
use Illuminate\Support\Str;

/**
 * Trait HasSeo.
 *
 * @category  Bespoke_Software
 * @author    Bespoke.ws Ltd <support@bespokeuk.com>
 * @copyright 2015-2020 Bespoke.ws Ltd, All Rights Reserved
 * @license   Proprietary and confidential
 * Unauthorized copying of this file, via any medium is strictly prohibited
 * @link      https://bespokeuk.com
 * @package   Bongo\Framework\Traits
 */
trait HasSeo
{
    /**
     * Boot.
     */
    public static function bootHasSeo()
    {
        static::saving(function ($model) {
            $model->setSlug($model);
            $model->setMetaTitle($model);
            $model->setMetaDescription($model);
            Cache::forget('page.' . $model->slug);
        });
    }

    /**
     * Set the default url slug.
     *
     * @param $model
     */
    private function setSlug($model): void
    {
        if (empty($model->slug)) {
            $model->slug = Str::slug(Str::limit($model->name, 75));
        } else {
            $model->slug = Str::slug(Str::limit($model->slug, 75));
        }
    }

    /**
     * Set default meta title if empty.
     *
     * @param $model
     */
    private function setMetaTitle($model): void
    {
        if (empty($model->meta_title)) {
            $model->meta_title = trim(Str::limit($model->name, 75));
        } else {
            $model->meta_title = trim(Str::limit($model->meta_title, 75));
        }
    }

    /**
     * Set default meta description if empty.
     *
     * @param $model
     */
    private function setMetaDescription($model): void
    {
        if (empty($model->meta_description)) {
            $model->meta_description = trim(Str::limit(trim(preg_replace('/\s\s+/', ' ', strip_tags($model->content))),
                150));
        } else {
            $model->meta_description = trim(Str::limit(trim(preg_replace('/\s\s+/', ' ',
                strip_tags($model->meta_description))), 150));
        }
    }
}
