# Bongo Framework - Architecture Documentation

## Overview

Bongo Framework is the foundational package for the Bongo Laravel monorepo system. It provides the core architecture that all other Bongo packages extend, including automatic service provider bootstrapping, base model classes with audit trails, reusable model traits, custom Eloquent casts, PHP 8.1+ enums, Schema.org structured data generation, and comprehensive helper classes.

**Version**: Compatible with PHP 8.2+ and Laravel 10+

**Dependencies**:
- `illuminate/contracts: ^10.0` - Laravel framework contracts
- `spatie/schema-org: ^3.23` - Schema.org structured data generation
- `maatwebsite/excel: ^3.1` - Excel import/export functionality

## Package Structure

```
framework/
├── src/
│   ├── Casts/                  # Custom Eloquent attribute casts (12 classes)
│   ├── Config/                 # Package configuration files
│   │   ├── cloudflare.php
│   │   ├── developer.php
│   │   ├── schema.php
│   │   └── settings.php
│   ├── Console/                # Artisan commands
│   ├── Enums/                  # PHP 8.1+ backed enums (5 enums)
│   ├── Events/                 # Event classes
│   ├── Exceptions/             # Custom exception classes
│   ├── Helpers/                # Static helper classes (13 classes)
│   ├── Http/
│   │   ├── Controllers/        # Abstract base controllers
│   │   ├── Middleware/         # HTTP middleware (11 classes)
│   │   └── ViewComposers/      # View composers
│   ├── Interfaces/             # Contracts and interfaces
│   ├── Migrations/             # Database migrations (4 tables)
│   ├── Models/                 # AbstractModel base class
│   ├── Providers/              # Service providers
│   │   ├── AbstractServiceProvider.php  ★ CRITICAL
│   │   ├── AuthServiceProvider.php
│   │   ├── EventServiceProvider.php
│   │   └── RouteServiceProvider.php
│   ├── Schema/                 # Schema.org graph generation
│   │   ├── Concerns/           # Reusable schema traits (20 concerns)
│   │   ├── MetaSchema.php      # Entry point for SEO graphs
│   │   ├── PageGraph.php
│   │   ├── PostGraph.php
│   │   └── ProjectGraph.php
│   ├── Traits/                 # Reusable model traits (37 traits)
│   │   ├── Address/            # Address-related traits (3 traits)
│   │   ├── Audit/              # Audit trail traits (7 traits)
│   │   └── Contact/            # Contact information traits (14 traits)
│   ├── Views/                  # Blade templates
│   │   ├── auth/               # Authentication layouts
│   │   ├── backend/            # Admin panel layouts
│   │   ├── frontend/           # Public-facing layouts
│   │   └── mail/               # Email layouts
│   ├── FrameworkServiceProvider.php  ★ Main entry point
│   └── helpers.php             # Global helper functions
├── tests/
│   ├── Feature/
│   ├── Unit/
│   └── TestCase.php
├── composer.json
├── phpunit.xml
├── .editorconfig
├── .gitignore
├── LICENSE
└── README.md
```

## Core Architecture

### 1. Service Provider System

#### AbstractServiceProvider (The Foundation)

**Location**: `src/Providers/AbstractServiceProvider.php`

This is the **most critical class** in the entire Bongo ecosystem. Every Bongo package MUST extend this class.

**Automatic Bootstrapping Flow**:

```
AbstractServiceProvider::boot()
    │
    ├─→ setDir()                 # Set provider directory via reflection
    ├─→ bootConfig()             # Auto-load config from Config/{module}.php
    ├─→ bootCommands()           # Register commands from $commands array
    ├─→ bootComposers()          # Register view composers from $composers array
    ├─→ bootListeners()          # Register event listeners from $listeners array
    ├─→ bootMiddleware()         # Register middleware from $middlewares array
    ├─→ bootMigrations()         # Auto-load migrations from Migrations/
    ├─→ bootRoutes()             # Auto-register routes from Routes/
    │   ├─→ bootApiRoutes()      # Routes/api.php → /api prefix, auth:sanctum
    │   ├─→ bootAdminRoutes()    # Routes/backend.php → /admin prefix, auth+employee
    │   ├─→ bootFrontendRoutes() # Routes/frontend.php → web middleware, frontend.* names
    │   ├─→ bootWebRoutes()      # Routes/web.php → web middleware
    │   └─→ bootCustomRoutes()   # Routes/custom.php → no middleware (webhooks)
    ├─→ bootSubscribers()        # Register event subscribers from $subscribers array
    ├─→ bootTranslations()       # Auto-load translations from Translations/
    └─→ bootViews()              # Auto-load views from Views/ with {module} namespace
```

**Required Properties**:

```php
abstract class AbstractServiceProvider extends ServiceProvider
{
    protected string $module;      // REQUIRED: Must match config filename and view namespace
    protected array $commands = [];     // Optional: Artisan commands to register
    protected array $composers = [];    // Optional: View composers [Class => [views]]
    protected array $listeners = [];    // Optional: Event listeners [Event => [Listeners]]
    protected array $subscribers = [];  // Optional: Event subscribers
    protected array $middlewares = [];  // Optional: Middleware [alias => Class]
}
```

**Route Registration Pattern**:

| File | URL Prefix | Route Name Prefix | Middleware | Use Case |
|------|-----------|-------------------|------------|----------|
| `api.php` | `/api` | `api.*` | `api, auth:sanctum, noIndex` | Protected API endpoints |
| `backend.php` | `/admin` | `backend.*` | `web, auth, employee, noIndex` | Admin panel routes |
| `frontend.php` | None | `frontend.*` | `web` | Public-facing routes |
| `web.php` | None | None | `web` | Standard web routes |
| `custom.php` | None | None | None | Webhooks, payment callbacks (no session) |

**Config Registration**:
- File: `src/Config/{module}.php` (must match `$module` property)
- Auto-merged: `config('{module}.key')`
- Publishable: `php artisan vendor:publish --tag=config`

**View Registration**:
- Directory: `src/Views/`
- Namespace: `{module}::`
- Usage: `view('{module}::backend.index')`
- Publishable: `php artisan vendor:publish --tag=frontend`

#### FrameworkServiceProvider (Main Entry Point)

**Location**: `src/FrameworkServiceProvider.php`

```
FrameworkServiceProvider::boot()
    │
    ├─→ parent::boot()                 # Run AbstractServiceProvider boot
    ├─→ checkEnv()                     # Ensure .env exists
    ├─→ checkStoragePath()             # Create storage directories
    ├─→ Schema::defaultStringLength(191) # MySQL compatibility
    ├─→ Request::setTrustedProxies()   # Cloudflare proxy support
    ├─→ Model::preventLazyLoading()    # N+1 detection (non-prod)
    ├─→ AliasLoader                    # Register global aliases
    ├─→ bootFlashResponseMacro()       # Add success/error/warning methods
    ├─→ bootConfig()                   # Merge framework configs
    ├─→ loadCoreProviders()            # Load essential packages
    │   ├─→ RouteServiceProvider
    │   ├─→ ReferrerServiceProvider
    │   ├─→ AssetServiceProvider
    │   ├─→ BladeServiceProvider
    │   ├─→ EnumServiceProvider
    │   └─→ PackageServiceProvider
    │
    ├─→ IF install_complete = false
    │   └─→ loadInstaller()            # Installation flow
    │
    └─→ ELSE
        └─→ loadPackages()             # Load all Bongo packages
            ├─→ Standard packages (always loaded)
            │   ├─→ BuilderServiceProvider
            │   ├─→ CaptchaServiceProvider
            │   ├─→ DashboardServiceProvider
            │   ├─→ UserServiceProvider
            │   └─→ ... (11 total)
            │
            ├─→ Conditional packages (via package manager)
            │   ├─→ EstimateServiceProvider (if enabled)
            │   ├─→ GalleryServiceProvider (if enabled)
            │   ├─→ PostServiceProvider (if enabled)
            │   └─→ ... (5 total)
            │
            └─→ app()->booted() callback
                ├─→ RedirectServiceProvider (second to last)
                └─→ PageServiceProvider (MUST be last, has /{slug} route)
```

### 2. Model System

#### AbstractModel (Base Model Class)

**Location**: `src/Models/AbstractModel.php`

```
Illuminate\Database\Eloquent\Model
    │
    └─→ AbstractModel
        ├─→ use HasCreatedAt     # Adds: created_at timestamp
        ├─→ use HasCreatedBy     # Adds: created_by (User relationship)
        ├─→ use HasUpdatedAt     # Adds: updated_at timestamp
        ├─→ use HasUpdatedBy     # Adds: updated_by (User relationship)
        ├─→ use HasDeletedAt     # Adds: deleted_at (soft delete)
        ├─→ use HasDeletedBy     # Adds: deleted_by (User relationship)
        └─→ use HasDiff          # Tracks: changes between saves
```

**Audit Relationships**:

```php
$model->createdBy; // BelongsTo User
$model->updatedBy; // BelongsTo User
$model->deletedBy; // BelongsTo User (if soft deleted)
```

**Methods**:

```php
public function attributeExists(string $key): bool
```

#### Model Traits (37 Available)

**Audit Traits** (`src/Traits/Audit/`):

| Trait | Adds | Relationship |
|-------|------|--------------|
| `HasCreatedAt` | `created_at` column | - |
| `HasCreatedBy` | `created_by` column | `createdBy()` → User |
| `HasUpdatedAt` | `updated_at` column | - |
| `HasUpdatedBy` | `updated_by` column | `updatedBy()` → User |
| `HasDeletedAt` | `deleted_at` column | - |
| `HasDeletedBy` | `deleted_by` column | `deletedBy()` → User |
| `HasDiff` | Change tracking | `getDiff()` method |

**Contact Traits** (`src/Traits/Contact/`) - 14 traits:

| Trait | Columns | Methods |
|-------|---------|---------|
| `HasEmail` | `email` | `hasEmail(): bool` |
| `HasPhone` | `phone` | `hasPhone(): bool` |
| `HasMobile` | `mobile` | `hasMobile(): bool` |
| `HasWebsite` | `website` | `hasWebsite(): bool` |
| `HasFirstName` | `first_name` | - |
| `HasLastName` | `last_name` | - |
| `HasFullName` | - | `getFullName(): string` |
| `HasSocial` | `facebook_url`, `twitter_url`, etc. | Various `has*()` methods |
| `HasMarketingEmails` | `marketing_emails` (bool) | - |
| `HasMarketingSms` | `marketing_sms` (bool) | - |
| `HasNotificationEmails` | `notification_emails` (bool) | - |
| `HasNotificationSms` | `notification_sms` (bool) | - |
| `HasAcceptedTerms` | `accepted_terms_at` | `hasAcceptedTerms(): bool` |

**Address Traits** (`src/Traits/Address/`) - 3 traits:

| Trait | Column | Type |
|-------|--------|------|
| `HasPostcode` | `postcode` | string |
| `HasLatitude` | `latitude` | decimal |
| `HasLongitude` | `longitude` | decimal |

**Other Traits** (`src/Traits/`):

| Trait | Purpose | Key Methods |
|-------|---------|-------------|
| `HasSeo` | Auto-generates SEO fields on save | `setSlug()`, `setMetaTitle()`, `setMetaDescription()` |
| `HasRecursive` | Parent/child hierarchies | `parent()`, `children()`, `nestedChildren()`, `siblings()` |
| `HasStatus` | Status field with StatusEnum | - |
| `HasVisible` | Visibility toggle | - |
| `HasDate` | Date field | - |
| `HasKey` | Unique key field | - |
| `HasHeaderClass` | CSS header class | - |

**HasRecursive Trait Details**:

```
Model (uses HasRecursive)
    │
    ├─→ Relationships
    │   ├─→ parent()           # BelongsTo self (parent_id)
    │   ├─→ nestedParents()    # BelongsTo with recursive loading
    │   ├─→ children()         # HasMany self, ordered by sort_order
    │   ├─→ nestedChildren()   # HasMany with recursive loading
    │   └─→ siblings()         # Collection (same parent_id)
    │
    ├─→ Check Methods
    │   ├─→ hasParent(): bool
    │   ├─→ hasNestedParents(): bool
    │   ├─→ hasChildren(): bool
    │   ├─→ hasNestedChildren(): bool
    │   ├─→ hasSiblings(): bool
    │   ├─→ isParent(): bool   # parent_id is null
    │   └─→ isChild(): bool    # parent_id is not null
    │
    └─→ Collection Methods
        ├─→ getAncestors(): Collection       # All parents recursively
        ├─→ getAncestorsAsArray(): array     # Nested array structure
        ├─→ getDescendents(): Collection     # All children recursively
        └─→ getDescendentsAsArray(): array   # Nested array structure
```

### 3. Custom Casts System

**Location**: `src/Casts/`

All casts implement `Illuminate\Contracts\Database\Eloquent\CastsAttributes`:

| Cast Class | Purpose | get() | set() |
|------------|---------|-------|-------|
| `Encoded` | HTML entity encoding | Decode entities | Encode entities |
| `PlainText` | Strip HTML tags | Return as-is | Strip tags |
| `PlainWords` | Extract plain words | Return as-is | Remove non-word chars |
| `PlainNumber` | Extract numbers only | Return as-is | Remove non-numeric |
| `PlainLowerText` | Lowercase plain text | Return as-is | Lowercase + strip tags |
| `PlainUpperText` | Uppercase plain text | Return as-is | Uppercase + strip tags |
| `Date` | Date formatting | Format date | Parse date |
| `DateTime` | DateTime formatting | Format datetime | Parse datetime |
| `Time` | Time formatting | Format time | Parse time |
| `Pence` | Monetary (pence/pounds) | Divide by 100 (£) | Multiply by 100 (p) |
| `Domain` | Domain normalization | Return as-is | Normalize domain |
| `Checkbox` | Boolean checkbox | Cast to bool | Cast to int |

**Usage Example**:

```php
use Bongo\Framework\Casts\Encoded;
use Bongo\Framework\Casts\Pence;

class Product extends AbstractModel
{
    protected $casts = [
        'description' => Encoded::class,  // Stores: &lt;p&gt;Hello&lt;/p&gt;
                                          // Returns: <p>Hello</p>
        'price' => Pence::class,          // Stores: 1999 (pence)
                                          // Returns: 19.99 (pounds)
    ];
}
```

### 4. Enum System

**Location**: `src/Enums/`

All enums are PHP 8.1+ backed enums implementing framework interfaces:

```
enum MyEnum: int
    │
    ├─→ implements ArrayInterface    # Provides toArray() for dropdowns
    ├─→ implements DefaultInterface  # Provides getDefault() for defaults
    └─→ use WithArray               # Trait implementing toArray()
```

**Available Enums**:

```php
// BooleanEnum (int backed)
enum BooleanEnum: int
{
    case YES = 1;
    case NO = 0;
}

// StatusEnum (int backed)
enum StatusEnum: int
{
    case ACTIVE = 1;
    case DISABLED = 0;
}
```

**Enum Interfaces**:

```php
interface ArrayInterface
{
    public static function toArray(): array; // For select dropdowns
}

interface DefaultInterface
{
    public static function getDefault(): int|string; // Default value
}
```

**Usage Example**:

```php
use Bongo\Framework\Enums\BooleanEnum;
use Bongo\Framework\Enums\StatusEnum;

class Page extends AbstractModel
{
    protected $casts = [
        'is_featured' => BooleanEnum::class,
        'status' => StatusEnum::class,
    ];

    protected $attributes = [
        'is_featured' => BooleanEnum::NO->value,
        'status' => StatusEnum::ACTIVE->value,
    ];
}

// Access in code
$page->is_featured === BooleanEnum::YES; // true/false
$page->status->value; // 1 or 0

// Use in dropdowns
BooleanEnum::toArray(); // ['YES' => 1, 'NO' => 0]
StatusEnum::toArray(); // ['ACTIVE' => 1, 'DISABLED' => 0]
```

### 5. Controller System

#### Controller Hierarchy

```
Illuminate\Routing\Controller
    │
    └─→ AbstractController
        ├─→ use AuthorizesRequests
        ├─→ use ValidatesRequests
        │
        ├─→ AbstractApiController
        │   └─→ Standardized JSON API responses
        │
        └─→ AbstractDatatableController
            └─→ DataTables JSON endpoint pattern
```

#### AbstractDatatableController Flow

**Location**: `src/Http/Controllers/AbstractDatatableController.php`

```
index() method execution flow:
    │
    ├─→ setup()
    │   ├─→ Get request and input
    │   ├─→ Call getBaseQuery() ★ OVERRIDE THIS
    │   └─→ Set pagination params (length, start)
    │
    ├─→ applyFilters() ★ OVERRIDE THIS
    │   ├─→ applyStatusFilter()   # If ?status= provided
    │   └─→ applyTypeFilter()     # If ?type= provided
    │
    ├─→ applySearchQuery()
    │   └─→ Search by 'name' column (multi-word support)
    │
    ├─→ applyOrderBy()
    │   └─→ Order by ?order_by= and ?order_direction=
    │
    ├─→ runQuery()
    │   ├─→ setTotalResults()    # Count before pagination
    │   ├─→ applyPagination()    # LIMIT and OFFSET
    │   └─→ setResults()         # Execute query
    │
    └─→ createResponse()
        └─→ Return JSON: { results: [...], totalResults: N }
```

**Abstract Methods (Must Override)**:

```php
abstract protected function getBaseQuery(): Builder;
```

**Overridable Methods**:

```php
protected function applyFilters(): void;        # Add custom filters
protected function applySearchQuery(): void;    # Customize search logic
protected function applyOrderBy(): void;        # Customize ordering
```

**Query Parameters**:

| Parameter | Purpose | Default |
|-----------|---------|---------|
| `length` | Results per page | 10 |
| `start` | Offset for pagination | 0 |
| `order_by` | Column to order by | id |
| `order_direction` | Order direction | desc |
| `search` | Search query | - |
| `status` | Filter by status | - |
| `type` | Filter by type | - |

### 6. Schema.org System

#### Architecture

**Location**: `src/Schema/`

```
MetaSchema (Entry point)
    │
    ├─→ graphForPage(Model $entity): string
    │   └─→ PageGraph
    │       ├─→ setReferenceIds()
    │       ├─→ setEntity($entity)
    │       ├─→ addWebPage()               # Specific to pages
    │       ├─→ addWebSite()               # Generic
    │       ├─→ addLocalBusiness()         # Generic
    │       ├─→ addSocialLinksToLocalBusiness()
    │       ├─→ addReviewsToLocalBusiness()
    │       ├─→ addContactPoint()          # Generic
    │       ├─→ addPostalAddress()         # Generic
    │       └─→ toScript()                 # Render <script> tag
    │
    ├─→ graphForPost(Model $entity): string
    │   └─→ PostGraph
    │       ├─→ setReferenceIds()
    │       ├─→ setEntity($entity)
    │       ├─→ addBlogPost()              # Specific to posts
    │       └─→ ... (same generic methods)
    │
    └─→ graphForProject(Model $entity): string
        └─→ ProjectGraph
            ├─→ setReferenceIds()
            ├─→ setEntity($entity)
            ├─→ addArticle()               # Specific to projects
            └─→ ... (same generic methods)
```

**Schema Concerns** (`src/Schema/Concerns/`) - 20 traits:

| Concern | Adds Schema Type | Methods |
|---------|------------------|---------|
| `HandlesWebPage` | WebPage | `addWebPage()` |
| `HandlesBlogPost` | BlogPosting | `addBlogPost()` |
| `HandlesArticle` | Article | `addArticle()` |
| `HandlesWebSite` | WebSite | `addWebSite()` |
| `HandlesLocalBusiness` | LocalBusiness | `addLocalBusiness()` |
| `HandlesOrganization` | Organization | `addOrganization()` |
| `HandlesSocialLinks` | sameAs property | `addSocialLinksTo*()` |
| `HandlesReviews` | AggregateRating | `addReviewsTo*()` |
| `HandlesContactPoint` | ContactPoint | `addContactPoint()` |
| `HandlesPostalAddress` | PostalAddress | `addPostalAddress()` |
| `HandlesGeoCoordinates` | GeoCoordinates | `addGeoCoordinates()` |
| `HandlesPlace` | Place | `addPlace()` |
| `HandlesAuthor` | Person (author) | `addAuthor()` |
| `HandlesFounder` | Person (founder) | `addFounder()` |
| `HandlesEntity` | Various | `setEntity()` |
| `HandlesReferenceIds` | @id generation | `setReferenceIds()` |
| `HandlesOutput` | Script rendering | `toScript()` |

**Usage**:

```php
// In controller
use Bongo\Framework\Helpers\SEO;

$schema = SEO::getMetaSchemaForPage($page);

// In blade template
{!! SEO::getMetaSchemaForPage($page) !!}
```

### 7. Helper System

**Location**: `src/Helpers/`

All helpers use static methods:

| Helper Class | Key Methods | Purpose |
|--------------|-------------|---------|
| `SEO` | `getMetaSchemaFor*(Model)` | Generate Schema.org graphs |
| `Str` | `camelWords()`, `key()`, `id()` | String manipulation |
| `Route` | `exists()`, `is()` | Route utilities |
| `URL` | Various URL helpers | URL manipulation |
| `File` | File handling methods | File operations |
| `CloudFlare` | API integration methods | Cloudflare API |
| `Cookie` | `enabled()` | Cookie consent checking |
| `Password` | `generate()` | Secure password generation |
| `Html` | HTML manipulation | HTML utilities |
| `ShortCode` | `process()` | Custom shortcode processing |
| `Tax` | Tax calculation | Tax utilities |
| `Console` | `print()` | CLI output formatting |
| `Log` | `exception()` | Enhanced exception logging |

**Global Helper Functions** (`src/helpers.php`):

```php
// User helpers
function user(): ?Authenticatable;

// Route helpers
function route_exists(string $name): bool;
function route_is(string $name): bool;

// String helpers
function camel_words(?string $input, bool $plural = false): ?string;
function make_key(string $input): string;
function make_id(string $input, $append = ''): string;

// Security helpers
function generate_password(int $noOfChars = 32): bool|string;

// Logging helpers
function log_exception($exception, int $code = 500): void;

// Console helpers
function console_print(string $message, string $type = 'info'): void;

// Cookie helpers
function cookie_enabled($name): bool;

// Text processing
function plain_text($input): string;
function encode_uri_component($str): string;
```

### 8. Middleware System

**Location**: `src/Http/Middleware/`

| Middleware | Purpose | Usage |
|------------|---------|-------|
| `MinifyHtml` | Minify HTML output | Enabled via config |
| `NoIndex` | Add noindex meta tag | Auto-applied to admin/API |
| `Authenticate` | Laravel auth | Standard |
| `EncryptCookies` | Cookie encryption | Standard |
| `VerifyCsrfToken` | CSRF protection | Standard |
| `TrimStrings` | Trim input strings | Standard |
| `TrustProxies` | Cloudflare proxies | Standard |
| `TrustHosts` | Host validation | Standard |
| `ValidateSignature` | Signed URL validation | Standard |
| `PreventRequestsDuringMaintenance` | Maintenance mode | Standard |
| `RedirectIfAuthenticated` | Guest only routes | Standard |

## Data Flow Diagrams

### Package Boot Sequence

```
Application Bootstrap
    │
    ├─→ Load FrameworkServiceProvider (registered in config/app.php)
    │
    ├─→ FrameworkServiceProvider::register()
    │   └─→ Check $module property exists
    │
    ├─→ FrameworkServiceProvider::boot()
    │   │
    │   ├─→ AbstractServiceProvider::boot()
    │   │   ├─→ Register config, routes, views, migrations
    │   │   └─→ Register middleware, commands, composers
    │   │
    │   ├─→ Environment checks (create .env, storage dirs)
    │   ├─→ Database configuration (string length, proxies)
    │   ├─→ Load global aliases (BooleanEnum, StatusEnum, SEO)
    │   ├─→ Load core providers (Asset, Blade, Enum, etc.)
    │   │
    │   ├─→ IF install_complete = false
    │   │   └─→ Load InstallServiceProvider only
    │   │
    │   └─→ ELSE
    │       ├─→ Load standard packages (Builder, Captcha, User, etc.)
    │       ├─→ Load conditional packages (if enabled)
    │       └─→ app()->booted()
    │           ├─→ Load RedirectServiceProvider
    │           └─→ Load PageServiceProvider (LAST - has catch-all route)
    │
    └─→ Application Ready
```

### Model Save Lifecycle (with HasSeo trait)

```
$model->save()
    │
    ├─→ Eloquent: firing 'saving' event
    │
    ├─→ HasSeo::bootHasSeo() static listener
    │   ├─→ setSlug($model)
    │   │   └─→ Generate slug from name (if empty)
    │   │
    │   ├─→ setMetaTitle($model)
    │   │   └─→ Set meta_title from name (75 char limit)
    │   │
    │   ├─→ setMetaDescription($model)
    │   │   └─→ Extract from content (150 char limit)
    │   │
    │   └─→ Cache::forget('page.'.$model->slug)
    │
    ├─→ HasUpdatedBy::bootHasUpdatedBy() (from AbstractModel)
    │   └─→ Set updated_by to current user ID
    │
    ├─→ Save to database
    │
    └─→ Return model
```

### DataTable Request Flow

```
HTTP GET /admin/mypackage/datatable?search=test&status=1&length=25&start=0
    │
    ├─→ MyDatatableController::index()
    │   │
    │   ├─→ setup()
    │   │   ├─→ Get request and input
    │   │   ├─→ $query = getBaseQuery()    # MyModel::query()
    │   │   └─→ $pagination = [25, 0]
    │   │
    │   ├─→ applyFilters()
    │   │   ├─→ applyStatusFilter()        # WHERE status = 1
    │   │   └─→ [custom filters]           # WHERE category_id = X
    │   │
    │   ├─→ applySearchQuery()
    │   │   └─→ WHERE name LIKE '%test%'
    │   │
    │   ├─→ applyOrderBy()
    │   │   └─→ ORDER BY id DESC
    │   │
    │   ├─→ runQuery()
    │   │   ├─→ $totalResults = $query->count()  # COUNT(*) before pagination
    │   │   ├─→ $query->limit(25)->offset(0)
    │   │   └─→ $results = $query->get()
    │   │
    │   └─→ createResponse()
    │       └─→ return json([
    │               'results' => [...],
    │               'totalResults' => 150
    │           ])
    │
    └─→ JSON Response
        {
            "results": [ { id: 1, name: "Test" }, ... ],
            "totalResults": 150
        }
```

### Route Registration Flow

```
AbstractServiceProvider::bootRoutes()
    │
    ├─→ bootApiRoutes('src/Routes/')
    │   └─→ IF src/Routes/api.php exists
    │       └─→ Route::prefix(config('settings.api_prefix'))  # /api
    │           →middleware(['api', 'auth:sanctum', 'noIndex'])
    │           →as('api.')
    │           →group(load api.php)
    │
    ├─→ bootAdminRoutes('src/Routes/')
    │   └─→ IF src/Routes/backend.php exists
    │       └─→ Route::prefix(config('settings.backend_prefix'))  # /admin
    │           →middleware(['web', 'auth', 'employee', 'noIndex'])
    │           →as('backend.')
    │           →group(load backend.php)
    │
    ├─→ bootFrontendRoutes('src/Routes/')
    │   └─→ IF src/Routes/frontend.php exists
    │       └─→ Route::middleware(['web'])
    │           →as('frontend.')
    │           →group(load frontend.php)
    │
    ├─→ bootWebRoutes('src/Routes/')
    │   └─→ IF src/Routes/web.php exists
    │       └─→ Route::middleware(['web'])
    │           →group(load web.php)
    │
    └─→ bootCustomRoutes('src/Routes/')
        └─→ IF src/Routes/custom.php exists
            └─→ Load routes with NO middleware (webhooks, callbacks)
```

## Extension Points

### 1. Creating a New Package

**Required Structure**:

```
my-package/
├── src/
│   ├── Config/
│   │   └── mypackage.php         # Required: matches $module
│   ├── Http/
│   │   └── Controllers/
│   ├── Models/
│   ├── Routes/
│   │   ├── backend.php           # Optional: admin routes
│   │   ├── frontend.php          # Optional: public routes
│   │   └── api.php               # Optional: API routes
│   ├── Views/
│   │   └── mypackage/            # Matches $module namespace
│   ├── Migrations/               # Optional: migrations
│   └── MyPackageServiceProvider.php  # Required: extends AbstractServiceProvider
├── composer.json
└── README.md
```

**Service Provider Template**:

```php
<?php

declare(strict_types=1);

namespace Bongo\MyPackage;

use Bongo\Framework\Providers\AbstractServiceProvider;

class MyPackageServiceProvider extends AbstractServiceProvider
{
    // Required: must match config filename and view namespace
    protected string $module = 'mypackage';

    // Optional: register middleware
    protected array $middlewares = [
        'aliasName' => MiddlewareClass::class,
    ];

    // Optional: register commands
    protected array $commands = [
        MyCommand::class,
    ];

    // Optional: register event listeners
    protected array $listeners = [
        MyEvent::class => [
            MyListener::class,
        ],
    ];

    // Optional: register event subscribers
    protected array $subscribers = [
        MySubscriber::class,
    ];

    // Optional: register view composers
    protected array $composers = [
        MyComposer::class => [
            'mypackage::view.name',
        ],
    ];

    // Optional: additional boot logic
    public function boot(): void
    {
        parent::boot(); // REQUIRED: call parent

        // Custom boot logic here
    }
}
```

### 2. Adding New Model Traits

**Template**:

```php
<?php

declare(strict_types=1);

namespace Bongo\MyPackage\Traits;

trait MyTrait
{
    // Optional: Add fillable attributes
    public function initializeMyTrait(): void
    {
        $this->mergeFillable(['my_field']);
    }

    // Optional: Boot logic (runs when model boots)
    public static function bootMyTrait()
    {
        static::saving(function ($model) {
            // Logic before save
        });
    }

    // Trait methods
    public function myMethod(): bool
    {
        return !empty($this->my_field);
    }
}
```

### 3. Adding New Custom Casts

**Template**:

```php
<?php

declare(strict_types=1);

namespace Bongo\MyPackage\Casts;

use Illuminate\Contracts\Database\Eloquent\CastsAttributes;

class MyCast implements CastsAttributes
{
    public function get($model, string $key, $value, array $attributes): mixed
    {
        // Transform value when retrieving from database
        return $value ? json_decode($value, true) : [];
    }

    public function set($model, string $key, $value, array $attributes): mixed
    {
        // Transform value when storing to database
        return $value ? json_encode($value) : null;
    }
}
```

### 4. Adding New Enums

**Template**:

```php
<?php

declare(strict_types=1);

namespace Bongo\MyPackage\Enums;

use Bongo\Framework\Enums\ArrayInterface;
use Bongo\Framework\Enums\DefaultInterface;
use Bongo\Framework\Enums\WithArray;

enum MyEnum: string implements ArrayInterface, DefaultInterface
{
    use WithArray;

    case OPTION_ONE = 'option_one';
    case OPTION_TWO = 'option_two';
    case OPTION_THREE = 'option_three';

    public static function getDefault(): string
    {
        return self::OPTION_ONE->value;
    }

    public function label(): string
    {
        return match($this) {
            self::OPTION_ONE => 'Option One',
            self::OPTION_TWO => 'Option Two',
            self::OPTION_THREE => 'Option Three',
        };
    }
}
```

### 5. Adding Schema.org Support

**Create Graph Class**:

```php
<?php

declare(strict_types=1);

namespace Bongo\MyPackage\Schema;

use Bongo\Framework\Schema\Concerns\HandlesEntity;
use Bongo\Framework\Schema\Concerns\HandlesOutput;
use Bongo\Framework\Schema\Concerns\HandlesReferenceIds;
use Bongo\Framework\Schema\Concerns\HandlesWebSite;
use Spatie\SchemaOrg\Graph;

class MyGraph extends Graph
{
    use HandlesEntity;
    use HandlesOutput;
    use HandlesReferenceIds;
    use HandlesWebSite;

    public function addMySchema(): self
    {
        $this->add($this->mySchema());
        return $this;
    }

    private function mySchema()
    {
        return \Spatie\SchemaOrg\Schema::thing()
            ->name($this->entity->name)
            ->description($this->entity->description);
    }
}
```

**Add to Helper**:

```php
// In Bongo\Framework\Helpers\SEO
public static function getMetaSchemaForMyEntity(Model $entity): string
{
    return (new MetaSchema())->graphForMyEntity($entity);
}

// In Bongo\Framework\Schema\MetaSchema
public function graphForMyEntity(Model $entity): string
{
    $graph = new MyGraph();
    $graph->setReferenceIds()
        ->setEntity($entity)
        ->addMySchema()
        ->addWebSite();

    return $graph->toScript();
}
```

## Configuration Reference

### settings.php

```php
return [
    // API Configuration
    'api_prefix' => 'api',                    # URL prefix for API routes
    'api_type' => 'application/json',         # API content type

    // Admin Configuration
    'backend_prefix' => 'admin',              # URL prefix for admin routes

    // Performance
    'cache_default' => 300,                   # Default cache TTL (seconds)
    'memory_limit' => '2048M',                # PHP memory limit

    // Features
    'password_reset_enabled' => true,         # Enable password reset
    'install_complete' => env('INSTALL_COMPLETE', false),  # Installation flag
    'minify_html_enabled' => env('MINIFY_HTML_ENABLED', false),  # HTML minification

    // Email
    'mail_from_name' => env('MAIL_FROM_NAME'),
    'mail_from_address' => env('MAIL_FROM_ADDRESS'),
];
```

### schema.php

Contains Schema.org configuration for:
- Organization details (name, logo, description)
- Contact information (phone, email, address)
- Social media links
- Business hours
- Review aggregates

### cloudflare.php

Contains Cloudflare API configuration:
- API key
- Zone ID
- Cache purging settings

### developer.php

Developer-specific settings:
- Debug flags
- Development tools
- Testing configurations

## Testing Architecture

**Test Base Class**: `tests/TestCase.php`

Extends Orchestra Testbench for package testing.

**Test Structure**:
```
tests/
├── Feature/          # Integration tests
│   └── ...
├── Unit/             # Unit tests
│   └── ...
└── TestCase.php      # Base test class
```

**Run Tests**:
```bash
vendor/bin/phpunit
```

## Database Migrations

Framework includes 4 base migrations:

| Migration | Table | Purpose |
|-----------|-------|---------|
| `2020_01_01_000001_create_failed_jobs_table.php` | `failed_jobs` | Queue failures |
| `2020_01_01_000002_create_jobs_table.php` | `jobs` | Queue jobs |
| `2020_01_01_000003_create_cache_table.php` | `cache` | Cache storage |
| `2020_01_01_000004_create_sessions_table.php` | `sessions` | Session storage |

## Performance Considerations

1. **Lazy Loading Prevention**: Enabled in non-production environments to catch N+1 queries
2. **Cache Strategy**: Default 5-minute cache for common queries
3. **String Length**: MySQL compatibility with 191 char limit
4. **HTML Minification**: Optional via config for production
5. **Memory Limit**: Default 2048M for large operations

## Security Features

1. **CSRF Protection**: Via VerifyCsrfToken middleware
2. **Cookie Encryption**: Via EncryptCookies middleware
3. **Trusted Proxies**: Cloudflare proxy support
4. **Input Trimming**: Auto-trim via TrimStrings middleware
5. **Audit Trail**: Automatic created_by/updated_by tracking
6. **Route Protection**: Backend routes require auth + employee middleware
7. **API Protection**: API routes require auth:sanctum middleware

## Troubleshooting

### Common Issues

**1. Routes not registering**
- Check `$module` property matches config filename
- Ensure route files are in `src/Routes/`
- Verify `parent::boot()` is called in service provider

**2. Views not found**
- Verify views are in `src/Views/{module}/`
- Check namespace matches `$module` property
- Use `{module}::path.to.view` syntax

**3. Config not loading**
- Config file must be named `{module}.php`
- Must be in `src/Config/` directory
- Check `$module` property is set correctly

**4. Migrations not running**
- Migrations must be in `src/Migrations/`
- Check file naming follows Laravel convention
- Verify `parent::boot()` is called

**5. Middleware not working**
- Check middleware is in `$middlewares` array
- Verify array key (alias) is used in route definitions
- Ensure class exists and is imported
