# Bongo Framework

**The foundation package for the Bongo Laravel monorepo ecosystem.**

Bongo Framework provides the core architecture that all other Bongo packages extend, including automatic service provider bootstrapping, base model classes with audit trails, reusable model traits, custom Eloquent casts, PHP 8.1+ enums, Schema.org structured data generation, and comprehensive helper classes.

[![PHP Version](https://img.shields.io/badge/PHP-8.2%2B-blue)](https://www.php.net/)
[![Laravel Version](https://img.shields.io/badge/Laravel-10%2B-red)](https://laravel.com/)
[![License](https://img.shields.io/badge/License-MIT-green)](LICENSE)

## Features

- 🚀 **AbstractServiceProvider** - Automatic bootstrapping for routes, config, views, migrations, and more
- 📦 **AbstractModel** - Base model with automatic audit trails (created_by, updated_by, deleted_by)
- 🔧 **37 Reusable Traits** - Contact info, addresses, SEO, hierarchical relationships, and more
- 🎨 **12 Custom Casts** - HTML encoding, pence/pounds conversion, plain text extraction
- 🔢 **PHP 8.1+ Enums** - Type-safe enums for status, boolean, and other fields
- 🌐 **Schema.org Generation** - Automatic structured data for SEO
- 🛠️ **13 Helper Classes** - String manipulation, routing, cookies, passwords, and more
- 🎯 **AbstractDatatableController** - Rapid DataTables endpoint development
- 🔒 **Security Built-in** - CSRF protection, cookie encryption, trusted proxies

## Requirements

- PHP 8.2 or higher
- Laravel 10.0 or higher
- Composer

## Installation

```bash
composer require bongo/framework
```

For Laravel 5/6 projects, add the service provider to `config/app.php`:

```php
'providers' => [
    // ...
    Bongo\Framework\FrameworkServiceProvider::class,
],
```

For Laravel 10+ with package auto-discovery, the service provider registers automatically.

## Quick Start

### Creating a New Package

All Bongo packages must extend `AbstractServiceProvider`:

```php
<?php

declare(strict_types=1);

namespace Bongo\MyPackage;

use Bongo\Framework\Providers\AbstractServiceProvider;

class MyPackageServiceProvider extends AbstractServiceProvider
{
    protected string $module = 'mypackage'; // Required: matches config and views

    protected array $middlewares = [
        'myMiddleware' => MyMiddleware::class,
    ];

    protected array $commands = [
        MyCommand::class,
    ];

    public function boot(): void
    {
        parent::boot(); // Required
    }
}
```

### Creating a Model

Extend `AbstractModel` for automatic audit trails:

```php
<?php

namespace Bongo\MyPackage\Models;

use Bongo\Framework\Models\AbstractModel;
use Bongo\Framework\Traits\HasSeo;
use Bongo\Framework\Traits\HasRecursive;
use Bongo\Framework\Enums\StatusEnum;

class Page extends AbstractModel
{
    use HasSeo;       // Auto-generates: slug, meta_title, meta_description
    use HasRecursive; // Adds: parent/child relationships

    protected $casts = [
        'status' => StatusEnum::class,
    ];
}
```

### Using Helper Functions

```php
// Global helper functions
$user = user();                     // Get authenticated user
$exists = route_exists('home');     // Check if route exists
$words = camel_words('myVariable'); // "My Variable"
$key = make_key('My Page Title');   // "my-page-title"
$password = generate_password(32);  // Random secure password
```

### Adding Schema.org SEO

In your Blade templates:

```blade
{!! SEO::getMetaSchemaForPage($page) !!}
{!! SEO::getMetaSchemaForPost($post) !!}
{!! SEO::getMetaSchemaForProject($project) !!}
```

## Core Concepts

### AbstractServiceProvider

The most critical class in the framework. It provides automatic bootstrapping for:

- ✅ **Config** from `src/Config/{module}.php`
- ✅ **Routes** from `src/Routes/*.php` (api, backend, frontend, web, custom)
- ✅ **Views** from `src/Views/{module}/`
- ✅ **Migrations** from `src/Migrations/`
- ✅ **Translations** from `src/Translations/`
- ✅ **Middleware, Commands, Composers, Listeners, Subscribers** via protected arrays

Route files are automatically registered with appropriate middleware:

| File | URL Prefix | Route Name | Middleware | Use Case |
|------|-----------|-----------|------------|----------|
| `api.php` | `/api` | `api.*` | `auth:sanctum` | Protected APIs |
| `backend.php` | `/admin` | `backend.*` | `auth, employee` | Admin panel |
| `frontend.php` | None | `frontend.*` | `web` | Public routes |
| `web.php` | None | None | `web` | Standard web |
| `custom.php` | None | None | None | Webhooks |

### AbstractModel

Base model with automatic audit trails:

- Tracks `created_by`, `updated_by`, `deleted_by` (User relationships)
- Tracks `created_at`, `updated_at`, `deleted_at` timestamps
- Includes `HasDiff` trait for change tracking
- Method: `attributeExists(string $key): bool`

### Model Traits (37 Available)

**Audit Trails**: `HasCreatedBy`, `HasUpdatedBy`, `HasDeletedBy`, `HasDiff`

**Contact Info**: `HasEmail`, `HasPhone`, `HasMobile`, `HasWebsite`, `HasFullName`, `HasSocial`, `HasMarketingEmails`, etc.

**SEO**: `HasSeo` - Automatically generates slug, meta_title, meta_description on model save

**Hierarchical**: `HasRecursive` - Parent/child/sibling relationships with nested loading

**Other**: `HasStatus`, `HasVisible`, `HasDate`, `HasKey`, `HasHeaderClass`

### Custom Casts (12 Available)

Transform data between database and application:

```php
use Bongo\Framework\Casts\Encoded;
use Bongo\Framework\Casts\Pence;

class Product extends AbstractModel
{
    protected $casts = [
        'description' => Encoded::class,  // HTML entity encode/decode
        'price' => Pence::class,          // Store pence, return pounds
    ];
}
```

Available casts: `Encoded`, `Pence`, `PlainText`, `PlainWords`, `PlainNumber`, `Date`, `DateTime`, `Time`, `Domain`, `Checkbox`, and more.

### Enums (5 Available)

Type-safe enums for common fields:

```php
use Bongo\Framework\Enums\BooleanEnum;
use Bongo\Framework\Enums\StatusEnum;

class Page extends AbstractModel
{
    protected $casts = [
        'is_featured' => BooleanEnum::class,  // YES = 1, NO = 0
        'status' => StatusEnum::class,        // ACTIVE = 1, DISABLED = 0
    ];
}

// Use in code
if ($page->is_featured === BooleanEnum::YES) {
    // ...
}

// Use in dropdowns
BooleanEnum::toArray(); // ['YES' => 1, 'NO' => 0]
```

## Documentation

- **[CLAUDE.md](CLAUDE.md)** - Quick reference guide for Claude Code
- **[ARCHITECTURE.md](ARCHITECTURE.md)** - Detailed architecture, diagrams, and data flows
- **[.cursorrules](.cursorrules)** - Development guidelines for Cursor AI
- **[.github/copilot-instructions.md](.github/copilot-instructions.md)** - Code templates for GitHub Copilot

## Development

### Install Dependencies

```bash
composer install
```

### Run Tests

```bash
vendor/bin/phpunit
```

### Code Style

```bash
# Fix code style issues
vendor/bin/pint

# Check without fixing
vendor/bin/pint --test
```

## Configuration

Publish the configuration files:

```bash
php artisan vendor:publish --provider="Bongo\Framework\FrameworkServiceProvider" --tag="config"
```

This creates:
- `config/settings.php` - Framework settings (API prefix, cache, etc.)
- `config/schema.php` - Schema.org organization details
- `config/cloudflare.php` - Cloudflare API settings
- `config/developer.php` - Developer settings

## Testing

The framework uses Orchestra Testbench for package testing:

```bash
vendor/bin/phpunit
```

Test structure:
```
tests/
├── Feature/          # Integration tests
├── Unit/             # Unit tests
└── TestCase.php      # Base test class
```

## Key Components

### AbstractDatatableController

Rapid development for DataTables JSON endpoints:

```php
use Bongo\Framework\Http\Controllers\AbstractDatatableController;

class MyDatatableController extends AbstractDatatableController
{
    protected function getBaseQuery()
    {
        return MyModel::query()->with('createdBy');
    }

    protected function applyFilters()
    {
        parent::applyFilters(); // status & type filters

        if ($category = $this->request->get('category')) {
            $this->query->where('category_id', $category);
        }
    }
}
```

### Helper Classes

13 static helper classes for common tasks:

- **SEO** - Schema.org graph generation
- **Str** - String manipulation (`camelWords()`, `key()`, `id()`)
- **Route** - Route utilities (`exists()`, `is()`)
- **Cookie** - Cookie consent helpers
- **Password** - Secure password generation
- **File**, **URL**, **Html** - Various utilities
- **CloudFlare**, **Tax**, **ShortCode**, **Console**, **Log** - Specialized helpers

### Flash Messages

Controllers can chain flash methods on redirects:

```php
return redirect()
    ->route('backend.mypackage.index')
    ->success('Operation successful');

// Available: success(), error(), warning(), info(), danger()
```

## Package Dependencies

**Runtime**:
- `illuminate/contracts: ^10.0` - Laravel framework contracts
- `spatie/schema-org: ^3.23` - Schema.org structured data
- `maatwebsite/excel: ^3.1` - Excel import/export

**Development**:
- `laravel/pint: ^1.0` - Code style fixer
- `orchestra/testbench: ^8.0` - Package testing
- `phpunit/phpunit: ^10.0` - Unit testing
- `nunomaduro/collision: ^7.9` - CLI error reporting
- `nunomaduro/larastan: ^2.0.1` - Static analysis

## Stability Notice

This package is actively being developed and we would like to get feedback to improve it. Please feel free to submit feedback.

## Contributing

This package is part of the Bongo monorepo. See the main repository for contribution guidelines.

## Security

If you discover a security vulnerability, please email stuart.elliott@bespokeuk.com.

## License

The MIT License (MIT). Please see [License File](LICENSE) for more information.

## Credits

Developed by [Bespoke UK](https://bespokeuk.com)

- **Author**: Stuart Elliott
- **Email**: stuart.elliott@bespokeuk.com

## Related Packages

Bongo Framework is the foundation for 30+ modular packages in the Bongo ecosystem:

**CMS Packages**:
- bongo/builder - Page builder
- bongo/page - Page management
- bongo/post - Blog system
- bongo/menu - Menu builder
- bongo/gallery - Image galleries
- And more...

**Core Packages**:
- bongo/asset - Asset management
- bongo/blade - Blade enhancements
- bongo/captcha - Captcha integration
- bongo/image - Image processing
- bongo/user - User management
- And more...

See the main Bongo repository for the complete package list.
