<?php

namespace Bongo\Framework\Http\Controllers;

use Illuminate\Database\Eloquent\Builder;
use Illuminate\Http\JsonResponse;
use Psr\Container\ContainerExceptionInterface;
use Psr\Container\NotFoundExceptionInterface;

abstract class AbstractDatatableController extends AbstractController
{
    protected Builder $query;

    protected ?array $pagination;

    protected $request;

    protected $input;

    protected $results;

    protected ?int $totalResults;

    /**
     * @return JsonResponse
     * @throws ContainerExceptionInterface
     * @throws NotFoundExceptionInterface
     */
    public function index()
    {
        $this->setup();
        $this->applyFilters();
        $this->applySearchQuery();
        $this->applyOrderBy();
        $this->runQuery();

        return $this->createResponse();
    }

    /**
     * Set up the class for handling data tables.
     *
     * @return void
     * @throws ContainerExceptionInterface
     * @throws NotFoundExceptionInterface
     */
    protected function setup()
    {
        // Get the input
        $this->request = request();
        $this->input = request()->input();

        // Set the base query
        $this->query = $this->getBaseQuery();

        // Check if the user clicked a paginated item
        $this->pagination = [
            (int) $this->request->get('length', 10),
            (int) $this->request->get('start', 0),
        ];
    }

    protected function applyFilters()
    {
        $this->applyStatusFilter();
        $this->applyTypeFilter();
    }

    protected function applyStatusFilter()
    {
        if ($status = $this->request->get('status')) {
            $this->query->where('status', $status);
        }
    }

    protected function applyTypeFilter()
    {
        if ($type = $this->request->get('type')) {
            $this->query->where('type', $type);
        }
    }

    protected function applySearchQuery()
    {
        if (isset($this->input['search']) && ! empty($this->input['search'])) {
            $search_query = strtolower(trim($this->input['search']));

            $words = explode(' ', $search_query);
            foreach ($words as $word) {
                if (empty($word)) {
                    continue;
                }
                $this->query->where(function ($q) use ($word) {
                    $q->where('name', 'LIKE', "%{$word}%");
                });
            }
        }
    }

    protected function applyOrderBy()
    {
        $this->query->orderBy(
            $this->request->get('order_by', 'id'),
            $this->request->get('order_direction', 'desc')
        );
    }

    protected function setTotalResults()
    {
        $this->totalResults = $this->query->count();
    }

    protected function applyPagination()
    {
        $query = $this->query;
        if ($this->pagination[0] != -1) {
            $query->take($this->pagination[0]);
            if ($this->totalResults > $this->pagination[0]) {
                $query->offset($this->pagination[1]);
            }
        }
        $this->query = $query;
    }

    protected function setResults()
    {
        $this->results = $this->query->get();
    }

    protected function runQuery()
    {
        $this->setTotalResults();
        $this->applyPagination();
        $this->setResults();

        return $this;
    }

    protected function createResponse()
    {
        $data = [
            'results' => $this->results,
            'totalResults' => $this->totalResults,
        ];

        return response()->json($data);
    }
}
