<?php

namespace Bongo\Framework\Providers;

use Exception;
use Illuminate\Console\Scheduling\Schedule;
use Illuminate\Contracts\Container\BindingResolutionException;
use Illuminate\Support\Facades\Route;
use Illuminate\Support\ServiceProvider;
use ReflectionClass;

abstract class AbstractServiceProvider extends ServiceProvider
{
    protected array $commands = [];

    protected array $composers = [];

    protected string $dir;

    protected string $module;

    protected array $listeners = [];

    protected array $subscribers = [];

    protected array $middlewares = [];

    /**
     * @throws Exception
     */
    public function register(): void
    {
        if (! $this->module) {
            throw new Exception('The module name property in '.self::class.' is required');
        }
    }

    public function boot(): void
    {
        $this->setDir();
        $this->bootConfig();
        $this->bootCommands();
        $this->bootComposers();
        $this->bootListeners();
        $this->bootMiddleware();
        $this->bootMigrations();
        $this->bootRoutes();
        $this->bootSubscribers();
        $this->bootTranslations();
        $this->bootViews();

        // On the application booted event
        $this->app->booted(function () {
            $this->bootCronSchedule($this->getCronScheduler());
        });
    }

    public static function getDir(): string
    {
        $reflector = new ReflectionClass(get_called_class());

        return dirname($reflector->getFileName());
    }

    protected function setDir(): void
    {
        $this->dir = static::getDir();
    }

    protected function bootConfig(): void
    {
        $configPath = $this->dir.'/Config/';

        if (file_exists($configPath)) {
            $this->mergeConfigFrom(
                $configPath.$this->module.'.php',
                $this->module
            );
            $this->publishes([
                $configPath.$this->module.'.php' => config_path($this->module.'.php'),
            ], 'config');
        }
    }

    protected function bootCommands(): void
    {
        if ($this->app->runningInConsole() && ! empty($this->commands)) {
            $this->commands($this->commands);
        }
    }

    protected function bootComposers(): void
    {
        if (! empty($this->composers)) {
            $this->app['view']->composers($this->composers);
        }
    }

    protected function bootListeners(): void
    {
        if (! empty($this->listeners)) {
            foreach ($this->listeners as $event => $listeners) {
                foreach ($listeners as $listener) {
                    $this->app['events']->listen($event, $listener);
                }
            }
        }
    }

    protected function bootMigrations(): void
    {
        $migrationPath = $this->dir.'/Migrations/';

        if (file_exists($migrationPath)) {
            $this->loadMigrationsFrom($migrationPath);
            $this->publishes([$migrationPath => database_path('migrations')], 'migrations');
        }
    }

    protected function bootRoutes(): void
    {
        $routePath = $this->dir.'/Routes/';
        $this->bootApiRoutes($routePath);
        $this->bootAdminRoutes($routePath);
        $this->bootFrontendRoutes($routePath);
        $this->bootWebRoutes($routePath);
        $this->bootCustomRoutes($routePath);
    }

    protected function bootApiRoutes(string $routePath): void
    {
        if (file_exists($routePath.'api.php')) {
            Route::prefix(config('settings.api_prefix', 'api').'/')
                ->middleware(['api', 'auth:sanctum', 'noIndex'])
                ->as('api.')
                ->group(function () use ($routePath) {
                    $this->loadRoutesFrom($routePath.'api.php');
                });
        }
    }

    protected function bootAdminRoutes(string $routePath): void
    {
        if (file_exists($routePath.'backend.php')) {
            Route::prefix(config('settings.backend_prefix', 'admin'))
                ->middleware(['web', 'auth', 'employee', 'noIndex'])
                ->as('backend.')
                ->group(function () use ($routePath) {
                    $this->loadRoutesFrom($routePath.'backend.php');
                });
        }
    }

    protected function bootFrontendRoutes(string $routePath): void
    {
        if (file_exists($routePath.'frontend.php')) {
            Route::middleware(['web'])
                ->as('frontend.')
                ->group(function () use ($routePath) {
                    $this->loadRoutesFrom($routePath.'frontend.php');
                });
        }
    }

    protected function bootWebRoutes(string $routePath): void
    {
        if (file_exists($routePath.'web.php')) {
            Route::middleware(['web'])
                ->group(function () use ($routePath) {
                    $this->loadRoutesFrom($routePath.'web.php');
                });
        }
    }

    protected function bootCustomRoutes(string $routePath): void
    {
        // Used for routes that need no session or middleware e.g. Worldpay
        if (file_exists($routePath.'custom.php')) {
            $this->loadRoutesFrom($routePath.'custom.php');
        }
    }

    protected function bootSubscribers()
    {
        if (! empty($this->subscribers)) {
            foreach ($this->subscribers as $subscriber) {
                $this->app['events']->subscribe($subscriber);
            }
        }
    }

    protected function bootTranslations(): void
    {
        $translationPath = $this->dir.'/Translations/';

        if (file_exists($translationPath)) {
            $this->loadTranslationsFrom($translationPath, $this->module);
            $this->publishes([$translationPath => resource_path('lang/vendor/'.$this->module)]);
        }
    }

    protected function bootViews(): void
    {
        $viewPath = $this->dir.'/Views/';

        if (file_exists($viewPath)) {
            $this->loadViewsFrom($viewPath, $this->module);
            $this->publishes([
                $viewPath.'frontend' => resource_path('views/vendor/'.$this->module.'/frontend'),
            ], 'frontend');
        }
    }

    protected function bootCronSchedule(Schedule $schedule): void
    {
        //
    }

    /**
     * @throws BindingResolutionException
     */
    protected function getCronScheduler(): Schedule
    {
        return $this->app->make(Schedule::class);
    }

    protected function bootMiddleware(): void
    {
        if (! empty($this->middlewares)) {
            foreach ($this->middlewares as $key => $middleware) {
                $this->app['router']->aliasMiddleware($key, $middleware);
            }
        }
    }
}
