<?php

namespace Bongo\Framework\Schema\Concerns;

use Bongo\Image\Interfaces\Imageable;
use DateTimeInterface;
use Spatie\SchemaOrg\Graph;
use Spatie\SchemaOrg\LocalBusiness;
use Spatie\SchemaOrg\ReferencedType;
use Spatie\SchemaOrg\WebSite;

trait HandlesArticle
{
    public function addArticle(): self
    {
        $this->article($this->blogPostId)
            ->identifier($this->blogPostId)
            ->url($this->getArticleUrl())
            ->name($this->getArticleName())
            ->headline($this->getArticleHeadline())
            ->description($this->getArticleDescription())
            ->inLanguage($this->getArticleLanguage())
            ->articleBody($this->getArticleBody())
            ->wordCount($this->getArticleWordCount())
            ->image($this->getArticleImage())
            ->author($this->getArticleAuthor())
            ->isPartOf($this->getArticlePartOf())
            ->datePublished($this->getArticleDateCreated())
            ->dateCreated($this->getArticleDateCreated())
            ->dateModified($this->getArticleDateUpdated());

        return $this;
    }

    protected function getArticleUrl(): ?string
    {
        return $this->getUrlForEntity($this->entity) ?? '--';
    }

    protected function getArticleName(): ?string
    {
        return $this->entity->meta_title ?? $this->entity->name ?? '--';
    }

    protected function getArticleHeadline(): ?string
    {
        return $this->entity->meta_title ?? $this->entity->name ?? '--';
    }

    protected function getArticleDescription(): ?string
    {
        return $this->entity->meta_description ?? $this->entity->summary ?? '--';
    }

    protected function getArticleBody(): ?string
    {
        return $this->entity->getContentAsPlainText() ?? '--';
    }

    protected function getArticleWordCount(): array|int|null
    {
        return ! empty($this->getArticleBody())
            ? str_word_count($this->getArticleBody())
            : null;
    }

    protected function getArticleImage(): ?string
    {
        // If the entity is not imageable or does not have a primary image, use a default image
        if (! $this->entity instanceof Imageable || ! $this->entity->hasPrimaryImage()) {
            return asset('images/frontend/screenshot.jpeg');
        }

        // Otherwise, use the entity's primary image
        return $this->entity->getPrimaryImage(['preset' => 'medium']);
    }

    protected function getArticleLanguage(): string
    {
        return app()->getLocale() ?? 'en-GB';
    }

    protected function getArticleAuthor(): LocalBusiness|ReferencedType|Graph
    {
        return $this->localBusiness($this->localBusinessId)->referenced();
    }

    protected function getArticlePartOf(): WebSite|ReferencedType|Graph
    {
        return $this->webSite($this->webSiteId)->referenced();
    }

    protected function getArticleDateCreated(): DateTimeInterface|string|null
    {
        return $this->entity->created_at ?? '--';
    }

    protected function getArticleDateUpdated(): DateTimeInterface|string|null
    {
        return $this->entity->updated_at ?? '--';
    }
}
