<?php

namespace Bongo\Framework\Traits;

use Illuminate\Database\Eloquent\Relations\BelongsTo;
use Illuminate\Database\Eloquent\Relations\HasMany;
use Illuminate\Support\Collection;

trait HasRecursive
{
    public array $ancestors = [];

    public array $descendents = [];

    public function initializeHasRecursive(): void
    {
        $this->mergeFillable(['parent_id']);
    }

    public function parent(): BelongsTo
    {
        return $this->belongsTo(self::class, 'parent_id');
    }
    public function hasParent(): bool
    {
        $this->loadMissing('parent');

        return ! empty($this->parent_id) && ! is_null($this->parent);
    }
    public function nestedParents(): BelongsTo
    {
        return $this->belongsTo(self::class, 'parent_id')
            ->with('nestedParents');
    }
    public function hasNestedParents(): bool
    {
        $this->loadMissing('nestedParents');

        return ! empty($this->nestedParents) && count($this->nestedParents) > 0;
    }
    public function isParent(): bool
    {
        return empty($this->parent_id);
    }

    public function getAncestors(): Collection
    {
        if ($this->hasParent()) {
            $this->collectAncestors($this->parent);
        }

        return collect($this->ancestors);
    }
    private function collectAncestors($parent): void
    {
        $this->ancestors[] = $parent;

        if ($parent->hasParent()) {
            $this->collectAncestors($parent->parent);
        }
    }
    public function getAncestorsAsArray(): array
    {
        // Get the root ancestor
        $rootAncestor = $this->getAncestors()->firstWhere('parent_id', null);
        if (! $rootAncestor || ! $rootAncestor->hasNestedChildren()) {
            return [];
        }

        return [
            $rootAncestor->toArray(),
        ];
    }

    public function children(): HasMany
    {
        return $this
            ->hasMany(self::class, 'parent_id');
    }
    public function hasChildren(): bool
    {
        $this->loadMissing('children');

        return ! empty($this->children) && count($this->children) > 0;
    }

    public function nestedChildren(): HasMany
    {
        return $this->children()
            ->with('nestedChildren');
    }
    public function hasNestedChildren(): bool
    {
        $this->loadMissing('nestedChildren');

        return ! empty($this->nestedChildren) && count($this->nestedChildren) > 0;
    }

    public function isChild(): bool
    {
        return ! empty($this->parent_id);
    }

    public function getDescendents(): Collection
    {
        if ($this->hasChildren()) {
            $this->collectDescendents($this->children);
        }

        return collect($this->descendents);
    }
    private function collectDescendents($children): void
    {
        foreach ($children as $child) {
            $this->descendents[] = $child;

            if ($child->hasChildren()) {
                $this->collectDescendents($child->children);
            }
        }
    }
    public function getDescendentsAsArray(): array
    {
        if (! $this->hasNestedChildren()) {
            return [];
        }

        return [
            $this->toArray(),
        ];
    }

    public function siblings()
    {
        return self::all()
            ->where('id', '!=', $this->id)
            ->where('parent_id', $this->parent_id);
    }
    public function hasSiblings(): bool
    {
        $this->loadMissing('siblings');

        return ! empty($this->siblings) && count($this->siblings) > 0;
    }
}
