<?php

namespace Bongo\Framework\Schema\Concerns;

use Bongo\Image\Interfaces\Imageable;
use DateTimeInterface;
use Spatie\SchemaOrg\Graph;
use Spatie\SchemaOrg\LocalBusiness;
use Spatie\SchemaOrg\ReferencedType;
use Spatie\SchemaOrg\WebSite;

trait HandlesBlogPost
{
    public function addBlogPost(): self
    {
        $this->blogPosting($this->blogPostId)
            ->identifier($this->blogPostId)
            ->url($this->getBlogPostUrl())
            ->name($this->getBlogPostName())
            ->headline($this->getBlogPostHeadline())
            ->description($this->getBlogPostDescription())
            ->inLanguage($this->getBlogPostLanguage())
            ->articleBody($this->getBlogPostBody())
            ->wordCount($this->getBlogPostWordCount())
            ->image($this->getBlogPostImage())
            ->author($this->getBlogPostAuthor())
            ->isPartOf($this->getBlogPostPartOf())
            ->datePublished($this->getBlogPostDateCreated())
            ->dateCreated($this->getBlogPostDateCreated())
            ->dateModified($this->getBlogPostDateUpdated());

        return $this;
    }

    protected function getBlogPostUrl(): ?string
    {
        return $this->getUrlForEntity($this->entity) ?? '--';
    }

    protected function getBlogPostName(): ?string
    {
        return $this->entity->meta_title ?? $this->entity->name ?? '--';
    }

    protected function getBlogPostHeadline(): ?string
    {
        return $this->entity->meta_title ?? $this->entity->name ?? '--';
    }

    protected function getBlogPostDescription(): ?string
    {
        return $this->entity->meta_description ?? $this->entity->summary ?? '--';
    }

    protected function getBlogPostBody(): ?string
    {
        return $this->entity->getContentAsPlainText() ?? '--';
    }

    protected function getBlogPostWordCount(): array|int|null
    {
        return ! empty($this->getBlogPostBody())
            ? str_word_count($this->getBlogPostBody())
            : null;
    }

    protected function getBlogPostImage(): ?string
    {
        // If the entity is not imageable or does not have a primary image, use a default image
        if (! $this->entity instanceof Imageable || ! $this->entity->hasPrimaryImage()) {
            return asset('images/frontend/screenshot.jpeg');
        }

        // Otherwise, use the entity's primary image
        return $this->entity->getPrimaryImage(['preset' => 'medium']);
    }

    protected function getBlogPostLanguage(): string
    {
        return app()->getLocale() ?? 'en-GB';
    }

    protected function getBlogPostAuthor(): LocalBusiness|ReferencedType|Graph
    {
        return $this->localBusiness($this->localBusinessId)->referenced();
    }

    protected function getBlogPostPartOf(): WebSite|ReferencedType|Graph
    {
        return $this->webSite($this->webSiteId)->referenced();
    }

    protected function getBlogPostDateCreated(): DateTimeInterface|string|null
    {
        return $this->entity->created_at ?? '--';
    }

    protected function getBlogPostDateUpdated(): DateTimeInterface|string|null
    {
        return $this->entity->updated_at ?? '--';
    }
}
