<?php

namespace Bongo\Framework\Schema\Concerns;

use Illuminate\Contracts\Container\BindingResolutionException;
use Spatie\SchemaOrg\ContactPoint;
use Spatie\SchemaOrg\Graph;
use Spatie\SchemaOrg\PostalAddress;
use Spatie\SchemaOrg\ReferencedType;

trait HandlesLocalBusiness
{
    use HandlesReviews;
    use HandlesSocialLinks;

    /** @throws BindingResolutionException */
    public function addLocalBusiness(): self
    {
        $this->localBusiness($this->localBusinessId)
            ->identifier($this->localBusinessId)
            ->url($this->getLocalBusinessUrl())
            ->name($this->getLocalBusinessName())
            ->alternateName($this->getLocalBusinessAlternateName())
            ->email($this->getLocalBusinessEmail())
            ->telephone($this->getLocalBusinessPhone())
            ->logo($this->getLocalBusinessLogo())
            ->taxID($this->getLocalBusinessTaxId())
            ->vatID($this->getLocalBusinessVatID())
            ->address($this->getLocalBusinessAddress())
            ->contactPoint($this->getLocalBusinessContactPoint());

        return $this;
    }

    /** @throws BindingResolutionException */
    public function addSocialLinksToLocalBusiness(): self
    {
        $socialLinks = $this->getSocialLinks();

        if (! empty($socialLinks)) {
            $this->localBusiness($this->localBusinessId)->sameAs($socialLinks);
        }

        return $this;
    }

    public function addReviewsToLocalBusiness(): self
    {
        // If the review package is not enabled, or there are no reviews, return
        if (! $this->hasReviews()) {
            return $this;
        }

        // Otherwise add the aggregate rating and reviews
        $this->localBusiness($this->localBusinessId)
            ->aggregateRating($this->aggregateRatingOfReviews())
            ->review($this->getReviews());

        return $this;
    }

    protected function getLocalBusinessUrl(): string
    {
        return url()->to('/');
    }

    /** @throws BindingResolutionException */
    protected function getLocalBusinessName(): ?string
    {
        return setting('client::company.name') ?? '--';
    }

    /** @throws BindingResolutionException */
    protected function getLocalBusinessAlternateName(): ?string
    {
        return setting('client::company.trading_name');
    }

    /** @throws BindingResolutionException */
    protected function getLocalBusinessEmail(): ?string
    {
        return setting('client::company.email') ?? '--';
    }

    /** @throws BindingResolutionException */
    protected function getLocalBusinessPhone(): ?string
    {
        return setting('client::company.phone') ?? '--';
    }

    /** @throws BindingResolutionException */
    protected function getLocalBusinessLogo(): ?string
    {
        if ($logo = setting('theme::logo.default')) {
            return asset($logo);
        }

        return asset('images/frontend/logo.png') ?? '--';
    }

    /** @throws BindingResolutionException */
    protected function getLocalBusinessTaxId(): ?string
    {
        return setting('client::company.company_number') ?? '--';
    }

    /** @throws BindingResolutionException */
    protected function getLocalBusinessVatID(): ?string
    {
        return setting('client::company.vat_number') ?? '--';
    }

    protected function getLocalBusinessAddress(): PostalAddress|ReferencedType|Graph
    {
        return $this->postalAddress($this->postalAddressId)->referenced();
    }

    protected function getLocalBusinessContactPoint(): ContactPoint|ReferencedType|Graph
    {
        return $this->contactPoint($this->contactPointId)->referenced();
    }
}
