<?php

namespace Bongo\Framework\Schema\Concerns;

use Illuminate\Contracts\Container\BindingResolutionException;
use Spatie\SchemaOrg\ContactPoint;
use Spatie\SchemaOrg\Graph;
use Spatie\SchemaOrg\Person;
use Spatie\SchemaOrg\Place;
use Spatie\SchemaOrg\PostalAddress;
use Spatie\SchemaOrg\QuantitativeValue;
use Spatie\SchemaOrg\ReferencedType;
use Spatie\SchemaOrg\Schema;

trait HandlesOrganization
{
    use HandlesReviews;
    use HandlesSocialLinks;

    /** @throws BindingResolutionException */
    public function addOrganization(): self
    {
        $this->organization($this->organizationId)
            ->identifier($this->organizationId)
            ->url($this->getOrganizationUrl())
            ->name($this->getOrganizationName())
            ->alternateName($this->getOrganizationAlternateName())
            ->email($this->getOrganizationEmail())
            ->telephone($this->getOrganizationPhone())
            ->logo($this->getOrganizationLogo())
            ->image($this->getOrganizationImage())
            ->taxID($this->getOrganizationTaxId())
            ->vatID($this->getOrganizationVatID())
            ->address($this->getOrganizationAddress())
            ->contactPoint($this->getOrganizationContactPoint())
            ->location($this->getOrganizationLocation())
            ->founder($this->getOrganizationFounder())
            ->foundingLocation($this->getOrganizationLocation())
            ->numberOfEmployees($this->getOrganizationNumberOfEmployees());

        return $this;
    }

    /** @throws BindingResolutionException */
    public function addSocialLinksToOrganisation(): self
    {
        $socialLinks = $this->getSocialLinks();

        if (! empty($socialLinks)) {
            $this->organization($this->organizationId)->sameAs($socialLinks);
        }

        return $this;
    }

    public function addReviewsToOrganisation(): self
    {
        // If the review package is not enabled, or there are no reviews, return
        if (! $this->hasReviews()) {
            return $this;
        }

        // Otherwise add the aggregate rating and reviews
        $this->organization($this->organizationId)
            ->aggregateRating($this->aggregateRatingOfReviews())
            ->review($this->getReviews());

        return $this;
    }

    protected function getOrganizationUrl(): string
    {
        return url()->to('/');
    }

    /** @throws BindingResolutionException */
    protected function getOrganizationName(): ?string
    {
        return setting('client::company.name') ?? '--';
    }

    /** @throws BindingResolutionException */
    protected function getOrganizationAlternateName(): ?string
    {
        return setting('client::company.trading_name');
    }

    /** @throws BindingResolutionException */
    protected function getOrganizationEmail(): ?string
    {
        return setting('client::company.email') ?? '--';
    }

    /** @throws BindingResolutionException */
    protected function getOrganizationPhone(): ?string
    {
        return setting('client::company.phone') ?? '--';
    }

    /** @throws BindingResolutionException */
    protected function getOrganizationLogo(): ?string
    {
        if ($logo = setting('theme::logo.default')) {
            return asset($logo);
        }

        return asset('images/frontend/logo.png') ?? '--';
    }

    protected function getOrganizationImage(): ?string
    {
        return asset('images/frontend/screenshot.jpeg') ?? '--';
    }

    /** @throws BindingResolutionException */
    protected function getOrganizationTaxId(): ?string
    {
        return setting('client::company.company_number') ?? '--';
    }

    /** @throws BindingResolutionException */
    protected function getOrganizationVatID(): ?string
    {
        return setting('client::company.vat_number') ?? '--';
    }

    protected function getOrganizationAddress(): PostalAddress|ReferencedType|Graph
    {
        return $this->postalAddress($this->postalAddressId)->referenced();
    }

    protected function getOrganizationContactPoint(): ContactPoint|ReferencedType|Graph
    {
        return $this->contactPoint($this->contactPointId)->referenced();
    }

    protected function getOrganizationLocation(): Place|ReferencedType|Graph
    {
        return $this->place($this->placeId)->referenced();
    }

    protected function getOrganizationFounder(): Person|ReferencedType|Graph
    {
        return $this->person($this->founderId)->referenced();
    }

    protected function getOrganizationNumberOfEmployees(): QuantitativeValue
    {
        return Schema::quantitativeValue()
            ->minValue(1)
            ->maxValue(50);
    }
}
