<?php

declare(strict_types=1);

namespace Bongo\Framework\Traits\Audit;

use Bongo\User\Models\User;
use Illuminate\Database\Eloquent\Builder;
use Illuminate\Database\Eloquent\Relations\BelongsTo;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Schema;

trait HasCreatedBy
{
    public function initializeHasCreatedBy(): void
    {
        $this->mergeFillable(['created_by']);
    }

    public static function bootHasCreatedBy(): void
    {
        static::creating(function ($model) {
            if (empty($model->created_by) && Schema::hasColumn($model->getTable(), 'created_by')) {
                $model->created_by = Auth::user() ? Auth::user()->id : null;
            }
        });
    }

    public function creator(): BelongsTo
    {
        return $this->belongsTo(User::class, 'created_by')->withTrashed();
    }

    public function scopeByCreator(Builder $query, ?int $userId = null): Builder
    {
        return $query->where('created_by', $userId ?? Auth::user()->id);
    }
}
