<?php

namespace Bongo\Framework;

use Bongo\Asset\AssetServiceProvider;
use Bongo\Blade\BladeServiceProvider;
use Bongo\Builder\BuilderServiceProvider;
use Bongo\Dashboard\DashboardServiceProvider;
use Bongo\Document\DocumentServiceProvider;
use Bongo\Enquiry\EnquiryServiceProvider;
use Bongo\Enum\EnumServiceProvider;
use Bongo\Estimate\EstimateServiceProvider;
use Bongo\Form\FormServiceProvider;
use Bongo\Framework\Http\Middleware\NoIndex;
use Bongo\Framework\Http\ViewComposers\BreadcrumbComposer;
use Bongo\Framework\Http\ViewComposers\PackageComposer;
use Bongo\Framework\Providers\AbstractServiceProvider;
use Bongo\Framework\Providers\RouteServiceProvider;
use Bongo\Gallery\GalleryServiceProvider;
use Bongo\Image\ImageServiceProvider;
use Bongo\Install\InstallServiceProvider;
use Bongo\Menu\MenuServiceProvider;
use Bongo\Package\PackageServiceProvider;
use Bongo\Page\PageServiceProvider;
use Bongo\Post\PostServiceProvider;
use Bongo\Profile\ProfileServiceProvider;
use Bongo\Project\ProjectServiceProvider;
use Bongo\RealGreen\RealGreenServiceProvider;
use Bongo\Redirect\RedirectServiceProvider;
use Bongo\Referrer\ReferrerServiceProvider;
use Bongo\Review\ReviewServiceProvider;
use Bongo\Setting\SettingServiceProvider;
use Bongo\Sitemap\SitemapServiceProvider;
use Bongo\User\UserServiceProvider;
use Illuminate\Contracts\Container\BindingResolutionException;
use Illuminate\Http\RedirectResponse;
use Illuminate\Support\Facades\File;
use Illuminate\Support\Facades\Schema;
use ReflectionException;
use Symfony\Component\HttpFoundation\Request;

/**
 * Class FrameworkServiceProvider.
 *
 * @category  Bespoke_Software
 * @author    Bespoke.ws Ltd <support@bespokeuk.com>
 * @copyright 2015-2022 Bespoke.ws Ltd, All Rights Reserved
 * @license   Proprietary and confidential
 * Unauthorized copying of this file, via any medium is strictly prohibited
 * @link      https://bespokeuk.com
 * @package   Bongo\Framework
 */
class FrameworkServiceProvider extends AbstractServiceProvider
{
    /** @var string */
    protected $module = 'framework';

    /** @var array */
    protected $composers = [
        BreadcrumbComposer::class => [
            'blade::breadcrumbs',
            'framework::frontend.layouts.partials.breadcrumbs',
        ],
        PackageComposer::class => [
            'framework::backend.layouts.partials.sidebar_menu',
            'dashboard::backend.index',
        ],
    ];

    protected $middlewares = [
        'noIndex' => NoIndex::class,
    ];

    /**
     * @throws BindingResolutionException
     * @throws ReflectionException
     */
    public function boot()
    {
        parent::boot();

        // Temp fix for MySql
        Schema::defaultStringLength(191);

        // Fix for cloudflare proxy ip address
        Request::setTrustedProxies(['REMOTE_ADDR'], Request::HEADER_X_FORWARDED_FOR);

        // One off boots
        $this->bootFlashResponseMacro();
        $this->bootConfig();

        // Load the core providers
        $this->loadCoreProviders();

        // Has the installer been run
        if (! config('settings.install_complete')) {
            $this->loadInstaller();
        } else {
            $this->checkEnv();
            $this->checkStoragePath();
            $this->loadPackages();
        }
    }

    protected function loadCoreProviders(): void
    {
        $this->app->register(RouteServiceProvider::class);
        $this->app->register(ReferrerServiceProvider::class);
        $this->app->register(AssetServiceProvider::class);
        $this->app->register(BladeServiceProvider::class);
        $this->app->register(EnumServiceProvider::class);
        $this->app->register(PackageServiceProvider::class);
    }

    protected function loadInstaller(): void
    {
        $this->app->register(InstallServiceProvider::class);

        // Must be loaded last so that the catchall redirect to /install is picked up
        $this->app->booted(function () {
            $this->app->register(UserServiceProvider::class);
        });
    }

    protected function checkEnv(): void
    {
        if (!file_exists(base_path('.env'))) {
            if (file_exists(base_path('.env.example'))) {
                copy(base_path('.env.example'), base_path('.env'));
            } else {
                touch(base_path('.env'));
            }
        }
    }

    protected function checkStoragePath(): void
    {
        // Make sure we have the /storage/app/public directory
        File::ensureDirectoryExists(storage_path('app/public'));

        // Once the app is booted, make sure we have the storage symlink setup
        $this->app->booted(function () {
            if (!File::exists(public_path('storage'))) {
                File::link(storage_path('app/public'), public_path('storage'));
            }
        });
    }

    /**
     * @throws BindingResolutionException
     */
    protected function loadPackages(): void
    {
        // Register the standard framework packages
        $this->app->register(BuilderServiceProvider::class);
        $this->app->register(DashboardServiceProvider::class);
        $this->app->register(DocumentServiceProvider::class);
        $this->app->register(EnquiryServiceProvider::class);
        $this->app->register(FormServiceProvider::class);
        $this->app->register(ImageServiceProvider::class);
        $this->app->register(MenuServiceProvider::class);
        $this->app->register(ProfileServiceProvider::class);
        $this->app->register(SettingServiceProvider::class);
        $this->app->register(SitemapServiceProvider::class);
        $this->app->register(UserServiceProvider::class);

        // Register if enabled
        $packageManager = $this->app->make('package_manager');

        if ($packageManager->isEnabled('estimate')) {
            $this->app->register(EstimateServiceProvider::class);

            if ($packageManager->isEnabled('real_green')) {
                $this->app->register(RealGreenServiceProvider::class);
            }
        }
        if ($packageManager->isEnabled('gallery')) {
            $this->app->register(GalleryServiceProvider::class);
        }
        if ($packageManager->isEnabled('post')) {
            $this->app->register(PostServiceProvider::class);
        }
        if ($packageManager->isEnabled('project')) {
            $this->app->register(ProjectServiceProvider::class);
        }
        if ($packageManager->isEnabled('review')) {
            $this->app->register(ReviewServiceProvider::class);
        }

        // Once the app is booted
        $this->app->booted(function () {

            // Has to be loaded second to last
            $this->app->register(RedirectServiceProvider::class);

            // Has to be registered last due to /{slug}
            $this->app->register(PageServiceProvider::class);
            include __DIR__ . '/helpers.php';
        });
    }

    protected function bootFlashResponseMacro()
    {
        $types = [
            'success' => 'success',
            'info' => 'info',
            'warning' => 'warning',
            'danger' => 'danger',
            'error' => 'danger',
        ];

        foreach ($types as $method => $type) {
            RedirectResponse::macro($method, function ($message) use ($type) {
                $this->with('message_type', $type)->with('message', $message);

                return $this;
            });
        }
    }

    protected function bootConfig(): void
    {
        // Config Merge
        $this->mergeConfigFrom(__DIR__ . '/Config/cloudflare.php', 'cloudflare');
        $this->mergeConfigFrom(__DIR__ . '/Config/developer.php', 'developer');
        $this->mergeConfigFrom(__DIR__ . '/Config/settings.php', 'settings');

        // Config Publish
        $this->publishes([
            __DIR__ . '/Config/cloudflare.php' => config_path('cloudflare.php'),
            __DIR__ . '/Config/developer.php' => config_path('developer.php'),
            __DIR__ . '/Config/settings.php' => config_path('settings.php'),
        ], 'config');
    }
}
