<?php

namespace Bongo\Framework\Http\Controllers;

use Illuminate\Http\JsonResponse;

/**
 * Class AbstractDatatableController.
 *
 * @category  Bespoke_Software
 * @author    Bespoke.ws Ltd <support@bespokeuk.com>
 * @copyright 2015-2020 Bespoke.ws Ltd, All Rights Reserved
 * @license   Proprietary and confidential
 * Unauthorized copying of this file, via any medium is strictly prohibited
 * @link      https://bespokeuk.com
 * @package   Bongo\Framework\Http\Controllers
 */
abstract class AbstractDatatableController extends AbstractController
{
    /**
     * @var
     */
    protected $query;

    /**
     * @var
     */
    protected $pagination;

    /**
     * @var
     */
    protected $request;

    /**
     * @var
     */
    protected $input;

    /**
     * @var
     */
    protected $results;

    /**
     * @var
     */
    protected $totalResults;

    /**
     * @return JsonResponse
     */
    public function index()
    {
        $this->setup();
        $this->applyFilters();
        $this->applySearchQuery();
        $this->applyOrderBy();
        $this->runQuery();

        return $this->createResponse();
    }

    /**
     * Setup the class for handling data tables.
     *
     * @return void
     */
    protected function setup()
    {
        // Get the input
        $this->request = request();
        $this->input = request()->input();

        // Set the base query
        $this->query = $this->getBaseQuery();

        // Check if the user clicked a paginated item
        $this->pagination = [
            (int) $this->request->get('length', 10),
            (int) $this->request->get('start', 0),
        ];
    }

    /**
     * Apply filters.
     */
    protected function applyFilters()
    {
        $this->applyStatusFilter();
        $this->applyTypeFilter();
    }

    /**
     * Status filter.
     */
    protected function applyStatusFilter()
    {
        if ($status = $this->request->get('status')) {
            $this->query->where('status', $status);
        }
    }

    /**
     * Type filter.
     */
    protected function applyTypeFilter()
    {
        if ($type = $this->request->get('type')) {
            $this->query->where('type', $type);
        }
    }

    /**
     * Apply search query.
     */
    protected function applySearchQuery()
    {
        if (isset($this->input['search']) && !empty($this->input['search'])) {
            $search_query = strtolower(trim($this->input['search']));

            $words = explode(' ', $search_query);
            foreach ($words as $word) {
                if (empty($word)) {
                    continue;
                }
                $this->query->where(function ($q) use ($word) {
                    $q->where('name', 'LIKE', "%{$word}%");
                });
            }
        }
    }

    /**
     * Apply order by.
     */
    protected function applyOrderBy()
    {
        $this->query->orderBy(
            $this->request->get('order_by', 'id'),
            $this->request->get('order_direction', 'desc')
        );
    }

    /**
     * Set the total results.
     */
    protected function setTotalResults()
    {
        $this->totalResults = $this->query->count();
    }

    /**
     * Apply pagination.
     */
    protected function applyPagination()
    {
        $query = $this->query;
        if ($this->pagination[0] != -1) {
            $query->take($this->pagination[0]);
            if ($this->totalResults > $this->pagination[0]) {
                $query->offset($this->pagination[1]);
            }
        }
        $this->query = $query;
    }

    /**
     * Set the results.
     */
    protected function setResults()
    {
        $this->results = $this->query->get();
    }

    /**
     * Run the query and capture results.
     *
     * @return $this
     */
    protected function runQuery()
    {
        $this->setTotalResults();
        $this->applyPagination();
        $this->setResults();

        return $this;
    }

    /**
     * Create the json response.
     */
    protected function createResponse()
    {
        $data = [
            'results'      => $this->results,
            'totalResults' => $this->totalResults,
        ];

        return response()->json($data, 200);
    }
}
