<?php

namespace Bongo\Framework\Providers;

use Exception;
use Illuminate\Console\Scheduling\Schedule;
use Illuminate\Contracts\Container\BindingResolutionException;
use Illuminate\Support\ServiceProvider;
use ReflectionClass;
use ReflectionException;

/**
 * Class AbstractServiceProvider.
 *
 * @category  Bespoke_Software
 * @author    Bespoke.ws Ltd <support@bespokeuk.com>
 * @copyright 2015-2020 Bespoke.ws Ltd, All Rights Reserved
 * @license   Proprietary and confidential
 * Unauthorized copying of this file, via any medium is strictly prohibited
 * @link      https://bespokeuk.com
 * @package   Bongo\Framework\Providers
 */
abstract class AbstractServiceProvider extends ServiceProvider
{
    /**
     * Commands array.
     *
     * @var array
     */
    protected $commands = [];

    /**
     * Composers array.
     *
     * @var array
     */
    protected $composers = [];

    /**
     * The base directory path of the calling class.
     *
     * @var string
     */
    protected $dir;

    /**
     * The name of this module e.g task.
     *
     * @var null
     */
    protected $module = null;

    /**
     * Listeners.
     *
     * @var array
     */
    protected $listeners = [];

    /**
     * Subscribers.
     *
     * @var array
     */
    protected $subscribers = [];

    /**
     * Register any application services.
     *
     * @return void
     * @throws Exception
     */
    public function register()
    {
        if (!$this->module) {
            throw new Exception('The module name property in ' . self::class . ' is required');
        }
    }

    /**
     * Bootstrap any application services.
     *
     * @return void
     * @throws ReflectionException
     */
    public function boot()
    {
        $this->setDir();
        $this->bootConfig();
        $this->bootCommands();
        $this->bootComposers();
        $this->bootListeners();
        $this->bootMiddleware();
        $this->bootMigrations();
        $this->bootRoutes();
        $this->bootSubscribers();
        $this->bootTranslations();
        $this->bootViews();

        // On the application booted event
        $this->app->booted(function () {
            $this->bootCronSchedule($this->getCronScheduler());
        });
    }

    /**
     * Set the directory of the calling class.
     *
     * @throws ReflectionException
     */
    protected function setDir()
    {
        $reflector = new ReflectionClass(get_called_class());
        $this->dir = dirname($reflector->getFileName());
    }

    /**
     * Boot config.
     */
    protected function bootConfig()
    {
        $configPath = $this->dir . '/Config/';

        if (file_exists($configPath)) {
            $this->mergeConfigFrom(
                $configPath . $this->module . '.php',
                $this->module
            );
            $this->publishes([
                $configPath . $this->module . '.php' => config_path($this->module . '.php'),
            ], 'config');
        }
    }

    /**
     * Boot commands.
     */
    protected function bootCommands()
    {
        if ($this->app->runningInConsole() && !empty($this->commands)) {
            $this->commands($this->commands);
        }
    }

    /**
     * Boot composers.
     */
    protected function bootComposers()
    {
        if (!empty($this->composers)) {
            $this->app['view']->composers($this->composers);
        }
    }

    /**
     * Boot listeners.
     */
    protected function bootListeners()
    {
        if (!empty($this->listeners)) {
            foreach ($this->listeners as $event => $listeners) {
                foreach ($listeners as $listener) {
                    $this->app['events']->listen($event, $listener);
                }
            }
        }
    }

    /**
     * Boot migrations.
     */
    protected function bootMigrations()
    {
        $migrationPath = $this->dir . '/Migrations/';

        if (file_exists($migrationPath)) {
            $this->loadMigrationsFrom($migrationPath);
            $this->publishes([$migrationPath => database_path('migrations')], 'migrations');
        }
    }

    /**
     * Boot routes.
     */
    protected function bootRoutes()
    {
        $routePath = $this->dir . '/Routes/';

        if (file_exists($routePath . 'api.php')) {
            $this->loadRoutesFrom($routePath . 'api.php');
        }
        if (file_exists($routePath . 'web.php')) {
            $this->loadRoutesFrom($routePath . 'web.php');
        }
        if (file_exists($routePath . 'backend.php')) {
            $this->loadRoutesFrom($routePath . 'backend.php');
        }
        if (file_exists($routePath . 'frontend.php')) {
            $this->loadRoutesFrom($routePath . 'frontend.php');
        }
    }

    /**
     * Boot subscribers.
     */
    protected function bootSubscribers()
    {
        if (!empty($this->subscribers)) {
            foreach ($this->subscribers as $subscriber) {
                $this->app['events']->subscribe($subscriber);
            }
        }
    }

    /**
     * Boot translations.
     */
    protected function bootTranslations()
    {
        $translationPath = $this->dir . '/Translations/';

        if (file_exists($translationPath)) {
            $this->loadTranslationsFrom($translationPath, $this->module);
            $this->publishes([$translationPath => resource_path('lang/vendor/' . $this->module)]);
        }
    }

    /**
     * Boot views.
     */
    protected function bootViews()
    {
        $viewPath = $this->dir . '/Views/';

        if (file_exists($viewPath)) {
            $this->loadViewsFrom($viewPath, $this->module);
            $this->publishes([$viewPath . 'frontend' => resource_path('views/vendor/' . $this->module . '/frontend')], 'frontend');
        }
    }

    /**
     * Boot Cron Schedule.
     *
     * @param Schedule $schedule
     *
     * @return void
     */
    protected function bootCronSchedule(Schedule $schedule)
    {
        //
    }

    /**
     * Get cron scheduler instance.
     *
     * @return Schedule
     * @throws BindingResolutionException
     */
    protected function getCronScheduler()
    {
        return $this->app->make(Schedule::class);
    }

    /**
     * Boot middleware.
     */
    protected function bootMiddleware()
    {
        if (!empty($this->middlewares)) {
            foreach ($this->middlewares as $key => $middleware) {
                $this->app['router']->aliasMiddleware($key, $middleware);
            }
        }
    }
}
