<?php

use Bongo\Framework\Helpers\Referrer;
use Illuminate\Contracts\Auth\Authenticatable;
use Illuminate\Contracts\Container\BindingResolutionException;
use Illuminate\Support\Facades\Log;
use Illuminate\Support\Facades\Route;
use Illuminate\Support\Str;

/*
 * Custom Helpers
 *
 * @category  Bespoke_Software
 * @author    Bespoke.ws Ltd <support@bespokeuk.com>
 * @copyright 2015-2020 Bespoke.ws Ltd, All Rights Reserved
 * @license   Proprietary and confidential
 * Unauthorized copying of this file, via any medium is strictly prohibited
 * @link      https://bespokeuk.com
 */
if (!function_exists('user')) {
    /**
     * Get the authenticated user instance.
     *
     * @return Authenticatable|null
     */
    function user()
    {
        return auth()->user();
    }
}

if (!function_exists('route_exists')) {
    /**
     * Helper to check if a route exists.
     *
     * @param $name
     *
     * @return string
     */
    function route_exists($name)
    {
        $routes = Route::getRoutes();

        foreach ($routes as $route) {
            if ($route->getName() == $name) {
                return true;
            }
        }

        return false;
    }
}

if (!function_exists('route_is')) {
    /**
     * Helper to check if the current route name is.
     *
     * @param $route
     *
     * @return bool
     */
    function route_is($route)
    {
        $route = strtolower($route);

        return in_array(Route::currentRouteName(), [
            "{$route}.index",
            "{$route}.create",
            "{$route}.show",
            "{$route}.edit",
            "{$route}.delete"
        ]);
    }
}

if (!function_exists('camel_words')) {
    /**
     * Camel case words.
     *
     * @param      $input
     * @param bool $plural
     *
     * @return mixed|string
     */
    function camel_words($input, $plural = false)
    {
        $input = str_replace(['-', '_'], ' ', $input);
        if ($plural) {
            $words = explode(' ', $input);
            if ($pluralized = Str::plural(array_pop($words))) {
                $words = array_merge($words, [$pluralized]);
            }
            $input = implode(' ', $words);
        }
        $input = lcfirst(ucwords($input));
        $input = str_replace(' ', '', $input);

        return $input;
    }
}

if (!function_exists('plain_text')) {
    /**
     * Plain text.
     *
     * @param $input
     *
     * @return null|string|string[]
     */
    function plain_text($input)
    {
        $input = preg_replace('<\/h[1-6]>', '<br/>', $input);
        $input = html_entity_decode($input);
        $input = urldecode($input);
        $input = strip_tags($input, '<p><br>');

        return $input;
    }
}

if (!function_exists('console_print')) {
    /**
     * If in console the print line.
     *
     * @param        $message
     * @param string $type
     */
    function console_print($message, $type = 'info')
    {
        if (app()->runningInConsole()) {
            $output = new Symfony\Component\Console\Output\ConsoleOutput();
            $output->writeln("<$type>$message</$type>");
        }
    }
}

if (!function_exists('log_exception')) {
    /**
     * Log Exception on the Bespoke admin.
     *
     * @param Exception $exception
     * @param int       $code
     */
    function log_exception(Exception $exception, $code = 500)
    {
        // Otherwise report the error to Bespoke
        $error = [
            'type'        => get_class($exception),
            'environment' => app()->environment(),
            'message'     => $exception->getMessage(),
            'file'        => $exception->getFile(),
            'line_number' => $exception->getLine(),
            'uri'         => request()->getUri(),
            'port'        => request()->getPort(),
            'client_ip'   => request()->getClientIp(),
            'method'      => request()->method(),
            'format'      => request()->format(),
            'headers'     => json_encode(request()->header()),
            'stacktrace'  => json_encode(debug_backtrace()),
            'user'        => json_encode((!auth()->check()) ? [] : auth()->user()->toArray()),
            'input'       => json_encode(request()->all()),
            'session'     => json_encode(session()->all()),
            'server'      => json_encode($_SERVER),
            'code'        => $code,
        ];

        // If this is from the console echo the message
        if (app()->runningInConsole()) {
            console_print(
                'Error Message: ' . $exception->getMessage()
                . ' File: ' . $exception->getFile()
                . ' Line: ' . $exception->getLine(),
                'error'
            );
        } else {
            Log::info(json_encode($error));
        }
    }
}

if (!function_exists('generate_password')) {
    /**
     * Generate Random Password.
     *
     * @param int $length
     *
     * @return bool|string
     */
    function generate_password($length = 32)
    {
        $random = str_shuffle('abcdefghjklmnopqrstuvwxyzABCDEFGHJKLMNOPQRSTUVWXYZ234567890!$%^&!$%^&');

        return substr($random, 0, $length);
    }
}

if (!function_exists('make_key')) {
    /**
     * @param $input
     *
     * @return string
     */
    function make_key($input)
    {
        $input = str_replace('/', '_', $input);
        $input = Str::slug($input, '_');
        $input = trim($input);
        $input = strtolower($input);

        return $input;
    }
}

if (!function_exists('api_token')) {
    /**
     * Get the API token value for the logged in user.
     *
     * @return string
     */
    function api_token()
    {
        return user() ? user()->getApiToken() : null;
    }
}

if (!function_exists('encode_uri_component')) {
    /**
     * @param $str
     *
     * @return string
     */
    function encode_uri_component($str)
    {
        $revert = ['%21' => '!', '%2A' => '*', '%27' => "'", '%28' => '(', '%29' => ')'];

        return strtr(rawurlencode($str), $revert);
    }
}

if (!function_exists('cookie_enabled')) {
    /**
     * @param $name
     *
     * @return string
     */
    function cookie_enabled($name)
    {
        return isset($_COOKIE[$name]) && $_COOKIE[$name] === 'true';
    }
}
