# GitHub Copilot Instructions - Bongo Framework

## Project Overview

Bongo Framework is the core foundation package for the Bongo Laravel monorepo. It provides the base architecture that all other Bongo packages extend, including AbstractServiceProvider, AbstractModel, reusable traits, custom casts, enums, Schema.org generation, and helper classes.

**Tech Stack**: PHP 8.2+, Laravel 10+, Spatie Schema.org, Maatwebsite Excel

## Key Classes and Relationships

### Service Provider Hierarchy

```
Illuminate\Support\ServiceProvider
    └── AbstractServiceProvider (Bongo\Framework\Providers\AbstractServiceProvider)
        ├── FrameworkServiceProvider (Main entry point)
        ├── RouteServiceProvider
        ├── AuthServiceProvider
        └── EventServiceProvider
        └── [All other Bongo package service providers extend this]
```

**AbstractServiceProvider** is the most important class - it provides automatic bootstrapping for:
- Config registration from `src/Config/{module}.php`
- Route registration from `src/Routes/*.php`
- View registration from `src/Views/{module}/`
- Migration loading from `src/Migrations/`
- Translation loading from `src/Translations/`
- Middleware, commands, composers, listeners, subscribers via protected arrays

### Model Hierarchy

```
Illuminate\Database\Eloquent\Model
    └── AbstractModel (Bongo\Framework\Models\AbstractModel)
        ├── Uses: HasCreatedAt, HasCreatedBy
        ├── Uses: HasUpdatedAt, HasUpdatedBy
        ├── Uses: HasDeletedAt, HasDeletedBy
        ├── Uses: HasDiff
        └── Method: attributeExists(string $key): bool
```

### Controller Hierarchy

```
Illuminate\Routing\Controller
    └── AbstractController
        ├── Uses: AuthorizesRequests, ValidatesRequests
        ├── AbstractApiController (API responses)
        └── AbstractDatatableController (DataTables endpoints)
            └── Flow: setup() → applyFilters() → applySearchQuery() → runQuery()
```

### Schema.org Graph Generation

```
MetaSchema (Entry point)
    ├── graphForPage(Model): string
    ├── graphForPost(Model): string
    └── graphForProject(Model): string
        └── PageGraph / PostGraph / ProjectGraph
            └── Uses Concerns:
                ├── HandlesWebPage, HandlesBlogPost, HandlesArticle
                ├── HandlesWebSite, HandlesLocalBusiness
                ├── HandlesSocialLinks, HandlesReviews
                ├── HandlesContactPoint, HandlesPostalAddress
                └── HandlesOutput (renders to script tag)
```

## Code Templates

### Creating a Service Provider

```php
<?php

declare(strict_types=1);

namespace Bongo\MyPackage;

use Bongo\Framework\Providers\AbstractServiceProvider;
use Bongo\MyPackage\Http\Middleware\MyMiddleware;
use Bongo\MyPackage\Console\MyCommand;

class MyPackageServiceProvider extends AbstractServiceProvider
{
    protected string $module = 'mypackage'; // Must match config filename

    protected array $middlewares = [
        'myMiddleware' => MyMiddleware::class,
    ];

    protected array $commands = [
        MyCommand::class,
    ];

    protected array $listeners = [
        MyEvent::class => [
            MyEventListener::class,
        ],
    ];

    protected array $composers = [
        MyComposer::class => [
            'mypackage::backend.index',
        ],
    ];

    public function boot(): void
    {
        parent::boot(); // MUST call parent boot

        // Additional boot logic here
    }
}
```

### Creating a Model with Traits

```php
<?php

declare(strict_types=1);

namespace Bongo\MyPackage\Models;

use Bongo\Framework\Casts\Encoded;
use Bongo\Framework\Casts\Pence;
use Bongo\Framework\Enums\BooleanEnum;
use Bongo\Framework\Enums\StatusEnum;
use Bongo\Framework\Models\AbstractModel;
use Bongo\Framework\Traits\HasRecursive;
use Bongo\Framework\Traits\HasSeo;
use Bongo\Framework\Traits\HasStatus;
use Illuminate\Database\Eloquent\Relations\BelongsTo;

class MyModel extends AbstractModel
{
    use HasSeo;       // Auto-generates: slug, meta_title, meta_description
    use HasRecursive; // Adds: parent/child/sibling relationships
    use HasStatus;    // Adds: status field

    protected $table = 'my_models';

    protected $fillable = [
        'name',
        'slug',
        'content',
        'status',
        'is_featured',
        'price',
        'parent_id',
    ];

    protected $casts = [
        'content' => Encoded::class,      // HTML entity encode/decode
        'price' => Pence::class,          // Store pence, return pounds
        'is_featured' => BooleanEnum::class,
        'status' => StatusEnum::class,
    ];

    public function category(): BelongsTo
    {
        return $this->belongsTo(Category::class);
    }
}
```

### Creating a DataTable Controller

```php
<?php

declare(strict_types=1);

namespace Bongo\MyPackage\Http\Controllers\Backend;

use Bongo\Framework\Http\Controllers\AbstractDatatableController;
use Bongo\MyPackage\Models\MyModel;
use Illuminate\Database\Eloquent\Builder;

class MyModelDatatableController extends AbstractDatatableController
{
    protected function getBaseQuery(): Builder
    {
        return MyModel::query()
            ->with(['category', 'createdBy'])
            ->select('my_models.*');
    }

    protected function applyFilters(): void
    {
        parent::applyFilters(); // Applies status & type filters

        if ($categoryId = $this->request->get('category_id')) {
            $this->query->where('category_id', $categoryId);
        }

        if ($featured = $this->request->get('featured')) {
            $this->query->where('is_featured', $featured);
        }
    }

    protected function applySearchQuery(): void
    {
        if (isset($this->input['search']) && !empty($this->input['search'])) {
            $search = strtolower(trim($this->input['search']));
            $words = explode(' ', $search);

            foreach ($words as $word) {
                if (empty($word)) continue;

                $this->query->where(function ($q) use ($word) {
                    $q->where('name', 'LIKE', "%{$word}%")
                      ->orWhere('slug', 'LIKE', "%{$word}%");
                });
            }
        }
    }
}
```

### Creating a Standard Controller

```php
<?php

declare(strict_types=1);

namespace Bongo\MyPackage\Http\Controllers\Backend;

use Bongo\Framework\Http\Controllers\AbstractController;
use Bongo\MyPackage\Models\MyModel;
use Illuminate\Http\RedirectResponse;
use Illuminate\View\View;

class MyModelController extends AbstractController
{
    public function index(): View
    {
        return view('mypackage::backend.index');
    }

    public function create(): View
    {
        return view('mypackage::backend.create');
    }

    public function store(MyModelRequest $request): RedirectResponse
    {
        $model = MyModel::create($request->validated());

        return redirect()
            ->route('backend.mypackage.edit', $model)
            ->success('Model created successfully');
    }

    public function edit(MyModel $model): View
    {
        return view('mypackage::backend.edit', compact('model'));
    }

    public function update(MyModelRequest $request, MyModel $model): RedirectResponse
    {
        $model->update($request->validated());

        return redirect()
            ->route('backend.mypackage.edit', $model)
            ->success('Model updated successfully');
    }

    public function destroy(MyModel $model): RedirectResponse
    {
        $model->delete();

        return redirect()
            ->route('backend.mypackage.index')
            ->success('Model deleted successfully');
    }
}
```

### Creating Route Files

**backend.php** (Admin routes):
```php
<?php

use Bongo\MyPackage\Http\Controllers\Backend\MyModelController;
use Illuminate\Support\Facades\Route;

// Routes automatically prefixed with /admin
// Named with backend.* prefix
// Middleware: web, auth, employee, noIndex

Route::prefix('mypackage')->as('mypackage.')->group(function () {
    Route::resource('models', MyModelController::class);

    Route::get('datatable', [MyModelDatatableController::class, 'index'])
        ->name('datatable');
});
```

**frontend.php** (Public routes):
```php
<?php

use Bongo\MyPackage\Http\Controllers\Frontend\MyModelController;
use Illuminate\Support\Facades\Route;

// Routes named with frontend.* prefix
// Middleware: web

Route::prefix('models')->as('mypackage.')->group(function () {
    Route::get('/', [MyModelController::class, 'index'])->name('index');
    Route::get('{model:slug}', [MyModelController::class, 'show'])->name('show');
});
```

**api.php** (API routes):
```php
<?php

use Bongo\MyPackage\Http\Controllers\Api\MyModelController;
use Illuminate\Support\Facades\Route;

// Routes automatically prefixed with /api
// Named with api.* prefix
// Middleware: api, auth:sanctum, noIndex

Route::apiResource('models', MyModelController::class);
```

### Creating a Custom Cast

```php
<?php

declare(strict_types=1);

namespace Bongo\MyPackage\Casts;

use Illuminate\Contracts\Database\Eloquent\CastsAttributes;

class MyCast implements CastsAttributes
{
    public function get($model, string $key, $value, array $attributes): mixed
    {
        // Transform value when retrieving from database
        return $value ? unserialize($value) : null;
    }

    public function set($model, string $key, $value, array $attributes): mixed
    {
        // Transform value when storing to database
        return $value ? serialize($value) : null;
    }
}
```

### Creating an Enum

```php
<?php

declare(strict_types=1);

namespace Bongo\MyPackage\Enums;

use Bongo\Framework\Enums\ArrayInterface;
use Bongo\Framework\Enums\DefaultInterface;
use Bongo\Framework\Enums\WithArray;

enum MyEnum: int implements ArrayInterface, DefaultInterface
{
    use WithArray;

    case DRAFT = 0;
    case PUBLISHED = 1;
    case ARCHIVED = 2;

    public static function getDefault(): int
    {
        return self::DRAFT->value;
    }

    public function label(): string
    {
        return match($this) {
            self::DRAFT => 'Draft',
            self::PUBLISHED => 'Published',
            self::ARCHIVED => 'Archived',
        };
    }
}
```

### Creating a Model Trait

```php
<?php

declare(strict_types=1);

namespace Bongo\MyPackage\Traits;

use Illuminate\Database\Eloquent\Relations\BelongsTo;

trait HasAuthor
{
    public function initializeHasAuthor(): void
    {
        $this->mergeFillable(['author_id']);
    }

    public function author(): BelongsTo
    {
        return $this->belongsTo(User::class, 'author_id');
    }

    public function hasAuthor(): bool
    {
        $this->loadMissing('author');
        return !empty($this->author_id) && !is_null($this->author);
    }
}
```

### Using Schema.org Helper

In your blade template:
```blade
{{-- Page schema --}}
{!! SEO::getMetaSchemaForPage($page) !!}

{{-- Post schema --}}
{!! SEO::getMetaSchemaForPost($post) !!}

{{-- Project schema --}}
{!! SEO::getMetaSchemaForProject($project) !!}
```

In your controller:
```php
use Bongo\Framework\Helpers\SEO;

$schema = SEO::getMetaSchemaForPage($page);
```

### Using Helper Functions

```php
// Global helper functions
$user = user(); // Get authenticated user
$exists = route_exists('frontend.page.show'); // Check route exists
$matches = route_is('backend.*'); // Check route name pattern
$words = camel_words('myVariableName'); // "My Variable Name"
$key = make_key('My Page Title'); // "my-page-title"
$id = make_id('My Section', '123'); // "my-section-123"
$password = generate_password(32); // Random secure password
log_exception($exception, 500); // Log exception with code
$enabled = cookie_enabled('analytics'); // Check cookie consent
$plain = plain_text('<p>Hello</p>'); // Strip most tags
$encoded = encode_uri_component('hello world'); // URL encode
```

## Common Patterns

### Flash Messages

Controllers can chain flash message methods on redirects:
```php
return redirect()
    ->route('backend.mypackage.index')
    ->success('Operation completed successfully');

// Available methods: success(), error(), warning(), info(), danger()
```

### Recursive/Hierarchical Models

Use `HasRecursive` trait for parent/child relationships:
```php
$model->hasParent(); // Check if has parent
$model->parent; // Get parent (BelongsTo)
$model->nestedParents; // Get all ancestors recursively
$model->getAncestors(); // Collection of ancestors

$model->hasChildren(); // Check if has children
$model->children; // Get direct children (HasMany)
$model->nestedChildren; // Get all descendants recursively
$model->getDescendents(); // Collection of descendants

$model->hasSiblings(); // Check if has siblings
$model->siblings(); // Get siblings (same parent)
```

### SEO Auto-Generation

Use `HasSeo` trait to auto-generate SEO fields on save:
```php
$model->name = "My Page Title";
$model->save();
// Automatically sets:
// - slug: "my-page-title"
// - meta_title: "My Page Title" (75 char limit)
// - meta_description: excerpt from content (150 char limit)
```

### Audit Trail

All models extending `AbstractModel` automatically track:
- `created_at`, `created_by`
- `updated_at`, `updated_by`
- `deleted_at`, `deleted_by` (if using SoftDeletes)

Access via relationships:
```php
$model->createdBy; // User who created
$model->updatedBy; // User who last updated
$model->deletedBy; // User who deleted (soft delete)
```

## Configuration Access

```php
// Framework settings
config('settings.api_prefix'); // 'api'
config('settings.backend_prefix'); // 'admin'
config('settings.cache_default'); // 300 seconds
config('settings.minify_html_enabled'); // true/false

// Schema.org settings
config('schema.organization.name');
config('schema.organization.logo');

// Cloudflare settings
config('cloudflare.api_key');
```

## Important Rules

1. **Always extend AbstractServiceProvider** for new packages
2. **Always set `$module` property** - must match config filename and view namespace
3. **Never manually register routes** - AbstractServiceProvider handles this automatically
4. **Use `declare(strict_types=1);`** for all new PHP files
5. **Use type hints** on all method parameters and return types
6. **Extend AbstractModel** for audit trail functionality
7. **Use enums** instead of magic numbers or string constants
8. **Call `parent::boot()`** when overriding boot() in service providers
9. **Use framework casts** for common data transformations
10. **Use framework traits** for common model functionality
11. **Route files are auto-detected** from `src/Routes/` directory
12. **Config files are auto-merged** from `src/Config/{module}.php`
13. **Views are auto-namespaced** as `{module}::`
14. **Migrations are auto-loaded** from `src/Migrations/`

## Related Files

- **Core Provider**: `src/Providers/AbstractServiceProvider.php`
- **Main Provider**: `src/FrameworkServiceProvider.php`
- **Base Model**: `src/Models/AbstractModel.php`
- **Helpers**: `src/Helpers/*.php` and `src/helpers.php`
- **Traits**: `src/Traits/**/*.php` (37 traits)
- **Casts**: `src/Casts/*.php` (12 casts)
- **Enums**: `src/Enums/*.php` (5 enums)
- **Schema**: `src/Schema/*.php` and `src/Schema/Concerns/*.php`
