# Bongo Framework - Claude Code Guide

## Overview

**Bongo Framework** is the core foundation package for the Bongo Laravel monorepo. Every other Bongo package extends this framework. It provides the AbstractServiceProvider for automatic bootstrapping, AbstractModel with audit trails, reusable traits, custom casts, enums, Schema.org generation, and helper classes.

**Tech Stack**: PHP 8.2+, Laravel 10+, Spatie Schema.org, Maatwebsite Excel

**Package Name**: `bongo/framework`
**Namespace**: `Bongo\Framework`
**Repository**: https://bitbucket.org/designtec/framework

## Quick Links

- **Detailed Architecture**: See `ARCHITECTURE.md` for comprehensive diagrams and flow charts
- **Cursor AI Rules**: See `.cursorrules` for development guidelines
- **GitHub Copilot**: See `.github/copilot-instructions.md` for code templates

## Commands

From `composer.json`:

### Testing
```bash
vendor/bin/phpunit                  # Run all tests
```

### Code Quality
```bash
vendor/bin/pint                     # Fix code style issues
vendor/bin/pint --test              # Check code style without fixing
```

### Static Analysis
```bash
vendor/bin/phpstan analyse          # Run static analysis (if configured)
```

## Architecture Quick Reference

### Core Concepts

**This package is THE FOUNDATION** - all other Bongo packages depend on it and extend its classes.

**Key Classes**:
1. **`AbstractServiceProvider`** - Base class for ALL Bongo packages (provides automatic bootstrapping)
2. **`FrameworkServiceProvider`** - Main entry point that loads all packages
3. **`AbstractModel`** - Base model with audit trails (created_by, updated_by, deleted_by)
4. **`AbstractController`** - Base controller with Laravel auth/validation traits
5. **`AbstractDatatableController`** - Base for DataTables JSON endpoints

### Package Structure

```
src/
├── Casts/              # 12 custom Eloquent casts (Encoded, Pence, PlainText, etc.)
├── Config/             # 4 config files (settings, schema, cloudflare, developer)
├── Enums/              # 5 PHP 8.1+ enums (BooleanEnum, StatusEnum)
├── Helpers/            # 13 static helper classes (SEO, Str, Route, etc.)
├── Http/
│   ├── Controllers/    # 3 abstract base controllers
│   ├── Middleware/     # 11 middleware classes
│   └── ViewComposers/  # View composers
├── Models/             # AbstractModel base class
├── Providers/          # Service providers (including AbstractServiceProvider)
├── Schema/             # Schema.org graph generation (MetaSchema + concerns)
├── Traits/             # 37 reusable model traits
│   ├── Address/        # HasPostcode, HasLatitude, HasLongitude
│   ├── Audit/          # HasCreatedBy, HasUpdatedBy, etc.
│   └── Contact/        # HasEmail, HasPhone, HasFullName, etc.
├── Views/              # Blade templates (auth, backend, frontend, mail layouts)
└── FrameworkServiceProvider.php
```

### AbstractServiceProvider (MOST IMPORTANT)

**All Bongo packages MUST extend this class.**

**Automatic Features**:
- ✅ Auto-loads config from `src/Config/{module}.php`
- ✅ Auto-registers routes from `src/Routes/*.php`
- ✅ Auto-loads views from `src/Views/{module}/`
- ✅ Auto-loads migrations from `src/Migrations/`
- ✅ Auto-loads translations from `src/Translations/`
- ✅ Registers middleware, commands, composers, listeners, subscribers via protected arrays

**Route Patterns** (automatic):

| File | URL Prefix | Route Name | Middleware | Use Case |
|------|-----------|-----------|------------|----------|
| `api.php` | `/api` | `api.*` | `auth:sanctum` | Protected APIs |
| `backend.php` | `/admin` | `backend.*` | `auth, employee` | Admin panel |
| `frontend.php` | None | `frontend.*` | `web` | Public routes |
| `web.php` | None | None | `web` | Standard web |
| `custom.php` | None | None | None | Webhooks |

**Required Property**:
```php
protected string $module = 'mypackage'; // Must match config filename and view namespace
```

### Model Traits (37 Available)

**Audit Trails** (included in AbstractModel):
- `HasCreatedAt`, `HasCreatedBy` - Track who created
- `HasUpdatedAt`, `HasUpdatedBy` - Track who updated
- `HasDeletedAt`, `HasDeletedBy` - Track who deleted (soft delete)
- `HasDiff` - Track changes between saves

**Common Traits**:
- `HasSeo` - Auto-generates slug, meta_title, meta_description on save
- `HasRecursive` - Parent/child/sibling hierarchical relationships
- `HasStatus` - Status field with StatusEnum
- `HasVisible` - Visibility toggle
- `HasEmail`, `HasPhone`, `HasMobile` - Contact information
- `HasPostcode`, `HasLatitude`, `HasLongitude` - Address data

### Custom Casts (12 Available)

| Cast | Purpose | Example |
|------|---------|---------|
| `Encoded` | HTML entity encode/decode | `<p>` ↔ `&lt;p&gt;` |
| `Pence` | Store pence, return pounds | `1999` ↔ `19.99` |
| `PlainText` | Strip HTML tags | Stores plain text |
| `Date`, `DateTime`, `Time` | Date formatting | Various formats |

Usage:
```php
protected $casts = [
    'description' => Encoded::class,
    'price' => Pence::class,
    'status' => StatusEnum::class,
];
```

### Enums (5 Available)

**BooleanEnum** (int):
```php
case YES = 1;
case NO = 0;
```

**StatusEnum** (int):
```php
case ACTIVE = 1;
case DISABLED = 0;
```

Both have `toArray()` for dropdowns and `getDefault()` for defaults.

### Schema.org Helper

Generate SEO-friendly structured data:

```php
use Bongo\Framework\Helpers\SEO;

// In blade templates
{!! SEO::getMetaSchemaForPage($page) !!}
{!! SEO::getMetaSchemaForPost($post) !!}
{!! SEO::getMetaSchemaForProject($project) !!}
```

Generates JSON-LD script tags with WebPage, BlogPost, or Article schemas plus LocalBusiness, Organization, ContactPoint, and PostalAddress data.

### Helper Functions

Global functions from `src/helpers.php`:

```php
user()                              // Get authenticated user
route_exists('route.name')          // Check if route exists
route_is('backend.*')               // Check route name pattern
camel_words('myVariable')           // "My Variable"
make_key('My Page')                 // "my-page"
make_id('section', '123')           // "section-123"
generate_password(32)               // Random secure password
log_exception($e, 500)              // Log exception with code
cookie_enabled('analytics')         // Check cookie consent
plain_text('<p>Hello</p>')          // Strip tags
encode_uri_component('hello world') // URL encode
```

### Flash Messages

Controllers can chain flash methods on redirects:

```php
return redirect()
    ->route('backend.mypackage.index')
    ->success('Operation successful');

// Available: success(), error(), warning(), info(), danger()
```

## Key Files Reference

| File Path | Purpose | Key Methods/Properties |
|-----------|---------|----------------------|
| `src/Providers/AbstractServiceProvider.php` | Base for all Bongo packages | `boot()`, `$module`, `$middlewares`, `$commands` |
| `src/FrameworkServiceProvider.php` | Main framework entry point | `boot()`, `loadPackages()`, `loadCoreProviders()` |
| `src/Models/AbstractModel.php` | Base model with audit trails | `attributeExists()` |
| `src/Http/Controllers/AbstractController.php` | Base controller | Uses AuthorizesRequests, ValidatesRequests |
| `src/Http/Controllers/AbstractDatatableController.php` | DataTables endpoint base | `getBaseQuery()`, `applyFilters()`, `index()` |
| `src/Helpers/SEO.php` | Schema.org generation | `getMetaSchemaForPage/Post/Project()` |
| `src/Helpers/Str.php` | String utilities | `camelWords()`, `key()`, `id()` |
| `src/Traits/HasSeo.php` | Auto-generate SEO fields | `setSlug()`, `setMetaTitle()`, `setMetaDescription()` |
| `src/Traits/HasRecursive.php` | Hierarchical relationships | `parent()`, `children()`, `nestedChildren()`, `siblings()` |
| `src/Enums/BooleanEnum.php` | Yes/No enum | `YES = 1`, `NO = 0` |
| `src/Enums/StatusEnum.php` | Active/Disabled enum | `ACTIVE = 1`, `DISABLED = 0` |
| `src/Casts/Encoded.php` | HTML entity encoding | `get()`, `set()` |
| `src/Casts/Pence.php` | Pence/pounds conversion | `get()` ÷ 100, `set()` × 100 |
| `src/Config/settings.php` | Framework configuration | `api_prefix`, `backend_prefix`, `cache_default` |
| `src/Schema/MetaSchema.php` | Schema.org entry point | `graphForPage()`, `graphForPost()`, `graphForProject()` |
| `src/helpers.php` | Global helper functions | `user()`, `route_exists()`, `make_key()`, etc. |

## Configuration

**Main Config**: `config/settings.php`

```php
'api_prefix' => 'api',              // API URL prefix
'backend_prefix' => 'admin',        // Admin URL prefix
'cache_default' => 300,             // Cache TTL (seconds)
'memory_limit' => '2048M',          // PHP memory limit
'install_complete' => false,        // Installation flag
'minify_html_enabled' => false,     // HTML minification
```

**Schema Config**: `config/schema.php` - Organization details for Schema.org
**Cloudflare Config**: `config/cloudflare.php` - API integration settings
**Developer Config**: `config/developer.php` - Development settings

## Code Style

- **PHP 8.2+** with strict types: `declare(strict_types=1);`
- **Laravel Pint** for code style (Laravel preset)
- **Type hints** required on all method parameters and return types
- **Enums** preferred over constants for status/boolean fields
- **snake_case** for database columns, **camelCase** for PHP properties/methods

## Common Patterns

### Creating a Package Service Provider

```php
<?php

declare(strict_types=1);

namespace Bongo\MyPackage;

use Bongo\Framework\Providers\AbstractServiceProvider;

class MyPackageServiceProvider extends AbstractServiceProvider
{
    protected string $module = 'mypackage'; // REQUIRED

    protected array $middlewares = [
        'myMiddleware' => MyMiddleware::class,
    ];

    protected array $commands = [
        MyCommand::class,
    ];

    public function boot(): void
    {
        parent::boot(); // REQUIRED
    }
}
```

### Creating a Model

```php
<?php

declare(strict_types=1);

namespace Bongo\MyPackage\Models;

use Bongo\Framework\Models\AbstractModel;
use Bongo\Framework\Traits\HasSeo;
use Bongo\Framework\Traits\HasRecursive;
use Bongo\Framework\Enums\StatusEnum;

class MyModel extends AbstractModel
{
    use HasSeo;       // Auto slug/meta
    use HasRecursive; // Parent/child

    protected $casts = [
        'status' => StatusEnum::class,
    ];
}
```

### Creating a DataTable Controller

```php
<?php

declare(strict_types=1);

namespace Bongo\MyPackage\Http\Controllers\Backend;

use Bongo\Framework\Http\Controllers\AbstractDatatableController;

class MyDatatableController extends AbstractDatatableController
{
    protected function getBaseQuery()
    {
        return MyModel::query()->with('createdBy');
    }

    protected function applyFilters()
    {
        parent::applyFilters(); // status & type

        if ($category = $this->request->get('category')) {
            $this->query->where('category_id', $category);
        }
    }
}
```

### Creating Backend Routes

**File**: `src/Routes/backend.php`

```php
<?php

use Illuminate\Support\Facades\Route;

// Automatically prefixed with /admin
// Named with backend.* prefix
// Middleware: web, auth, employee, noIndex

Route::prefix('mypackage')->as('mypackage.')->group(function () {
    Route::resource('items', MyController::class);
});
```

### Using Recursive Relationships

```php
// Parent/child navigation
$item->parent;              // Get parent
$item->children;            // Get direct children
$item->nestedChildren;      // Get all descendants
$item->getAncestors();      // Collection of all parents
$item->getDescendents();    // Collection of all children
$item->siblings();          // Same parent

// Checks
$item->hasParent();
$item->hasChildren();
$item->isParent();          // parent_id is null
$item->isChild();           // parent_id is not null
```

## Important Notes

1. **AbstractServiceProvider** is the foundation - all packages extend it
2. **`$module` property** must match config filename and view namespace
3. **Routes auto-register** - never manually register routes
4. **Config auto-merges** from `Config/{module}.php`
5. **Views auto-namespace** as `{module}::`
6. **Migrations auto-load** from `Migrations/`
7. **Always call `parent::boot()`** when overriding in service providers
8. **Use AbstractModel** for automatic audit trails
9. **Use framework traits** for common functionality
10. **Use enums** instead of magic numbers/strings

## Package Dependencies

**Required**:
- `php: >=8.2`
- `illuminate/contracts: ^10.0`
- `spatie/schema-org: ^3.23`
- `maatwebsite/excel: ^3.1`

**Dev**:
- `laravel/pint: ^1.0`
- `orchestra/testbench: ^8.0`
- `phpunit/phpunit: ^10.0`

## Development Workflow

1. **Install dependencies**: `composer install`
2. **Run tests**: `vendor/bin/phpunit`
3. **Fix code style**: `vendor/bin/pint`
4. **Check style**: `vendor/bin/pint --test`

## Troubleshooting

**Routes not found?**
- Check `$module` property matches config file
- Ensure route files are in `src/Routes/`
- Verify `parent::boot()` is called

**Views not found?**
- Views must be in `src/Views/{module}/`
- Use `{module}::path.to.view` syntax

**Config not loading?**
- Config file must be `{module}.php`
- Must be in `src/Config/`

**Migrations not running?**
- Must be in `src/Migrations/`
- Check Laravel naming convention

## Global Aliases

Available everywhere (registered in FrameworkServiceProvider):

```php
BooleanEnum  // → Bongo\Framework\Enums\BooleanEnum
StatusEnum   // → Bongo\Framework\Enums\StatusEnum
SEO          // → Bongo\Framework\Helpers\SEO
```

## Related Documentation

- **Main Monorepo Guide**: `/CLAUDE.md` (repository root)
- **Architecture Details**: `ARCHITECTURE.md` (this package)
- **Cursor Rules**: `.cursorrules` (this package)
- **Copilot Instructions**: `.github/copilot-instructions.md` (this package)
