# Architecture Documentation - bongo/gallery

## Table of Contents
1. [Overview](#overview)
2. [Directory Structure](#directory-structure)
3. [Class Diagrams](#class-diagrams)
4. [Lifecycle Diagrams](#lifecycle-diagrams)
5. [Traits and Interfaces](#traits-and-interfaces)
6. [Extension Points](#extension-points)
7. [How to Add Features](#how-to-add-features)

## Overview

The **bongo/gallery** package provides gallery management functionality for the Bongo CMS. It enables administrators to create and manage image galleries through a comprehensive admin interface and RESTful API.

**Key Capabilities:**
- Gallery CRUD operations via admin panel
- Image upload, update, and deletion via API
- Image sorting and metadata management
- File system integration with automatic cleanup
- DataTable integration for gallery listing
- Event dispatching for gallery lifecycle
- Developer preview functionality

**Dependencies:**
- `bongo/framework` ^3.0 - Base framework functionality
- `bongo/image` ^3.0 - Image model and management
- Laravel 10+
- PHP 8.2+

## Directory Structure

```
src/
├── Config/
│   └── gallery.php                              # Package configuration
│
├── Events/
│   ├── GalleryDeleted.php                       # Dispatched on gallery deletion
│   └── GalleryUpdated.php                       # Dispatched on gallery update
│
├── Http/
│   ├── Controllers/
│   │   ├── Api/
│   │   │   └── GalleryController.php            # Image management API
│   │   │       • index(int $id): JsonResponse   # Get gallery images
│   │   │       • store(StoreImageRequest, int $id): JsonResponse  # Upload image
│   │   │       • update(UpdateImageRequest, int $id): JsonResponse  # Update image
│   │   │       • destroy(Request, int $id): JsonResponse  # Delete image
│   │   │
│   │   ├── Backend/
│   │   │   ├── GalleryController.php            # Admin CRUD operations
│   │   │   │   • index(): View                  # List galleries
│   │   │   │   • create(): RedirectResponse     # Create gallery
│   │   │   │   • show(Gallery): View            # Display gallery
│   │   │   │   • edit(Gallery): View            # Edit form
│   │   │   │   • update(UpdateGalleryRequest, Gallery): RedirectResponse
│   │   │   │   • destroy(Gallery): RedirectResponse
│   │   │   │
│   │   │   └── GalleryDatatableController.php   # DataTable endpoint
│   │   │       • getBaseQuery(): Builder        # Query for DataTable
│   │   │
│   │   └── Frontend/
│   │       └── GalleryController.php            # Frontend preview
│   │           • preview(string $uuid): View    # Developer preview
│   │
│   └── Requests/
│       └── UpdateGalleryRequest.php             # Validation rules
│           • rules(): array                     # name: required
│
├── Migrations/
│   └── 2021_01_01_000001_create_galleries_table.php  # Database schema
│
├── Models/
│   └── Gallery.php                              # Main model
│       • images(): MorphToMany                  # Polymorphic image relationship
│
├── Routes/
│   ├── api.php                                  # API routes (image management)
│   ├── backend.php                              # Admin routes (gallery CRUD)
│   └── frontend.php                             # Frontend routes (preview)
│
├── Seeders/
│   └── PackageSeeder.php                        # Package metadata seeder
│       • run(): void                            # Seeds package entry
│
├── Translations/
│   └── en/
│       └── backend.php                          # English translations
│
├── Views/
│   ├── backend/
│   │   ├── index.blade.php                      # Gallery list with DataTable
│   │   ├── create.blade.php                     # Create form (if used)
│   │   ├── edit.blade.php                       # Edit form
│   │   ├── show.blade.php                       # Gallery detail view
│   │   └── partials/
│   │       └── form/
│   │           └── details.blade.php            # Form fields partial
│   │
│   └── frontend/
│       ├── preview.blade.php                    # Gallery preview page
│       └── partials/
│           └── gallery.blade.php                # Gallery display partial
│
└── GalleryServiceProvider.php                   # Service provider
    • boot(): void                               # Register package resources
```

## Class Diagrams

### Model Hierarchy

```
AbstractModel (framework)
    │
    ├── Properties:
    │   • created_by: int
    │   • updated_by: int
    │   • deleted_by: int
    │   • created_at: timestamp
    │   • updated_at: timestamp
    │
    └── Gallery
        │
        ├── Constants:
        │   • PENDING = 'pending'
        │   • ACTIVE = 'active'
        │   • INACTIVE = 'inactive'
        │
        ├── Properties:
        │   • id: int (primary key)
        │   • uuid: string (unique)
        │   • name: string
        │   • key: string (unique, from HasKey)
        │   • status: enum (from HasStatus)
        │   • deleted_at: timestamp (from SoftDeletes)
        │
        ├── Traits:
        │   • SoftDeletes (Laravel)
        │   • HasKey (framework)
        │   • HasImages (image package)
        │   • HasStatus (framework)
        │   • HasUUID (framework)
        │
        ├── Implements:
        │   • Imageable (image package)
        │
        └── Relationships:
            • images(): MorphToMany<Image>
              ├── Via: imageables pivot table
              ├── Pivot: sort_order (int)
              └── Type: polymorphic (imageable_type, imageable_id)
```

### Controller Hierarchy

```
AbstractController (framework)
    │
    ├── Backend\GalleryController
    │   ├── index(): View
    │   ├── create(): RedirectResponse
    │   ├── show(Gallery): View
    │   ├── edit(Gallery): View
    │   ├── update(UpdateGalleryRequest, Gallery): RedirectResponse
    │   └── destroy(Gallery): RedirectResponse
    │
    └── Frontend\GalleryController
        └── preview(string $uuid): View

AbstractApiController (framework)
    │
    └── Api\GalleryController
        ├── index(int $id): JsonResponse
        ├── store(StoreImageRequest, int $id): JsonResponse
        ├── update(UpdateImageRequest, int $id): JsonResponse
        └── destroy(Request, int $id): JsonResponse

AbstractDatatableController (framework)
    │
    └── Backend\GalleryDatatableController
        └── getBaseQuery(): Builder
```

### Service Provider Hierarchy

```
AbstractServiceProvider (framework)
    │
    ├── Auto-registers:
    │   • Config files (src/Config/{module}.php)
    │   • Routes (src/Routes/*.php with middleware)
    │   • Views (src/Views/{module}/)
    │   • Migrations (src/Migrations/)
    │   • Translations (src/Translations/)
    │
    └── GalleryServiceProvider
        │
        ├── Properties:
        │   • $module = 'gallery'
        │
        └── boot(): void
            ├── parent::boot()  // Registers all resources
            └── AliasLoader::getInstance()->alias('Gallery', Gallery::class)
```

### Event Hierarchy

```
Event (Laravel)
    │
    ├── Traits:
    │   • Dispatchable
    │   • InteractsWithSockets
    │   • SerializesModels
    │
    ├── GalleryUpdated
    │   │
    │   ├── Properties:
    │   │   • $gallery: Gallery
    │   │
    │   └── __construct(Gallery $gallery)
    │
    └── GalleryDeleted
        │
        ├── Properties:
        │   • $gallery: Gallery
        │
        └── __construct(Gallery $gallery)
```

## Lifecycle Diagrams

### Gallery Creation Flow

```
User Request (GET /admin/galleries/create)
    │
    ├─> Backend\GalleryController::create()
    │       │
    │       ├─> Gallery::create()
    │       │       │
    │       │       ├─> Generate UUID (HasUUID trait)
    │       │       ├─> Set status = PENDING
    │       │       └─> Save to database
    │       │
    │       └─> Redirect to edit route with success message
    │
    └─> Response: Redirect to backend.gallery.edit
```

### Gallery Update Flow

```
User Request (POST /admin/galleries/{gallery}/update)
    │
    ├─> UpdateGalleryRequest::validate()
    │       │
    │       └─> Rules: ['name' => 'required']
    │
    ├─> Backend\GalleryController::update()
    │       │
    │       ├─> $gallery->update($request->all())
    │       │       │
    │       │       ├─> Update name field
    │       │       ├─> Update updated_by (AbstractModel)
    │       │       └─> Update updated_at
    │       │
    │       ├─> event(new GalleryUpdated($gallery))
    │       │       │
    │       │       └─> Dispatch to registered listeners
    │       │
    │       └─> Redirect with success flash message
    │
    └─> Response: Redirect to backend.gallery.show
```

### Gallery Deletion Flow

```
User Request (DELETE /admin/galleries/{gallery}/delete)
    │
    ├─> Backend\GalleryController::destroy()
    │       │
    │       ├─> $gallery->delete()
    │       │       │
    │       │       ├─> Set deleted_at timestamp (SoftDeletes)
    │       │       ├─> Set deleted_by (AbstractModel)
    │       │       └─> Images remain but relationship is soft deleted
    │       │
    │       ├─> event(new GalleryDeleted($gallery))
    │       │       │
    │       │       └─> Dispatch to registered listeners
    │       │
    │       ├─> Check redirect target
    │       │       │
    │       │       └─> If from show page, redirect to index
    │       │           Else redirect back
    │       │
    │       └─> Flash success message
    │
    └─> Response: Redirect with success/error message
```

### Image Upload Flow (API)

```
API Request (POST /api/galleries/{id}/store)
    │
    ├─> StoreImageRequest::validate()
    │       │
    │       └─> Validate file upload
    │
    ├─> Api\GalleryController::store()
    │       │
    │       ├─> $gallery = Gallery::with('images')->findOrFail($id)
    │       │
    │       ├─> Extract file information
    │       │       │
    │       │       ├─> $file = $request->file('file')
    │       │       ├─> $ext = $file->getClientOriginalExtension()
    │       │       ├─> $name = rtrim(filename, extension)
    │       │       └─> $size = $file->getSize()
    │       │
    │       ├─> Generate unique filename
    │       │       │
    │       │       └─> File::generateName($name, $ext)
    │       │
    │       ├─> Upload file to storage
    │       │       │
    │       │       ├─> $filePath = config('image.public_path').'uploads/'
    │       │       └─> $file->storePubliclyAs($filePath, $fileName)
    │       │
    │       ├─> Create Image model
    │       │       │
    │       │       ├─> Set name, title, path, ext, size, type
    │       │       ├─> Set created_by, updated_by (current user)
    │       │       ├─> Get image dimensions via getimagesize()
    │       │       ├─> Set width, height, orientation
    │       │       └─> $image->save()
    │       │
    │       ├─> Attach to gallery
    │       │       │
    │       │       └─> $gallery->images()->save($image, [
    │       │           'sort_order' => max(sort_order) + 1
    │       │           ])
    │       │
    │       └─> Return JSON response with image
    │
    └─> Response: JSON { item: Image, status: 200 }
```

### Image Deletion Flow (API)

```
API Request (POST /api/galleries/{id}/delete)
    │
    ├─> Api\GalleryController::destroy()
    │       │
    │       ├─> $gallery = Gallery::with('images')->findOrFail($id)
    │       │
    │       ├─> $image = $gallery->images()->findOrFail($request->id)
    │       │
    │       ├─> Delete physical file
    │       │       │
    │       │       ├─> if ($image->fileExists())
    │       │       └─>     $image->deleteFile()
    │       │
    │       ├─> Delete pivot relationship
    │       │       │
    │       │       └─> DB::table('imageables')
    │       │           ->where('image_id', $image->id)
    │       │           ->delete()
    │       │
    │       ├─> Delete image model
    │       │       │
    │       │       └─> $image->delete()
    │       │
    │       └─> Return JSON response
    │
    └─> Response: JSON { item: Image|null, message: string }
```

### Image Rename Flow (API)

```
API Request (POST /api/galleries/{id}/update) with name_formatted
    │
    ├─> Api\GalleryController::update()
    │       │
    │       ├─> Get current and new filenames
    │       │       │
    │       │       ├─> $currentFileName = File::nameWithoutSuffix($image->getFileName())
    │       │       └─> $requestFileName = File::nameWithoutSuffix($request->name_formatted)
    │       │
    │       ├─> If filenames differ and file exists
    │       │       │
    │       │       ├─> Generate new filename
    │       │       │       │
    │       │       │       └─> File::generateName($requestFileName, $ext)
    │       │       │
    │       │       ├─> Update content in database tables
    │       │       │       │
    │       │       │       ├─> $tables = ['pages', 'posts', 'post_categories',
    │       │       │       │              'projects', 'project_categories']
    │       │       │       │
    │       │       │       └─> foreach ($tables as $table)
    │       │       │               DB::table($table)
    │       │       │                   ->where('content', 'LIKE', "%/photos/{oldFile}%")
    │       │       │                   ->update(['content' => REPLACE(...)])
    │       │       │
    │       │       ├─> Rename physical file
    │       │       │       │
    │       │       │       └─> $image->renameFile($newFileName)
    │       │       │
    │       │       └─> Update image model
    │       │               │
    │       │               └─> $image->name = $newFileName
    │       │
    │       ├─> Update title if provided
    │       │       │
    │       │       └─> $image->title = $request->title
    │       │
    │       ├─> Update sort_order if provided
    │       │       │
    │       │       └─> $image->pivot->sort_order = $request->sort_order
    │       │
    │       ├─> Save changes
    │       │
    │       └─> Return JSON response
    │
    └─> Response: JSON { item: Image, message: string }
```

### DataTable Request Flow

```
AJAX Request (GET /admin/galleries/datatable)
    │
    ├─> Backend\GalleryDatatableController::index()
    │       │
    │       ├─> getBaseQuery(): Builder
    │       │       │
    │       │       └─> $this->gallery->newQuery()
    │       │
    │       ├─> Apply DataTable filters/sorting/pagination
    │       │   (handled by AbstractDatatableController)
    │       │
    │       └─> Return JSON response
    │
    └─> Response: JSON { data: Gallery[], recordsTotal: int, ... }
```

## Traits and Interfaces

### Model Traits

| Trait | Source | Purpose | Added Properties | Added Methods |
|-------|--------|---------|------------------|---------------|
| `SoftDeletes` | Laravel | Soft delete functionality | `deleted_at: timestamp` | `delete()`, `restore()`, `forceDelete()`, `trashed()` |
| `HasKey` | bongo/framework | Unique key field | `key: string` | Key generation and management |
| `HasImages` | bongo/image | Polymorphic images | - | `images(): MorphToMany` |
| `HasStatus` | bongo/framework | Status management | `status: enum` | `isActive()`, `isPending()`, etc. |
| `HasUUID` | bongo/framework | UUID generation | `uuid: string` | UUID auto-generation on create |

### Model Interfaces

| Interface | Source | Purpose | Required Methods |
|-----------|--------|---------|-----------------|
| `Imageable` | bongo/image | Contract for image relationships | `images(): MorphToMany` |

### Event Traits

| Trait | Source | Purpose |
|-------|--------|---------|
| `Dispatchable` | Laravel | Enable event dispatching via `event()` |
| `InteractsWithSockets` | Laravel | Broadcasting support |
| `SerializesModels` | Laravel | Queue serialization support |

## Extension Points

### 1. Adding Custom Fields to Gallery

**Migration:**
```php
Schema::table('galleries', function (Blueprint $table) {
    $table->string('description')->nullable();
    $table->string('category')->nullable();
});
```

**Model:**
```php
protected $fillable = [
    'name',
    'status',
    'description',  // Add new field
    'category',     // Add new field
];
```

**Validation:**
```php
public function rules(): array
{
    return [
        'name' => 'required',
        'description' => 'nullable|string|max:500',
        'category' => 'nullable|string',
    ];
}
```

**Views:**
Update form partial and show view to include new fields.

### 2. Adding Event Listeners

**Create Listener:**
```php
namespace Bongo\Gallery\Listeners;

use Bongo\Gallery\Events\GalleryUpdated;

class NotifyGalleryUpdate
{
    public function handle(GalleryUpdated $event): void
    {
        // Handle the event
        $gallery = $event->gallery;
        // Send notification, log, etc.
    }
}
```

**Register in Service Provider:**
```php
protected array $listeners = [
    GalleryUpdated::class => [
        NotifyGalleryUpdate::class,
    ],
];
```

### 3. Adding Custom Routes

Add routes to appropriate route file:

**Backend Route:**
```php
// src/Routes/backend.php
Route::get('{gallery}/custom-action', [GalleryController::class, 'customAction'])
    ->name('custom-action');
```

**API Route:**
```php
// src/Routes/api.php
Route::post('{id}/custom-endpoint', [GalleryController::class, 'customEndpoint'])
    ->name('custom-endpoint');
```

### 4. Adding Model Relationships

```php
// In Gallery model
public function category(): BelongsTo
{
    return $this->belongsTo(Category::class);
}

public function tags(): MorphToMany
{
    return $this->morphToMany(Tag::class, 'taggable');
}
```

### 5. Adding Model Scopes

```php
// In Gallery model
public function scopeActive($query)
{
    return $query->where('status', self::ACTIVE);
}

public function scopeByCategory($query, string $category)
{
    return $query->where('category', $category);
}

// Usage:
Gallery::active()->get();
Gallery::byCategory('nature')->get();
```

### 6. Adding Custom View Composers

**Create Composer:**
```php
namespace Bongo\Gallery\Http\ViewComposers;

use Illuminate\View\View;

class GalleryComposer
{
    public function compose(View $view): void
    {
        $view->with('galleryCount', Gallery::count());
    }
}
```

**Register in Service Provider:**
```php
protected array $composers = [
    'gallery::backend.*' => GalleryComposer::class,
];
```

### 7. Adding Middleware

**Create Middleware:**
```php
namespace Bongo\Gallery\Http\Middleware;

class CheckGalleryAccess
{
    public function handle($request, Closure $next)
    {
        // Check access logic
        return $next($request);
    }
}
```

**Register in Service Provider:**
```php
protected array $middlewares = [
    'gallery.access' => CheckGalleryAccess::class,
];
```

**Use in Routes:**
```php
Route::get('/', [GalleryController::class, 'index'])
    ->middleware('gallery.access')
    ->name('index');
```

## How to Add Features

### Feature: Add Gallery Categories

**Step 1: Create Migration**
```php
Schema::create('gallery_categories', function (Blueprint $table) {
    $table->increments('id');
    $table->string('name');
    $table->string('slug')->unique();
    $table->timestamps();
});

Schema::table('galleries', function (Blueprint $table) {
    $table->unsignedInteger('category_id')->nullable()->after('name');
    $table->foreign('category_id')->references('id')->on('gallery_categories');
});
```

**Step 2: Create Category Model**
```php
namespace Bongo\Gallery\Models;

class GalleryCategory extends AbstractModel
{
    protected $fillable = ['name', 'slug'];

    public function galleries(): HasMany
    {
        return $this->hasMany(Gallery::class, 'category_id');
    }
}
```

**Step 3: Add Relationship to Gallery**
```php
// In Gallery model
public function category(): BelongsTo
{
    return $this->belongsTo(GalleryCategory::class, 'category_id');
}
```

**Step 4: Update Form Request**
```php
public function rules(): array
{
    return [
        'name' => 'required',
        'category_id' => 'nullable|exists:gallery_categories,id',
    ];
}
```

**Step 5: Update Views**
Add category dropdown to form partial and display in show view.

**Step 6: Add Routes (if needed)**
```php
Route::prefix('categories')
    ->as('categories.')
    ->group(function () {
        Route::get('/', [CategoryController::class, 'index'])->name('index');
        Route::post('/', [CategoryController::class, 'store'])->name('store');
    });
```

### Feature: Add Gallery Export API

**Step 1: Create Export Controller Method**
```php
// In Api\GalleryController
public function export(int $id): JsonResponse
{
    $gallery = Gallery::with('images')->findOrFail($id);

    $data = [
        'name' => $gallery->name,
        'uuid' => $gallery->uuid,
        'images' => $gallery->images->map(function ($image) {
            return [
                'title' => $image->title,
                'url' => $image->getUrl(),
                'dimensions' => [
                    'width' => $image->width,
                    'height' => $image->height,
                ],
            ];
        }),
    ];

    return response()->json($data);
}
```

**Step 2: Add Route**
```php
// In src/Routes/api.php
Route::get('{id}/export', [GalleryController::class, 'export'])
    ->name('export');
```

### Feature: Add Bulk Operations

**Step 1: Create Bulk Controller**
```php
namespace Bongo\Gallery\Http\Controllers\Backend;

class GalleryBulkController extends AbstractController
{
    public function activate(Request $request): JsonResponse
    {
        $ids = $request->input('ids', []);
        Gallery::whereIn('id', $ids)->update(['status' => Gallery::ACTIVE]);

        return response()->json([
            'message' => trans('gallery::backend.bulk_activate_success'),
        ]);
    }

    public function delete(Request $request): JsonResponse
    {
        $ids = $request->input('ids', []);
        Gallery::whereIn('id', $ids)->delete();

        return response()->json([
            'message' => trans('gallery::backend.bulk_delete_success'),
        ]);
    }
}
```

**Step 2: Add Routes**
```php
Route::prefix('bulk')
    ->as('bulk.')
    ->group(function () {
        Route::post('activate', [GalleryBulkController::class, 'activate'])->name('activate');
        Route::post('delete', [GalleryBulkController::class, 'delete'])->name('delete');
    });
```

**Step 3: Update DataTable View**
Add checkboxes and bulk action buttons to the index view.

## Database Schema Reference

### galleries Table

```sql
CREATE TABLE `galleries` (
    `id` INT UNSIGNED AUTO_INCREMENT PRIMARY KEY,
    `uuid` CHAR(36) NOT NULL,
    `name` VARCHAR(255) NULL,
    `key` VARCHAR(255) NULL,
    `status` ENUM('pending', 'active', 'inactive') DEFAULT 'pending',
    `created_by` INT UNSIGNED NULL,
    `updated_by` INT UNSIGNED NULL,
    `deleted_by` INT UNSIGNED NULL,
    `created_at` TIMESTAMP NULL,
    `updated_at` TIMESTAMP NULL,
    `deleted_at` TIMESTAMP NULL,
    INDEX `galleries_uuid_index` (`uuid`),
    INDEX `galleries_key_index` (`key`),
    INDEX `galleries_created_by_index` (`created_by`),
    INDEX `galleries_updated_by_index` (`updated_by`),
    INDEX `galleries_deleted_by_index` (`deleted_by`)
);
```

### imageables Pivot Table (from bongo/image)

```sql
CREATE TABLE `imageables` (
    `id` INT UNSIGNED AUTO_INCREMENT PRIMARY KEY,
    `image_id` INT UNSIGNED NOT NULL,
    `imageable_id` INT UNSIGNED NOT NULL,
    `imageable_type` VARCHAR(255) NOT NULL,
    `sort_order` INT DEFAULT 0,
    `created_at` TIMESTAMP NULL,
    `updated_at` TIMESTAMP NULL,
    FOREIGN KEY (`image_id`) REFERENCES `images` (`id`) ON DELETE CASCADE,
    INDEX `imageables_imageable_type_imageable_id_index` (`imageable_type`, `imageable_id`)
);
```

For `Gallery` model:
- `imageable_type` = `'Bongo\Gallery\Models\Gallery'`
- `imageable_id` = Gallery ID
- `sort_order` = Position in gallery

## Testing Strategy

### Unit Tests
- Model relationships
- Model scopes
- Trait functionality

### Feature Tests
- Gallery CRUD operations
- Image upload/update/delete
- Permission checks
- Event dispatching
- File system operations

### Example Test Structure
```php
tests/
├── Unit/
│   ├── Models/
│   │   └── GalleryTest.php
│   └── ...
└── Feature/
    ├── Backend/
    │   └── GalleryControllerTest.php
    ├── Api/
    │   └── GalleryControllerTest.php
    └── ...
```
