# Gallery Package

A Laravel package that provides gallery management functionality for the Bongo CMS. Create and manage custom image galleries with a comprehensive admin interface and RESTful API.

## Features

- **Gallery CRUD Operations** - Create, read, update, and delete galleries through an admin panel
- **Image Upload & Management** - Upload images via API with automatic dimension detection and file handling
- **Image Sorting** - Maintain custom sort order for images within galleries
- **File Operations** - Rename and delete image files with automatic content updates
- **DataTables Integration** - Server-side DataTable support for gallery listings
- **Event System** - Dispatch events on gallery updates and deletions
- **Developer Preview** - Preview galleries with authentication
- **Soft Deletes** - Galleries can be soft deleted and restored
- **Status Management** - Pending/Active/Inactive workflow for galleries

## Requirements

- **PHP:** 8.2 or higher
- **Laravel:** 10.x or higher
- **Dependencies:**
  - `bongo/framework` ^3.0
  - `bongo/image` ^3.0

## Installation

### Composer

Install the package via Composer:

```bash
composer require bongo/gallery
```

### Configuration

#### Laravel 11+

The service provider is auto-discovered. No manual registration required.

#### Laravel 10

Add the service provider to your `config/app.php` if auto-discovery is disabled:

```php
'providers' => [
    // ...
    Bongo\Gallery\GalleryServiceProvider::class,
],
```

### Publish Configuration (Optional)

To customise the configuration:

```bash
php artisan vendor:publish --provider="Bongo\Gallery\GalleryServiceProvider"
```

This publishes:
- Configuration file to `config/gallery.php`
- Migrations to `database/migrations/`
- Views to `resources/views/vendor/gallery/`

### Run Migrations

```bash
php artisan migrate
```

This creates the `galleries` table with the following structure:
- Gallery metadata (id, uuid, name, key, status)
- Audit fields (created_by, updated_by, deleted_by)
- Timestamps and soft deletes

### Seed Package (Optional)

If using the Bongo package management system:

```bash
php artisan db:seed --class="Bongo\Gallery\Seeders\PackageSeeder"
```

## Usage

### Basic Gallery Operations

#### Creating a Gallery

```php
use Bongo\Gallery\Models\Gallery;

// Create via admin (redirects to edit page)
// GET /admin/galleries/create

// Or programmatically
$gallery = Gallery::create([
    'name' => 'My Gallery',
    'status' => Gallery::ACTIVE,
]);
```

#### Retrieving Galleries

```php
// Get all active galleries
$galleries = Gallery::where('status', Gallery::ACTIVE)->get();

// Get a gallery with images
$gallery = Gallery::with('images')->find($id);

// Get gallery by UUID
$gallery = Gallery::where('uuid', $uuid)->first();

// Get gallery by key
$gallery = Gallery::where('key', 'my-gallery')->first();
```

#### Updating a Gallery

```php
$gallery->update([
    'name' => 'Updated Gallery Name',
    'status' => Gallery::ACTIVE,
]);

// Event GalleryUpdated is automatically dispatched
```

#### Deleting a Gallery

```php
// Soft delete
$gallery->delete();

// Event GalleryDeleted is automatically dispatched

// Force delete
$gallery->forceDelete();

// Restore
$gallery->restore();
```

### Working with Images

#### Upload Image (via API)

```
POST /api/galleries/{id}/store

Parameters:
- file: Image file (required)

Returns:
{
    "item": {
        "id": 1,
        "name": "image.jpg",
        "title": "image.jpg",
        "path": "/uploads/",
        "width": 1920,
        "height": 1080,
        "orientation": "landscape"
    }
}
```

#### Get Gallery Images (via API)

```
GET /api/galleries/{id}

Returns:
{
    "items": [
        {
            "id": 1,
            "name": "image.jpg",
            "title": "My Image",
            "sort_order": 1,
            ...
        }
    ]
}
```

#### Update Image (via API)

```
POST /api/galleries/{id}/update

Parameters:
- id: Image ID (required)
- title: Image title (optional)
- name_formatted: New filename without extension (optional)
- sort_order: Display order (optional)

Returns:
{
    "item": { ... },
    "message": "Image successfully updated!"
}
```

#### Delete Image (via API)

```
POST /api/galleries/{id}/delete

Parameters:
- id: Image ID (required)

Returns:
{
    "item": { ... },
    "message": "Image successfully deleted!"
}
```

### Gallery Status

Galleries have three status states:

```php
Gallery::PENDING   // 'pending' - Not yet published
Gallery::ACTIVE    // 'active' - Published and visible
Gallery::INACTIVE  // 'inactive' - Hidden
```

Check status:

```php
if ($gallery->status === Gallery::ACTIVE) {
    // Gallery is active
}
```

### Events

The package dispatches events for integration:

```php
use Bongo\Gallery\Events\GalleryUpdated;
use Bongo\Gallery\Events\GalleryDeleted;

// Listen to events
Event::listen(GalleryUpdated::class, function ($event) {
    $gallery = $event->gallery;
    // Handle gallery update
});

Event::listen(GalleryDeleted::class, function ($event) {
    $gallery = $event->gallery;
    // Handle gallery deletion
});
```

### Displaying Galleries in Views

```blade
@foreach($gallery->images as $image)
    <img src="{{ $image->path . $image->name }}"
         alt="{{ $image->title }}"
         width="{{ $image->width }}"
         height="{{ $image->height }}">
@endforeach
```

Or use the included partial:

```blade
@include('gallery::frontend.partials.gallery', [
    'gallery' => $gallery,
    'columns' => '4',
    'padding' => 'medium',
    'lazy' => true,
    'lightbox' => true,
])
```

## Configuration

The package configuration is in `config/gallery.php`:

```php
return [
    'prefix' => 'galleries',  // URL prefix for routes
];
```

Available configuration options:

- `prefix` - URL prefix for all gallery routes (default: 'galleries')

## Routes

### Backend Routes (Admin Panel)

**Prefix:** `/admin/galleries`
**Middleware:** `auth`, `employee`

| Method | URI | Name | Description |
|--------|-----|------|-------------|
| GET | `/admin/galleries` | `backend.gallery.index` | List all galleries |
| GET | `/admin/galleries/create` | `backend.gallery.create` | Create new gallery |
| GET | `/admin/galleries/datatable` | `backend.gallery.datatable` | DataTable JSON |
| GET | `/admin/galleries/{gallery}` | `backend.gallery.show` | View gallery |
| GET | `/admin/galleries/{gallery}/edit` | `backend.gallery.edit` | Edit gallery |
| POST | `/admin/galleries/{gallery}/update` | `backend.gallery.update` | Update gallery |
| DELETE | `/admin/galleries/{gallery}/delete` | `backend.gallery.destroy` | Delete gallery |

### API Routes (Image Management)

**Prefix:** `/api/galleries`
**Middleware:** `auth:sanctum`

| Method | URI | Name | Description |
|--------|-----|------|-------------|
| GET | `/api/galleries/{id}` | `api.gallery.index` | Get gallery images |
| POST | `/api/galleries/{id}/store` | `api.gallery.store` | Upload image |
| POST | `/api/galleries/{id}/update` | `api.gallery.update` | Update image |
| POST | `/api/galleries/{id}/delete` | `api.gallery.destroy` | Delete image |

### Frontend Routes

**Prefix:** `/galleries`

| Method | URI | Name | Description |
|--------|-----|------|-------------|
| GET | `/galleries/preview/{uuid}` | `frontend.gallery.preview` | Preview gallery (requires auth + developer) |

## Model Traits

The `Gallery` model uses the following traits:

| Trait | Source | Purpose |
|-------|--------|---------|
| `SoftDeletes` | Laravel | Soft delete functionality |
| `HasKey` | bongo/framework | Unique key field for programmatic access |
| `HasImages` | bongo/image | Polymorphic image relationships |
| `HasStatus` | bongo/framework | Status field management |
| `HasUUID` | bongo/framework | UUID generation for public URLs |

## API Integration

### Authentication

API routes require Sanctum authentication. Include the bearer token in requests:

```bash
curl -X GET "https://yourdomain.com/api/galleries/1" \
     -H "Authorization: Bearer YOUR_TOKEN_HERE" \
     -H "Accept: application/json"
```

### Upload Image Example

```bash
curl -X POST "https://yourdomain.com/api/galleries/1/store" \
     -H "Authorization: Bearer YOUR_TOKEN_HERE" \
     -H "Content-Type: multipart/form-data" \
     -F "file=@/path/to/image.jpg"
```

### Update Image Example

```bash
curl -X POST "https://yourdomain.com/api/galleries/1/update" \
     -H "Authorization: Bearer YOUR_TOKEN_HERE" \
     -H "Content-Type: application/json" \
     -d '{
         "id": 123,
         "title": "Updated Title",
         "sort_order": 2
     }'
```

## Testing

Run the test suite:

```bash
vendor/bin/phpunit
```

Run with coverage:

```bash
vendor/bin/phpunit --coverage-html coverage
```

## Code Style

The package follows Laravel coding standards with PSR-12.

Check code style:

```bash
vendor/bin/pint --test
```

Fix code style issues:

```bash
vendor/bin/pint
```

## Development

### Project Structure

```
src/
├── Config/              # Configuration files
├── Events/              # Gallery events
├── Http/
│   ├── Controllers/     # Backend, API, and Frontend controllers
│   └── Requests/        # Form validation
├── Migrations/          # Database migrations
├── Models/              # Gallery model
├── Routes/              # Route definitions
├── Seeders/             # Database seeders
├── Translations/        # Translation files
└── Views/               # Blade templates
```

### Adding Custom Fields

1. Create a migration to add the column
2. Add the field to the `$fillable` array in the `Gallery` model
3. Update the `UpdateGalleryRequest` validation rules
4. Update the views to display/edit the new field

### Extending the Package

See [ARCHITECTURE.md](ARCHITECTURE.md) for detailed extension points and examples.

## Documentation

- **[ARCHITECTURE.md](ARCHITECTURE.md)** - Detailed architecture documentation with class diagrams and lifecycle flows
- **[CLAUDE.md](CLAUDE.md)** - Quick reference guide for Claude Code
- **[.cursorrules](.cursorrules)** - Cursor AI instructions
- **[.github/copilot-instructions.md](.github/copilot-instructions.md)** - GitHub Copilot instructions

## Changelog

Please see [CHANGELOG.md](CHANGELOG.md) for recent changes.

## Contributing

Contributions are welcome! Please follow these guidelines:

1. Fork the repository
2. Create a feature branch
3. Make your changes
4. Run tests and code style checks
5. Submit a pull request

## Security

If you discover any security-related issues, please email stuart.elliott@bespokeuk.com instead of using the issue tracker.

## Credits

- **Stuart Elliott** - [Bespoke UK](https://bespokeuk.com)
- All contributors

## License

The MIT License (MIT). Please see [LICENSE](LICENSE) for more information.

## Support

For support, please contact:
- **Email:** stuart.elliott@bespokeuk.com
- **Website:** https://bespokeuk.com
- **Repository:** https://bitbucket.org/designtec/gallery
