<?php

namespace Bongo\Gallery\Http\Controllers\Api;

use Bongo\Framework\Helpers\File;
use Bongo\Framework\Http\Controllers\AbstractApiController;
use Bongo\Gallery\Models\Gallery;
use Bongo\Image\Http\Requests\StoreImageRequest;
use Bongo\Image\Http\Requests\UpdateImageRequest;
use Bongo\Image\Models\Image;
use Illuminate\Http\JsonResponse;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Str;

class GalleryController extends AbstractApiController
{
    public function index(int $id): JsonResponse
    {
        $gallery = Gallery::with('images')->findOrFail($id);

        return response()->json([
            'items' => $gallery->images,
        ]);
    }

    public function store(StoreImageRequest $request, int $id): JsonResponse
    {
        $gallery = Gallery::with('images')->findOrFail($id);

        // Get the file, its size and extension
        $file = $request->file('file');
        $ext = $file->getClientOriginalExtension();
        $name = rtrim($file->getClientOriginalName(), $ext);
        $size = $file->getSize();

        // Create a unique filename
        $fileName = File::generateName($name, $ext);

        // Get the upload path
        $filePath = config('image.public_path').'uploads/';

        // Upload the file and save
        if ($file->storePubliclyAs($filePath, $fileName)) {

            // Create the image
            $image = new Image();
            $image->name = $fileName;
            $image->title = $fileName;
            $image->path = '/'.$filePath;
            $image->ext = $ext;
            $image->size = $size;
            $image->type = Image::UPLOAD;
            $image->created_by = user() ? user()->id : null;
            $image->updated_by = user() ? user()->id : null;

            // Set the dimensions
            if (file_exists($file->path())) {
                [$width, $height] = getimagesize($file->path());
                $image->width = $width;
                $image->height = $height;
                $image->orientation = ($width > $height) ? Image::LANDSCAPE : Image::PORTRAIT;
            }

            // Save the image & entity relationship
            $image->save();
            $gallery->images()->save($image, [
                'sort_order' => $gallery->images()->max('sort_order') + 1,
            ]);

            return response()->json(['item' => $image], 200);
        }

        return response()->json('error', 400);
    }

    public function update(UpdateImageRequest $request, int $id): JsonResponse
    {
        $gallery = Gallery::with('images')->findOrFail($id);
        $image = $gallery->images()->findOrFail($request->id);

        $currentFileName = File::nameWithoutSuffix($image->getFileName());
        $requestFileName = File::nameWithoutSuffix($request->get('name_formatted'));

        if (Str::slug($requestFileName) !== Str::slug($currentFileName) && $image->fileExists()) {

            // Generate a new file name
            $ext = $image->getFileExtension();
            $newFileName = File::generateName($requestFileName, $ext);

            // Update the existing pages, posts, projects etc
            $tables = ['pages', 'posts', 'post_categories', 'projects', 'project_categories'];
            foreach ($tables as $table) {
                DB::table($table)
                    ->where('content', 'LIKE', "%/photos/{$image->getFileName()}%")
                    ->update([
                        'content' => DB::raw("REPLACE(`content`, '/photos/{$image->getFileName()}', '/photos/{$newFileName}')"),
                    ]);
            }

            // Rename and update the image
            $image->renameFile($newFileName);
            $image->name = $newFileName;

            // Update the existing pages, posts, projects etc
            DB::raw("UPDATE pages SET content = REPLACE(content, $currentFileName, $newFileName");
            DB::raw("UPDATE posts SET content = REPLACE(content, $currentFileName, $newFileName");
            DB::raw("UPDATE projects SET content = REPLACE(content, $currentFileName, $newFileName");
        }

        $image->title = $request->get('title');
        $image->save();

        if ($request->has('sort_order')) {
            $image->pivot->sort_order = $request->get('sort_order');
            $image->pivot->save();
        }

        return response()->json([
            'item' => $image,
            'message' => trans('image::image.update_success'),
        ]);
    }

    public function destroy(Request $request, int $id): JsonResponse
    {
        $gallery = Gallery::with('images')->findOrFail($id);
        $image = $gallery->images()->findOrFail($request->id);

        // If the file exists in storage then delete it
        if ($image->fileExists()) {
            $image->deleteFile();
        }

        // Delete model relationship and then the image
        DB::table('imageables')->where('image_id', $image->id)->delete();
        $image->delete();

        return response()->json([
            'item' => $image ?? null,
            'message' => trans('image::image.delete_success'),
        ]);
    }
}
