# GitHub Copilot Instructions - bongo/gallery

## Project Overview

**bongo/gallery** is a Laravel package that provides gallery management functionality for the Bongo CMS. It allows administrators to create custom image galleries and manage images through a RESTful API.

**Key Information:**
- **Namespace:** `Bongo\Gallery`
- **PHP:** 8.2+
- **Laravel:** 10+
- **Dependencies:** `bongo/framework` (base), `bongo/image` (image handling)

## Core Classes and Relationships

### Models

**Gallery** (`src/Models/Gallery.php`)
```php
class Gallery extends AbstractModel implements Imageable
{
    use SoftDeletes, HasKey, HasImages, HasStatus, HasUUID;

    // Status constants
    public const PENDING = 'pending';
    public const ACTIVE = 'active';
    public const INACTIVE = 'inactive';

    // Fillable fields
    protected $fillable = ['name', 'status'];

    // Relationships
    public function images(): MorphToMany  // From HasImages trait
}
```

**Relationships:**
- `images()` → Polymorphic many-to-many with `Bongo\Image\Models\Image` via `imageables` pivot (includes `sort_order`)

### Controllers

**Backend\GalleryController** - Admin CRUD operations
- `index()` → List galleries (uses DataTable)
- `create()` → Create empty gallery, redirect to edit
- `show(Gallery)` → Display gallery details
- `edit(Gallery)` → Display edit form
- `update(UpdateGalleryRequest, Gallery)` → Update gallery, dispatch event
- `destroy(Gallery)` → Soft delete gallery, dispatch event

**Api\GalleryController** - Image management API
- `index(int $id)` → Get gallery images as JSON
- `store(StoreImageRequest, int $id)` → Upload image to gallery
- `update(UpdateImageRequest, int $id)` → Update image title/sort order
- `destroy(Request, int $id)` → Delete image and file

**Backend\GalleryDatatableController** - DataTable JSON endpoint
- Extends `AbstractDatatableController`
- Provides server-side DataTable data

**Frontend\GalleryController** - Preview functionality
- `preview(string $uuid)` → Developer-only gallery preview

### Events

**GalleryUpdated** - Dispatched when gallery is updated
```php
event(new GalleryUpdated($gallery));
```

**GalleryDeleted** - Dispatched when gallery is deleted
```php
event(new GalleryDeleted($gallery));
```

### Form Requests

**UpdateGalleryRequest** - Validates gallery updates
```php
public function rules(): array
{
    return ['name' => 'required'];
}
```

## Code Style Templates

### Controller Method (Backend)

```php
public function methodName(Gallery $gallery): View|RedirectResponse
{
    // Business logic
    $gallery->update($data);

    // Dispatch event
    event(new GalleryUpdated($gallery));

    // Redirect with flash message
    return redirect()
        ->route('backend.gallery.show', $gallery->id)
        ->success(trans('gallery::backend.update_success'));
}
```

### API Controller Method

```php
public function methodName(Request $request, int $id): JsonResponse
{
    $gallery = Gallery::with('images')->findOrFail($id);

    // Business logic

    return response()->json([
        'item' => $item,
        'message' => trans('gallery::backend.success'),
    ]);
}
```

### Event Class

```php
namespace Bongo\Gallery\Events;

use Bongo\Gallery\Models\Gallery;
use Illuminate\Broadcasting\InteractsWithSockets;
use Illuminate\Foundation\Events\Dispatchable;
use Illuminate\Queue\SerializesModels;

class EventName
{
    use Dispatchable, InteractsWithSockets, SerializesModels;

    public Gallery $gallery;

    public function __construct(Gallery $gallery)
    {
        $this->gallery = $gallery;
    }
}
```

### Migration

```php
Schema::create('galleries', function (Blueprint $table) {
    // Primary
    $table->increments('id');
    $table->uuid('uuid')->index();

    // Fields
    $table->string('field_name')->nullable();
    $table->enum('status', [Gallery::PENDING, Gallery::ACTIVE, Gallery::INACTIVE])
        ->default(Gallery::PENDING);

    // Audit
    $table->unsignedInteger('created_by')->nullable()->index();
    $table->unsignedInteger('updated_by')->nullable()->index();
    $table->unsignedInteger('deleted_by')->nullable()->index();

    // Timestamps
    $table->timestamps();
    $table->softDeletes();
});
```

### Route Definition

```php
// backend.php
Route::as('gallery.')
    ->prefix(config('gallery.prefix'))
    ->group(function () {
        Route::get('/', [GalleryController::class, 'index'])->name('index');
        Route::get('{gallery}', [GalleryController::class, 'show'])->name('show');
    });
```

## Common Patterns

### Image Upload Pattern (API)

```php
public function store(StoreImageRequest $request, int $id): JsonResponse
{
    $gallery = Gallery::with('images')->findOrFail($id);
    $file = $request->file('file');

    // Generate unique filename
    $fileName = File::generateName($name, $ext);

    // Upload file
    $filePath = config('image.public_path').'uploads/';
    $file->storePubliclyAs($filePath, $fileName);

    // Create image model
    $image = new Image();
    $image->name = $fileName;
    $image->path = '/'.$filePath;
    $image->ext = $ext;
    $image->size = $size;
    $image->type = Image::UPLOAD;

    // Get dimensions
    if (file_exists($file->path())) {
        [$width, $height] = getimagesize($file->path());
        $image->width = $width;
        $image->height = $height;
        $image->orientation = ($width > $height) ? Image::LANDSCAPE : Image::PORTRAIT;
    }

    // Save and attach to gallery
    $image->save();
    $gallery->images()->save($image, [
        'sort_order' => $gallery->images()->max('sort_order') + 1,
    ]);

    return response()->json(['item' => $image], 200);
}
```

### Image Deletion Pattern

```php
public function destroy(Request $request, int $id): JsonResponse
{
    $gallery = Gallery::with('images')->findOrFail($id);
    $image = $gallery->images()->findOrFail($request->id);

    // Delete file from storage
    if ($image->fileExists()) {
        $image->deleteFile();
    }

    // Delete pivot relationship
    DB::table('imageables')->where('image_id', $image->id)->delete();

    // Delete image model
    $image->delete();

    return response()->json([
        'item' => $image ?? null,
        'message' => trans('image::image.delete_success'),
    ]);
}
```

### File Rename Pattern

```php
// When renaming an image file, update content in multiple tables
$tables = ['pages', 'posts', 'post_categories', 'projects', 'project_categories'];
foreach ($tables as $table) {
    DB::table($table)
        ->where('content', 'LIKE', "%/photos/{$oldFileName}%")
        ->update([
            'content' => DB::raw("REPLACE(`content`, '/photos/{$oldFileName}', '/photos/{$newFileName}')"),
        ]);
}

// Rename file and update image
$image->renameFile($newFileName);
$image->name = $newFileName;
$image->save();
```

### Flash Message Pattern

```php
// Success
return redirect()
    ->route('backend.gallery.show', $gallery->id)
    ->success(trans('gallery::backend.update_success'));

// Error
return back()->error(trans('gallery::backend.delete_failed'));
```

### Translation Pattern

```php
// Translation keys structure: {module}::{context}.{key}
trans('gallery::backend.store_success')
trans('gallery::backend.delete_failed')

// Translation file: src/Translations/en/backend.php
return [
    'store_success' => 'Gallery successfully created!',
    'delete_failed' => 'Unable to delete gallery!',
];
```

## Route Structure

### Backend Routes (`backend.php`)
- Prefix: `config('gallery.prefix')` (default: 'galleries')
- Middleware: `auth`, `employee` (auto-applied)
- Name prefix: `backend.gallery.*`

```
GET    /admin/galleries              → backend.gallery.index
GET    /admin/galleries/create       → backend.gallery.create
GET    /admin/galleries/datatable    → backend.gallery.datatable
GET    /admin/galleries/{gallery}    → backend.gallery.show
GET    /admin/galleries/{gallery}/edit → backend.gallery.edit
POST   /admin/galleries/{gallery}/update → backend.gallery.update
DELETE /admin/galleries/{gallery}/delete → backend.gallery.destroy
```

### API Routes (`api.php`)
- Prefix: `api.` + `config('gallery.prefix')`
- Middleware: `auth:sanctum` (auto-applied)
- Name prefix: `api.gallery.*`

```
GET  /api/galleries/{id}         → api.gallery.index
ANY  /api/galleries/{id}/store   → api.gallery.store
POST /api/galleries/{id}/update  → api.gallery.update
POST /api/galleries/{id}/delete  → api.gallery.destroy
```

### Frontend Routes (`frontend.php`)
- Prefix: `config('gallery.prefix')`
- Name prefix: `frontend.gallery.*`

```
GET /galleries/preview/{uuid} → frontend.gallery.preview (auth, developer)
```

## View Structure

### Blade Templates

**Backend Views:**
- `gallery::backend.index` - Gallery list with DataTable
- `gallery::backend.create` - Create form (if exists)
- `gallery::backend.show` - Gallery detail view
- `gallery::backend.edit` - Edit form
- `gallery::backend.partials.form.details` - Form partial

**Frontend Views:**
- `gallery::frontend.preview` - Gallery preview
- `gallery::frontend.partials.gallery` - Gallery display partial

### View Usage

```php
return view('gallery::backend.show', compact('gallery'));
return view('gallery::frontend.preview')->with([
    'gallery' => $gallery,
    'columns' => '4',
    'padding' => 'medium',
    'lazy' => true,
    'lightbox' => true,
]);
```

## Testing Patterns

### Feature Test Template

```php
namespace Bongo\Gallery\Tests\Feature;

use Bongo\Gallery\Models\Gallery;
use Tests\TestCase;

class GalleryControllerTest extends TestCase
{
    public function test_can_create_gallery(): void
    {
        $this->actingAs($user);

        $response = $this->post(route('backend.gallery.create'));

        $response->assertRedirect();
        $this->assertDatabaseHas('galleries', [
            'status' => Gallery::PENDING,
        ]);
    }
}
```

## Configuration

**File:** `src/Config/gallery.php`

```php
return [
    'prefix' => 'galleries',  // URL prefix for routes
];
```

**Access:** `config('gallery.prefix')`

## Important Traits and Interfaces

### From bongo/framework:
- `HasKey` - Adds unique `key` field for programmatic access
- `HasStatus` - Adds `status` field with helper methods
- `HasUUID` - Generates and manages UUID field

### From bongo/image:
- `Imageable` (interface) - Contract for models with images
- `HasImages` (trait) - Polymorphic `images()` relationship

## Database Conventions

- **Table naming:** Plural snake_case (`galleries`)
- **Primary key:** `id` (auto-incrementing)
- **UUID:** `uuid` field for public-facing URLs
- **Key:** `key` field for programmatic access
- **Status:** `status` enum field
- **Audit fields:** `created_by`, `updated_by`, `deleted_by`
- **Timestamps:** `created_at`, `updated_at`, `deleted_at`
- **Soft deletes:** Always use `SoftDeletes` trait

## Helper Functions

### From bongo/framework:

```php
// Get current authenticated user
user()  // Returns User model or null

// File operations
File::generateName($name, $ext)  // Generate unique filename
File::nameWithoutSuffix($filename)  // Strip extension from filename
```

### From bongo/image:

```php
// Image model methods
$image->getFileName()  // Get filename with extension
$image->getFileExtension()  // Get file extension
$image->fileExists()  // Check if physical file exists
$image->deleteFile()  // Delete physical file
$image->renameFile($newName)  // Rename physical file
```

## Key Features

1. **Gallery Management** - CRUD operations for galleries
2. **Image Upload** - API endpoint for uploading images with automatic processing
3. **Image Sorting** - Maintain sort order within galleries
4. **File Management** - Rename, delete image files with content updates
5. **DataTables** - Server-side DataTable integration
6. **Events** - Dispatch events on gallery updates/deletions
7. **Soft Deletes** - Galleries are soft deleted
8. **Developer Preview** - Preview galleries with authentication

## Security Considerations

- Preview route requires `auth` and `developer` middleware
- All API routes require `auth:sanctum` authentication
- Backend routes require `auth` and `employee` middleware
- Form requests validate all user input
- File uploads validate via `StoreImageRequest`
- SQL injection protection via query builder and Eloquent
