# CLAUDE.md - bongo/gallery

This package provides gallery management functionality for the Bongo CMS, allowing administrators to create custom image galleries and manage images via a RESTful API.

## Quick Links

- **Detailed Architecture:** See [ARCHITECTURE.md](ARCHITECTURE.md)
- **Cursor AI Instructions:** See [.cursorrules](.cursorrules)
- **GitHub Copilot Instructions:** See [.github/copilot-instructions.md](.github/copilot-instructions.md)
- **Repository:** https://bitbucket.org/designtec/gallery

## Package Information

**Name:** bongo/gallery
**Namespace:** `Bongo\Gallery`
**PHP:** 8.2+
**Laravel:** 10+

**Dependencies:**
- `bongo/framework` ^3.0 - Base framework functionality
- `bongo/image` ^3.0 - Image model and management

## Available Commands

From the package root directory:

```bash
# Run tests
vendor/bin/phpunit

# Check code style
vendor/bin/pint --test

# Fix code style
vendor/bin/pint

# Install dependencies
composer install

# Update dependencies
rm -f composer.lock && composer update -W
```

## Architecture Quick Reference

### Service Provider Pattern

This package extends `Bongo\Framework\Providers\AbstractServiceProvider`, which automatically registers:

- **Config:** `src/Config/gallery.php` → `config('gallery.*')`
- **Routes:** Auto-registered from `src/Routes/` with middleware:
  - `api.php` → `api.gallery.*` (auth:sanctum middleware)
  - `backend.php` → `backend.gallery.*` (auth + employee middleware)
  - `frontend.php` → `frontend.gallery.*` (custom middleware per route)
- **Views:** `src/Views/gallery/` → `view('gallery::*')`
- **Migrations:** `src/Migrations/`
- **Translations:** `src/Translations/` → `trans('gallery::*')`

The service provider only needs to define the module name:

```php
class GalleryServiceProvider extends AbstractServiceProvider
{
    protected string $module = 'gallery';
}
```

### Model Architecture

The `Gallery` model uses these traits from the framework:

| Trait | Purpose | Added Fields | Key Methods |
|-------|---------|--------------|-------------|
| `HasKey` | Unique key field | `key` | Key generation |
| `HasStatus` | Status management | `status` | `isActive()`, `isPending()` |
| `HasUUID` | UUID generation | `uuid` | Auto-generates on create |
| `HasImages` | Image relationships | - | `images(): MorphToMany` |
| `SoftDeletes` | Soft deletes | `deleted_at` | `delete()`, `restore()` |

**Status Constants:**
```php
Gallery::PENDING   // 'pending'
Gallery::ACTIVE    // 'active'
Gallery::INACTIVE  // 'inactive'
```

**Relationships:**
```php
$gallery->images()  // MorphToMany<Image> with sort_order in pivot
```

### Controllers

**Three controller types:**

1. **Backend\GalleryController** - Admin CRUD (routes: `backend.gallery.*`)
2. **Api\GalleryController** - Image management API (routes: `api.gallery.*`)
3. **Backend\GalleryDatatableController** - DataTable JSON endpoint

### Routes

**Backend Routes** (prefix: `/admin/galleries`):
```
GET    /admin/galleries              → backend.gallery.index
GET    /admin/galleries/create       → backend.gallery.create
GET    /admin/galleries/datatable    → backend.gallery.datatable
GET    /admin/galleries/{gallery}    → backend.gallery.show
GET    /admin/galleries/{gallery}/edit → backend.gallery.edit
POST   /admin/galleries/{gallery}/update → backend.gallery.update
DELETE /admin/galleries/{gallery}/delete → backend.gallery.destroy
```

**API Routes** (prefix: `/api/galleries`):
```
GET  /api/galleries/{id}          → api.gallery.index (get images)
ANY  /api/galleries/{id}/store    → api.gallery.store (upload image)
POST /api/galleries/{id}/update   → api.gallery.update (update image)
POST /api/galleries/{id}/delete   → api.gallery.destroy (delete image)
```

**Frontend Routes**:
```
GET /galleries/preview/{uuid} → frontend.gallery.preview (auth + developer)
```

### Events

**GalleryUpdated** - Dispatched after gallery update
**GalleryDeleted** - Dispatched after gallery deletion

```php
event(new GalleryUpdated($gallery));
event(new GalleryDeleted($gallery));
```

## Key Files Reference

| File | Purpose |
|------|---------|
| `src/GalleryServiceProvider.php` | Service provider with module registration |
| `src/Models/Gallery.php` | Main model with traits and relationships |
| `src/Http/Controllers/Backend/GalleryController.php` | Admin CRUD operations |
| `src/Http/Controllers/Api/GalleryController.php` | Image upload/update/delete API |
| `src/Http/Controllers/Backend/GalleryDatatableController.php` | DataTable endpoint |
| `src/Http/Requests/UpdateGalleryRequest.php` | Form validation |
| `src/Events/GalleryUpdated.php` | Update event |
| `src/Events/GalleryDeleted.php` | Delete event |
| `src/Config/gallery.php` | Configuration (route prefix) |
| `src/Migrations/2021_01_01_000001_create_galleries_table.php` | Database schema |
| `src/Routes/backend.php` | Admin routes |
| `src/Routes/api.php` | API routes for image management |
| `src/Routes/frontend.php` | Frontend preview route |

## Database Schema

**galleries table:**
- `id` (int) - Primary key
- `uuid` (string) - Unique identifier for public URLs
- `name` (string) - Gallery name
- `key` (string) - Unique key for programmatic access
- `status` (enum) - pending/active/inactive
- Audit fields: `created_by`, `updated_by`, `deleted_by`
- Timestamps: `created_at`, `updated_at`, `deleted_at`

**imageables pivot table** (polymorphic):
- `image_id` → Foreign key to images
- `imageable_id` → Foreign key to galleries
- `imageable_type` → 'Bongo\Gallery\Models\Gallery'
- `sort_order` → Image position in gallery

## Code Style Guidelines

### Type Declarations
- Always use `declare(strict_types=1);` at the top of PHP files
- Always specify return types on methods
- Always type-hint method parameters
- Always type-hint class properties

### Controller Patterns

**Backend Controller Method:**
```php
public function update(UpdateGalleryRequest $request, Gallery $gallery): RedirectResponse
{
    $gallery->update($request->all());
    event(new GalleryUpdated($gallery));

    return redirect()
        ->route('backend.gallery.show', $gallery->id)
        ->success(trans('gallery::backend.update_success'));
}
```

**API Controller Method:**
```php
public function index(int $id): JsonResponse
{
    $gallery = Gallery::with('images')->findOrFail($id);

    return response()->json([
        'items' => $gallery->images,
    ]);
}
```

### Flash Messages
Use framework helper methods:
```php
->success(trans('gallery::backend.update_success'))
->error(trans('gallery::backend.delete_failed'))
```

### Translation Keys
Format: `{module}::{context}.{key}`
```php
trans('gallery::backend.store_success')
```

## Common Workflows

### Adding a New Field

1. Create migration to add column to `galleries` table
2. Add field to `$fillable` array in `Gallery` model
3. Add validation rule to `UpdateGalleryRequest`
4. Update views (form partial and show view)
5. Run migration: `php artisan migrate`

### Creating a New Controller Method

1. Add method to appropriate controller (Backend/Api/Frontend)
2. Add route to corresponding route file (`backend.php`, `api.php`, `frontend.php`)
3. Add translation keys to `src/Translations/en/backend.php` if needed
4. Create/update view if needed
5. Test the endpoint

### Adding Event Listeners

1. Create listener class in `src/Listeners/`
2. Register in service provider:
```php
protected array $listeners = [
    GalleryUpdated::class => [
        YourListener::class,
    ],
];
```

## Integration Points

### From bongo/framework
- `AbstractServiceProvider` - Base provider with auto-registration
- `AbstractController` - Base controller with helper methods
- `AbstractApiController` - Base API controller
- `AbstractDatatableController` - Base DataTable controller
- `AbstractModel` - Base model with audit fields
- `HasKey`, `HasStatus`, `HasUUID` - Model traits
- `File::generateName()` - Unique filename generation

### From bongo/image
- `Imageable` interface - Contract for image relationships
- `HasImages` trait - Polymorphic image relationships
- `Image` model - Image entity with file handling
- `StoreImageRequest` - Image upload validation
- `UpdateImageRequest` - Image update validation

## Configuration

File: `src/Config/gallery.php`

```php
return [
    'prefix' => 'galleries',  // URL prefix for all routes
];
```

Access: `config('gallery.prefix')`

## Key Features

1. **Gallery CRUD** - Create, read, update, delete galleries via admin panel
2. **Image Upload** - API endpoint for uploading images with automatic processing
3. **Image Management** - Update titles, reorder images within galleries
4. **Image Deletion** - Delete images with automatic file cleanup
5. **File Renaming** - Rename image files with content replacement across tables
6. **DataTables** - Server-side DataTable support for gallery listing
7. **Preview** - Developer-only preview functionality
8. **Soft Deletes** - Galleries can be soft deleted and restored
9. **Events** - GalleryUpdated and GalleryDeleted events
10. **Status Management** - Pending/Active/Inactive workflow

## Testing

Run tests from package directory:
```bash
vendor/bin/phpunit
```

Test structure:
```
tests/
├── Unit/
│   └── .gitkeep
└── Feature/
    └── .gitkeep
```

## Important Notes

- The API controller handles **image operations** (upload/update/delete), not gallery CRUD
- Backend controller handles **gallery CRUD** operations
- Image file renaming updates content in `pages`, `posts`, `projects`, `post_categories`, and `project_categories` tables
- Preview route requires both `auth` and `developer` middleware
- Images are stored in `config('image.public_path').'uploads/'`
- The `Gallery` model is aliased as a facade in the service provider
- All routes use the configurable prefix from `gallery.prefix` config

## Getting Help

- **Architecture Details:** See [ARCHITECTURE.md](ARCHITECTURE.md) for detailed class diagrams and lifecycle flows
- **IDE Integration:** See [.cursorrules](.cursorrules) for Cursor AI instructions
- **Code Patterns:** See [.github/copilot-instructions.md](.github/copilot-instructions.md) for common patterns and templates
