# GitHub Copilot Instructions - Bongo Image Package

## Project Overview

Laravel package providing image management with uploads, on-the-fly manipulation, caching, and polymorphic relationships. Uses Intervention Image for manipulation and extends `bongo/framework` for service provider bootstrapping.

**Tech Stack:**
- PHP 8.2+ / Laravel 10+
- Intervention Image v2.6
- Bongo Framework v3.0

## Key Classes and Relationships

### Core Models

```php
// Image - Main image model with file operations
class Image extends AbstractModel
{
    // Types
    const AVATAR = 'avatar';
    const COVER_IMAGE = 'cover_image';
    const UPLOAD = 'upload';
    const WYSIWYG = 'wysiwyg';

    // Orientations
    const LANDSCAPE = 'landscape';
    const PORTRAIT = 'portrait';

    // Key methods
    public function getFileSrc(): string;
    public function getFileUrl(): string;
    public function getCachedFile(array $args): string;
    public function getCachedFileName(array $args): string;
}

// Imageable - Polymorphic pivot model
class Imageable extends Model
{
    // Pivot columns: imageable_id, imageable_type, image_id, sort_order
}
```

### Trait for Models

```php
trait HasImages
{
    public function images(): MorphToMany;
    public function primaryImage(): ?Image;
    public function secondaryImage(): ?Image;
    public function getPrimaryImage(array $args): string;
    public function getPlaceholder(array $args): string;
}

// Usage in any model:
class Post extends Model
{
    use HasImages;
}
```

### Service Classes

```php
// ImageManipulator - Intervention Image wrapper
class ImageManipulator
{
    public function setImage(string $fileData): self;
    public function resize(): self;
    public function crop(): self;
    public function fit(): self;
    public function setWidth(int $width): self;
    public function setHeight(int $height): self;
    public function setQuality(int $quality): self;
    public function stream(): StreamInterface;
}

// AvatarImage - Single avatar upload (replaces existing)
class AvatarImage
{
    public function __construct(Authenticatable $user, UploadedFile $file);
    public function save(): Authenticatable;
}

// CoverImage - Single cover image upload (replaces existing)
class CoverImage
{
    public function __construct(Imageable $entity, UploadedFile $file);
    public function save(): Imageable;
}

// WysiwygImage - Multi-image upload for WYSIWYG
class WysiwygImage
{
    public function __construct(Imageable $entity, UploadedFile $file);
    public function save(): Model;
}

// BuilderService - Extract base64 images from HTML
class BuilderService
{
    public function __construct($model, $imagePath);
    public function process(): ?string;
}

// ImagePlaceholder - Generate default placeholder images
class ImagePlaceholder
{
    public function __construct(array $args);
    public function url(): string;
    public function get(): string;
}
```

### Controllers

```php
// Frontend\ImageController - Public image delivery
public function show(Image $image): Response
// Handles: ?preset=large&w=800&h=600&q=80&mode=resize

// Backend\ImageController - Admin uploads
public function store(): JsonResponse;  // Dropzone upload
public function copy(): JsonResponse;   // Base64 paste
public function update(UpdateImageRequest $request, Image $image): JsonResponse;
public function destroy(Image $image): JsonResponse;

// Api\ImageController - API endpoints
// Same methods with auth:sanctum middleware
```

## Code Style Templates

### Creating a New Image Service

```php
namespace Bongo\Image\Services;

use Bongo\Framework\Helpers\File;
use Bongo\Image\Interfaces\Imageable;
use Bongo\Image\Models\Image;
use Illuminate\Http\UploadedFile;
use Illuminate\Support\Str;

class {ServiceName}Image
{
    protected UploadedFile $file;
    protected Imageable $entity;

    public function __construct(Imageable $entity, UploadedFile $file)
    {
        $this->entity = $entity;
        $this->file = $file;
    }

    public function save(): Imageable
    {
        // Delete existing if single-image type
        $this->delete();

        // Generate filename
        $ext = $this->file->getClientOriginalExtension();
        $name = rtrim($this->file->getClientOriginalName(), $ext);
        $size = $this->file->getSize();
        $fileName = File::generateName($name, $ext);

        // Store file
        $this->file->storePubliclyAs($this->getEntityPath(), $fileName);

        // Create database record
        $dbImage = new Image();
        $dbImage->name = $fileName;
        $dbImage->title = $this->entity->name ?? null;
        $dbImage->path = $this->getEntityPath();
        $dbImage->size = $size;
        $dbImage->ext = $ext;
        $dbImage->type = Image::NEW_TYPE; // Define new constant
        $dbImage->created_by = user()?->id;
        $dbImage->updated_by = user()?->id;

        // Set dimensions
        [$width, $height] = getimagesize($dbImage->getFileSrc());
        $dbImage->width = $width;
        $dbImage->height = $height;
        $dbImage->orientation = ($width > $height) ? Image::LANDSCAPE : Image::PORTRAIT;

        // Save and return
        $this->entity->images()->save($dbImage);

        return $this->entity;
    }

    protected function delete(): void
    {
        $images = $this->entity->images()->where('type', Image::NEW_TYPE)->get();

        if ($images) {
            foreach ($images as $image) {
                if ($image->fileExists()) {
                    $image->deleteFile();
                }
                $image->forceDelete();
            }
        }
    }

    private function getEntityPath(): string
    {
        $entityName = explode('\\', get_class($this->entity));
        $entityName = array_pop($entityName);
        $entityName = Str::snake($entityName);
        $entityName = Str::plural($entityName);

        return config('image.public_path') . "{$entityName}/{$this->entity->id}/";
    }
}
```

### Using HasImages Trait in Models

```php
namespace App\Models;

use Bongo\Image\Traits\HasImages;
use Bongo\Image\Models\Image;
use Illuminate\Database\Eloquent\Model;

class Post extends Model
{
    use HasImages;

    // Access images
    public function displayImage(): string
    {
        return $this->getPrimaryImage(['preset' => 'large']);
    }

    // Custom image query
    public function galleryImages()
    {
        return $this->images()
            ->where('type', Image::UPLOAD)
            ->orderBy('sort_order');
    }

    // Check for images
    public function hasGallery(): bool
    {
        return $this->images()->where('type', Image::UPLOAD)->exists();
    }
}
```

### Image Manipulation in Controllers

```php
public function show(Image $image)
{
    $args = request()->query();

    // Validate parameters
    if (isset($args['preset']) && !config("image.presets.{$args['preset']}")) {
        throw new Exception("Preset {$args['preset']} not found");
    }

    // Get cached file path
    $cachedFile = $image->getCachedFile($args);

    // Generate if not cached
    if (!Storage::exists($cachedFile)) {
        $manipulator = new ImageManipulator();
        $manipulator->setImage($image->getFileData());

        // Set dimensions
        if (isset($args['preset'])) {
            $manipulator->setWidth(config("image.presets.{$args['preset']}"));
        } elseif (isset($args['w'])) {
            $manipulator->setWidth($args['w']);
        }

        // Set quality
        if (isset($args['q'])) {
            $manipulator->setQuality($args['q']);
        }

        // Apply manipulation
        if ($args['mode'] === 'crop') {
            $manipulator->crop();
        } else {
            $manipulator->resize();
        }

        // Cache result
        Storage::put($cachedFile, $manipulator->stream());
    }

    // Return with cache headers
    return response()->stream(
        fn() => echo Storage::get($cachedFile),
        200,
        [
            'Content-Type' => $image->getMimeType(),
            'Cache-Control' => 'max-age=' . config('image.browser_cache'),
        ]
    );
}
```

### Uploading Images in Controllers

```php
// Single-image upload (avatar/cover)
public function uploadAvatar(Request $request)
{
    $request->validate([
        'avatar' => 'required|image|max:5120',
    ]);

    $avatar = new AvatarImage(auth()->user(), $request->file('avatar'));
    $user = $avatar->save();

    return response()->json([
        'url' => $user->getPrimaryImage(['preset' => 'thumb']),
    ]);
}

// Multi-image upload (WYSIWYG)
public function uploadWysiwyg(Request $request, Post $post)
{
    $request->validate([
        'file' => 'required|image|max:5120',
    ]);

    $wysiwyg = new WysiwygImage($post, $request->file('file'));
    $image = $wysiwyg->save();

    return response()->json([
        'url' => route('frontend.image.show', ['image' => $image]),
    ]);
}
```

### Processing HTML Content (BuilderService)

```php
use Bongo\Image\Services\BuilderService;

// In controller or service
public function savePost(Request $request)
{
    $post = Post::create($request->validated());

    // Extract base64 images from content
    $builder = new BuilderService($post, 'posts/images');
    $post->content = $builder->process(); // Updates paths in content
    $post->save();

    return $post;
}
```

## Common Patterns

### Eager Loading Images

```php
// Avoid N+1 queries
$posts = Post::with('images')->get();

// With specific order
$posts = Post::with(['images' => function($query) {
    $query->orderBy('sort_order')->limit(5);
}])->get();
```

### Image URLs in Blade

```php
{{-- Primary image with preset --}}
<img src="{{ $post->getPrimaryImage(['preset' => 'large']) }}" alt="{{ $post->title }}">

{{-- Custom dimensions --}}
<img src="{{ $post->getPrimaryImage(['w' => 800, 'h' => 600, 'mode' => 'crop']) }}">

{{-- Placeholder fallback --}}
<img src="{{ $post->getPlaceholder(['preset' => 'medium']) }}">

{{-- Direct image route --}}
@if($post->hasPrimaryImage())
    <img src="{{ route('frontend.image.show', ['image' => $post->primaryImage(), 'preset' => 'large']) }}">
@endif
```

### Path Configuration

```php
// Always use config for paths
$publicPath = config('image.public_path');    // 'public/'
$tmpPath = config('image.tmp_path');          // 'public/tmp/'
$cachePath = config('image.cache_path');      // 'public/cache/'

// Clean paths consistently
$path = ltrim($path, '/');
$path = rtrim($path, '/');
$fullPath = config('image.public_path') . $path . '/';
```

### Image Type Constants

```php
// Always use constants
Image::AVATAR           // 'avatar'
Image::COVER_IMAGE      // 'cover_image'
Image::UPLOAD           // 'upload'
Image::WYSIWYG          // 'wysiwyg'

// Orientation constants
Image::LANDSCAPE        // 'landscape'
Image::PORTRAIT         // 'portrait'
```

## Testing Patterns

```php
use Bongo\Image\Models\Image;
use Bongo\Image\Traits\HasImages;
use Illuminate\Http\UploadedFile;

public function test_image_upload()
{
    $file = UploadedFile::fake()->image('test.jpg', 800, 600);

    $response = $this->post('/admin/images/store', [
        'file' => $file,
    ]);

    $response->assertJson(['url' => true]);
    Storage::assertExists(config('image.tmp_path') . $file->hashName());
}

public function test_has_images_trait()
{
    $post = Post::factory()->create();
    $image = Image::factory()->create();

    $post->images()->attach($image);

    $this->assertTrue($post->hasImages());
    $this->assertEquals($image->id, $post->primaryImage()->id);
}
```

## Configuration Reference

```php
// config/image.php
return [
    'prefix' => 'photos',              // URL prefix for frontend routes
    'public_path' => 'public/',        // Base storage path
    'tmp_path' => 'public/tmp/',       // Temporary uploads
    'cache_path' => 'public/cache/',   // Cached manipulations
    'quality' => 100,                  // Default JPEG quality
    'presets' => [
        'thumb' => 150,
        'small' => 480,
        'medium' => 720,
        'large' => 1600,
        'full' => 1920,
    ],
    'browser_cache' => 86400,          // Cache-Control max-age
    'suffix' => '__',                  // Filename suffix separator
];
```

## Route Patterns

```php
// Frontend (public image delivery)
GET /photos/{image_name}?preset=large&q=80&mode=resize
// Route name: frontend.image.show

// Backend (admin panel)
GET /admin/images                      // backend.image.index
POST /admin/images/store               // backend.image.store
POST /admin/images/copy                // backend.image.copy (base64)
POST /admin/images/{image}/update      // backend.image.update
DELETE /admin/images/{image}/delete    // backend.image.destroy

// API (authenticated)
GET /api/images                        // api.image.index
POST /api/images/store                 // api.image.store
POST /api/images/update                // api.image.update
POST /api/images/delete                // api.image.destroy
```

## Key Files

- `Models/Image.php` - Core image model
- `Traits/HasImages.php` - Polymorphic relationships
- `Services/ImageManipulator.php` - Image manipulation
- `Controllers/Frontend/ImageController.php` - Public delivery
- `Services/BuilderService.php` - HTML base64 extraction
- `Config/image.php` - Configuration
