# Architecture Documentation - Bongo Image Package

## Overview

The Bongo Image package provides comprehensive image management for Laravel applications, including uploads, on-the-fly manipulation with caching, and polymorphic relationships. It uses Intervention Image for manipulation and extends the Bongo Framework's `AbstractServiceProvider` for automatic bootstrapping.

## Directory Structure

```
src/
├── Commands/
│   └── UpdateDimensionsCommand.php          # CLI command to update missing image dimensions
│
├── Config/
│   └── image.php                            # Package configuration
│
├── Exceptions/
│   ├── BuilderException.php                 # BuilderService-specific exception
│   ├── DropzoneException.php                # Dropzone upload exception
│   ├── FileNotFoundException.php            # File not found exception
│   ├── InvalidArgumentException.php         # Invalid argument exception
│   └── WysiwygException.php                 # WYSIWYG-specific exception
│
├── Http/
│   ├── Controllers/
│   │   ├── Api/ImageController.php          # API endpoints (auth:sanctum)
│   │   ├── Backend/ImageController.php      # Admin panel endpoints (auth + employee)
│   │   └── Frontend/ImageController.php     # Public image delivery endpoint
│   └── Requests/
│       ├── StoreImageRequest.php            # Validation for creating images
│       └── UpdateImageRequest.php           # Validation for updating images
│
├── Interfaces/
│   └── Imageable.php                        # Interface for models with images
│
├── Migrations/
│   ├── 2021_01_01_000001_create_images_table.php
│   ├── 2021_01_01_000002_create_imageables_table.php
│   ├── 2021_01_01_000003_add_uuid_column_to_images_table.php
│   ├── 2021_01_01_000004_add_orientation_column_to_images_table.php
│   └── 2021_01_01_000005_move_sort_order_column_to_imageables_table.php
│
├── Models/
│   ├── Image.php                            # Main image model (images table)
│   └── Imageable.php                        # Polymorphic pivot model (imageables table)
│
├── Routes/
│   ├── api.php                              # API routes → /api/images
│   ├── backend.php                          # Admin routes → /admin/images
│   └── frontend.php                         # Public routes → /photos/{image}
│
├── Seeders/
│   └── PackageSeeder.php                    # Package seeder
│
├── Services/
│   ├── AvatarImage.php                      # Single avatar upload service
│   ├── BuilderService.php                   # Extract base64 images from HTML
│   ├── CoverImage.php                       # Single cover image upload service
│   ├── ImageManipulator.php                 # Intervention Image wrapper
│   ├── ImagePlaceholder.php                 # Generate default placeholder images
│   └── WysiwygImage.php                     # WYSIWYG editor upload service
│
├── Traits/
│   └── HasImages.php                        # Polymorphic relationship trait for models
│
├── Translations/
│   └── en/image.php                         # English translations
│
├── Views/
│   └── backend/index.blade.php              # Admin image browser view
│
└── ImageServiceProvider.php                 # Service provider (extends AbstractServiceProvider)
```

## Database Schema

### images Table

```
┌──────────────────────────────────────────────┐
│              images                          │
├──────────────────┬──────────────┬────────────┤
│ Column           │ Type         │ Purpose    │
├──────────────────┼──────────────┼────────────┤
│ id               │ bigint       │ PK         │
│ uuid             │ uuid         │ UUID       │
│ name             │ varchar(255) │ Filename   │
│ title            │ varchar(255) │ Title      │
│ width            │ int          │ Pixels     │
│ height           │ int          │ Pixels     │
│ orientation      │ varchar(20)  │ l/p        │
│ path             │ varchar(255) │ Directory  │
│ ext              │ varchar(10)  │ Extension  │
│ size             │ int          │ Bytes      │
│ type             │ varchar(50)  │ Type*      │
│ created_by       │ bigint       │ User FK    │
│ updated_by       │ bigint       │ User FK    │
│ created_at       │ timestamp    │            │
│ updated_at       │ timestamp    │            │
│ deleted_at       │ timestamp    │ Soft Del   │
└──────────────────┴──────────────┴────────────┘

* Types: avatar, cover_image, upload, wysiwyg
```

### imageables Table (Polymorphic Pivot)

```
┌──────────────────────────────────────────────┐
│           imageables                         │
├──────────────────┬──────────────┬────────────┤
│ Column           │ Type         │ Purpose    │
├──────────────────┼──────────────┼────────────┤
│ id               │ bigint       │ PK         │
│ image_id         │ bigint       │ FK         │
│ imageable_id     │ bigint       │ Poly ID    │
│ imageable_type   │ varchar(255) │ Poly Type  │
│ sort_order       │ int          │ Order      │
│ created_at       │ timestamp    │            │
│ updated_at       │ timestamp    │            │
└──────────────────┴──────────────┴────────────┘
```

## Class Diagrams

### Core Models

```
┌─────────────────────────────────────────────────┐
│              Image (Model)                       │
├─────────────────────────────────────────────────┤
│ + AVATAR: string = 'avatar'                     │
│ + COVER_IMAGE: string = 'cover_image'           │
│ + UPLOAD: string = 'upload'                     │
│ + WYSIWYG: string = 'wysiwyg'                   │
│ + LANDSCAPE: string = 'landscape'               │
│ + PORTRAIT: string = 'portrait'                 │
├─────────────────────────────────────────────────┤
│ + imageables(): HasMany                         │
│ + scopeOfType($query, $type)                    │
│ + scopeForImageableId($query, $id)              │
│ + getNameFormattedAttribute(): ?string          │
│ + getFileName(): string                         │
│ + getFilePath(): string                         │
│ + getPublicFilePath(): string                   │
│ + getFileSrc(): string                          │
│ + getFileUrl(): string                          │
│ + getFileData(): string                         │
│ + fileExists(): string                          │
│ + deleteFile(): string                          │
│ + renameFile(string $newFileName): void         │
│ + getFileExtension(): string                    │
│ + getCachedFilePath(): string                   │
│ + getCachedFileName(array $args): string        │
│ + getCachedFile(array $args): string            │
│ + getMimeType(): string                         │
│ + getPresetByKey(string $key): array            │
│ + isPortrait(): bool                            │
│ + isLandscape(): bool                           │
└─────────────────────────────────────────────────┘
          │
          │ hasMany
          ▼
┌─────────────────────────────────────────────────┐
│          Imageable (Pivot Model)                 │
├─────────────────────────────────────────────────┤
│ Columns: image_id, imageable_id,                │
│          imageable_type, sort_order             │
└─────────────────────────────────────────────────┘
```

### Traits and Interfaces

```
┌─────────────────────────────────────────────────┐
│         Imageable (Interface)                    │
├─────────────────────────────────────────────────┤
│ + images(): MorphToMany                         │
└─────────────────────────────────────────────────┘
          ▲
          │ implements
          │
┌─────────────────────────────────────────────────┐
│         HasImages (Trait)                        │
├─────────────────────────────────────────────────┤
│ + images(): MorphToMany                         │
│ + hasImages(): bool                             │
│ + primaryImage(): ?Image                        │
│ + hasPrimaryImage(): bool                       │
│ + getPrimaryImage(array $args): string          │
│ + secondaryImage(): ?Image                      │
│ + hasSecondaryImage(): bool                     │
│ + getSecondaryImage(array $args): string        │
│ + thumbnails(int $limit = 12)                   │
│ + getThumbnail(int $offset): string             │
│ + getPlaceholder(array $args): string           │
└─────────────────────────────────────────────────┘
          ▲
          │ use
          │
┌─────────────────────────────────────────────────┐
│          Any Model (e.g., Post)                  │
├─────────────────────────────────────────────────┤
│ use HasImages;                                  │
│                                                 │
│ // Polymorphic relationship automatically       │
│ // available via the trait                      │
└─────────────────────────────────────────────────┘
```

### Service Classes Hierarchy

```
┌─────────────────────────────────────────────────┐
│        ImageManipulator (Service)                │
├─────────────────────────────────────────────────┤
│ - image: Image                                  │
│ - quality: int                                  │
│ - width: ?int                                   │
│ - height: ?int                                  │
├─────────────────────────────────────────────────┤
│ + resize(): ImageManipulator                    │
│ + crop(): ImageManipulator                      │
│ + fit(): ImageManipulator                       │
│ + stream(): StreamInterface                     │
│ + setImage(string $fileData): self              │
│ + setMode(string $mode): self                   │
│ + setOrientation(string $orientation): self     │
│ + setQuality(int $quality): self                │
│ + setPreset(string $preset): self               │
│ + setWidth(int $width): self                    │
│ + setHeight(int $height): self                  │
└─────────────────────────────────────────────────┘

┌─────────────────────────────────────────────────┐
│        ImagePlaceholder (Service)                │
├─────────────────────────────────────────────────┤
│ - width: ?int                                   │
│ - height: ?int                                  │
│ - fileName: string                              │
│ - cachedFile: ?string                           │
├─────────────────────────────────────────────────┤
│ + __construct(array $args)                      │
│ + get(): string                                 │
│ + url(): string                                 │
│ # setDimensions(array $args): void              │
│ # setCachedFile(): void                         │
└─────────────────────────────────────────────────┘

┌─────────────────────────────────────────────────┐
│         AvatarImage (Service)                    │
├─────────────────────────────────────────────────┤
│ - file: UploadedFile                            │
│ - user: Authenticatable                         │
├─────────────────────────────────────────────────┤
│ + __construct(Authenticatable, UploadedFile)    │
│ + save(): Authenticatable                       │
│ # delete(): void                                │
│ - getAvatarPath(): string                       │
└─────────────────────────────────────────────────┘

┌─────────────────────────────────────────────────┐
│         CoverImage (Service)                     │
├─────────────────────────────────────────────────┤
│ - file: UploadedFile                            │
│ - entity: Imageable                             │
├─────────────────────────────────────────────────┤
│ + __construct(Imageable, UploadedFile)          │
│ + save(): Imageable                             │
│ # delete(): void                                │
│ - getEntityPath(): string                       │
└─────────────────────────────────────────────────┘

┌─────────────────────────────────────────────────┐
│        WysiwygImage (Service)                    │
├─────────────────────────────────────────────────┤
│ - file: UploadedFile                            │
│ - entity: Imageable                             │
├─────────────────────────────────────────────────┤
│ + __construct(Imageable, UploadedFile)          │
│ + save(): Model                                 │
│ - getEntityPath(): string                       │
└─────────────────────────────────────────────────┘

┌─────────────────────────────────────────────────┐
│        BuilderService (Service)                  │
├─────────────────────────────────────────────────┤
│ - domDocument: DOMDocument                      │
│ - imagePath: ?string                            │
│ - images: DOMNodeList                           │
│ - model: Model                                  │
│ - relativePath: ?string                         │
├─────────────────────────────────────────────────┤
│ + __construct($model, $imagePath)               │
│ + process(): ?string                            │
│ # validateModel(): void                         │
│ # convertHtmlToDomObjects(): void               │
│ # extractImagesFromHtml(): void                 │
│ # setDirectoryPaths(): void                     │
│ # processImages(): void                         │
│ # processImage($htmlImage): void                │
│ # storeImageDetailsInDatabase(...): mixed       │
│ - getImageSrcAttribute($image): mixed           │
│ - createANewUniqueFileName($ext): ?string       │
│ - updateImageSrcAttribute($src, $img): void     │
└─────────────────────────────────────────────────┘
```

## Lifecycle Diagrams

### Image Upload Flow (Single-Image Types: Avatar/Cover)

```
┌─────────────┐
│   Client    │
│   Request   │
└──────┬──────┘
       │
       │ POST /admin/images/{endpoint}
       │ with UploadedFile
       │
       ▼
┌─────────────────────────────┐
│  Controller (Backend/Api)   │
│  - Validates request        │
└──────┬──────────────────────┘
       │
       │ Instantiate service
       │ new AvatarImage($user, $file)
       │
       ▼
┌──────────────────────────────┐
│   AvatarImage Service        │
│   1. Delete existing images  │
│   2. Generate filename       │
│   3. Store file to disk      │
│   4. Create Image record     │
│   5. Get dimensions          │
│   6. Save to database        │
└──────┬───────────────────────┘
       │
       │ Returns: Authenticatable
       │
       ▼
┌─────────────────────────────┐
│      Controller             │
│  - Returns JSON response    │
│  - Includes image URL       │
└─────────────────────────────┘
```

### Image Retrieval and Manipulation Flow

```
┌─────────────┐
│   Client    │
│   Request   │
└──────┬──────┘
       │
       │ GET /photos/{image}?preset=large&q=80
       │
       ▼
┌────────────────────────────────────────┐
│  Frontend\ImageController::show        │
│  1. Validate query params              │
│  2. Get cached file path               │
└──────┬─────────────────────────────────┘
       │
       │ Check if cached file exists
       │
   ┌───┴───┐
   │       │
   │ Yes   │ No
   │       │
   ▼       ▼
┌──────┐  ┌──────────────────────────────────┐
│Return│  │  Create ImageManipulator         │
│Cache │  │  1. Set image data               │
└──────┘  │  2. Set dimensions (preset/w/h)  │
          │  3. Set quality                  │
          │  4. Apply manipulation:          │
          │     - resize() / crop() / fit()  │
          │  5. Stream to cache              │
          └──────┬───────────────────────────┘
                 │
                 │ File now cached
                 │
                 ▼
          ┌─────────────────────────────┐
          │  Stream Response            │
          │  - Set Content-Type         │
          │  - Set Cache-Control        │
          │  - Stream file data         │
          └─────────────────────────────┘
```

### BuilderService Flow (Extract Base64 from HTML)

```
┌─────────────┐
│   Client    │
│   Submits   │
│   Content   │
└──────┬──────┘
       │
       │ POST with HTML content
       │ (contains base64 images)
       │
       ▼
┌───────────────────────────────────────┐
│  Controller                           │
│  1. Create/update model with content  │
│  2. Instantiate BuilderService        │
│     new BuilderService($model, $path) │
└──────┬────────────────────────────────┘
       │
       │ Call process()
       │
       ▼
┌───────────────────────────────────────────┐
│  BuilderService::process()                │
│  1. Validate model (has content field)    │
│  2. Validate model (has images() method)  │
└──────┬────────────────────────────────────┘
       │
       ▼
┌───────────────────────────────────────────┐
│  3. Convert HTML to DOMDocument           │
└──────┬────────────────────────────────────┘
       │
       ▼
┌───────────────────────────────────────────┐
│  4. Extract <img> tags                    │
└──────┬────────────────────────────────────┘
       │
       ▼
┌───────────────────────────────────────────┐
│  5. Create directory path                 │
│     {public_path}/{imagePath}/{model.id}/ │
└──────┬────────────────────────────────────┘
       │
       │ For each <img> tag:
       │
       ▼
┌───────────────────────────────────────────┐
│  6. Check if src is base64                │
│     - Decode base64 data                  │
│     - Generate unique filename            │
│     - Store file to directory             │
│     - Create Image record                 │
│     - Get dimensions & orientation        │
│     - Save to database                    │
│     - Replace src in HTML with new path   │
└──────┬────────────────────────────────────┘
       │
       │ Returns: Updated HTML content
       │
       ▼
┌───────────────────────────────────────────┐
│  Controller                               │
│  - Save updated content to model          │
│  - Return response                        │
└───────────────────────────────────────────┘
```

### HasImages Trait Usage Pattern

```
┌─────────────────────────────────────┐
│          Any Model                  │
│          (e.g., Post)               │
│                                     │
│          use HasImages;             │
└──────┬──────────────────────────────┘
       │
       │ Access images relationship
       │
       ▼
┌─────────────────────────────────────┐
│  $post->images()                    │
│  → MorphToMany relationship         │
│  → Returns Query Builder            │
└──────┬──────────────────────────────┘
       │
       ├──→ primaryImage()
       │    → First image (by type, then sort_order)
       │    → Returns: ?Image
       │
       ├──→ secondaryImage()
       │    → Second image (offset 1)
       │    → Returns: ?Image
       │
       ├──→ thumbnails($limit)
       │    → Skip first, take $limit
       │    → Returns: Collection
       │
       ├──→ getPrimaryImage($args)
       │    → URL with manipulation args
       │    → Returns: string (URL or placeholder)
       │
       └──→ getPlaceholder($args)
            → Placeholder URL
            → Returns: string
```

## Service Provider Bootstrap

The `ImageServiceProvider` extends `Bongo\Framework\Providers\AbstractServiceProvider`:

```php
class ImageServiceProvider extends AbstractServiceProvider
{
    protected string $module = 'image';

    protected array $commands = [
        UpdateDimensionsCommand::class,
    ];
}
```

### Automatic Bootstrapping

The parent `AbstractServiceProvider` automatically:

1. **Config Registration**: Loads `src/Config/image.php` → `config('image.*')`
2. **Route Registration**:
   - `src/Routes/api.php` → Prefix `/api/images`, name `api.image.*`, middleware `auth:sanctum`
   - `src/Routes/backend.php` → Prefix `/admin/images`, name `backend.image.*`, middleware `auth`, `employee`
   - `src/Routes/frontend.php` → Name `frontend.image.*`, no prefix (custom)
3. **View Registration**: Loads `src/Views/` → `view('image::*')`
4. **Migration Registration**: Auto-loads all files from `src/Migrations/`
5. **Translation Registration**: Loads `src/Translations/` → `trans('image::*')`
6. **Command Registration**: Registers commands from `$commands` array

## Extension Points

### Adding a New Image Type

1. **Add constant to Image model**:
```php
// src/Models/Image.php
public const NEW_TYPE = 'new_type';
```

2. **Create service class**:
```php
// src/Services/NewTypeImage.php
class NewTypeImage
{
    protected UploadedFile $file;
    protected Imageable $entity;

    public function save(): Imageable { /* ... */ }
    protected function delete(): void { /* ... */ }
}
```

3. **Add controller endpoint** (if needed):
```php
// src/Http/Controllers/Backend/ImageController.php
public function uploadNewType(Request $request, $entity)
{
    $service = new NewTypeImage($entity, $request->file('image'));
    return $service->save();
}
```

### Adding a New Manipulation Mode

1. **Add method to ImageManipulator**:
```php
// src/Services/ImageManipulator.php
public function newMode(): ImageManipulator
{
    $this->image->someOperation($this->width, $this->height);
    return $this;
}
```

2. **Update Frontend controller**:
```php
// src/Http/Controllers/Frontend/ImageController.php
elseif ($this->has('mode', $args) && $args['mode'] === 'newMode') {
    $imageManipulator->newMode();
}
```

3. **Update cached filename generation**:
```php
// src/Models/Image.php
public function getCachedFileName(array $args): string
{
    // Add new mode abbreviation
    if (isset($args['mode']) && $args['mode'] == 'newMode') {
        $m = 'nm';
    }
    // ...
}
```

### Extending HasImages Trait

Add custom methods in your model:

```php
class Post extends Model
{
    use HasImages;

    public function featuredImage(): ?Image
    {
        return $this->images()
            ->where('type', Image::COVER_IMAGE)
            ->first();
    }

    public function galleryImages()
    {
        return $this->images()
            ->where('type', Image::UPLOAD)
            ->orderBy('sort_order');
    }
}
```

### Custom Image Presets

Add to config:

```php
// config/image.php
'presets' => [
    'thumb' => 150,
    'small' => 480,
    'xlarge' => 2400,     // New preset
    'hero' => 3000,       // New preset
],
```

Use in views:

```php
{{ $post->getPrimaryImage(['preset' => 'xlarge']) }}
```

## Configuration Reference

### Key Configuration Options

| Option | Type | Default | Purpose |
|--------|------|---------|---------|
| `prefix` | string | `'photos'` | URL prefix for frontend image routes |
| `public_path` | string | `'public/'` | Base storage directory for images |
| `tmp_path` | string | `'public/tmp/'` | Temporary upload directory |
| `cache_path` | string | `'public/cache/'` | Cached/manipulated images directory |
| `quality` | int | `100` | Default JPEG quality (1-100) |
| `presets` | array | See below | Named size presets |
| `browser_cache` | int | `86400` | Cache-Control max-age in seconds |
| `suffix` | string | `'__'` | Filename suffix separator |

### Default Presets

| Preset | Width (px) |
|--------|-----------|
| thumb | 150 |
| small | 480 |
| medium | 720 |
| large | 1600 |
| full | 1920 |

## Route Reference

### Frontend Routes

```
GET /photos/{image:name}
    Name: frontend.image.show
    Controller: Frontend\ImageController@show
    Params: ?preset=large&w=800&h=600&q=80&mode=resize
```

### Backend Routes

```
GET    /admin/images                     → backend.image.index
POST   /admin/images/store               → backend.image.store
POST   /admin/images/copy                → backend.image.copy
POST   /admin/images/{image}/update      → backend.image.update
DELETE /admin/images/{image}/delete      → backend.image.destroy
```

### API Routes

```
GET    /api/images                       → api.image.index
POST   /api/images/store                 → api.image.store
POST   /api/images/update                → api.image.update
POST   /api/images/delete                → api.image.destroy
```

## Command Reference

### image:update_dimensions

Updates width, height, and orientation for images that are missing these values.

```bash
php artisan image:update_dimensions
```

**Process:**
1. Queries images where width OR height is null
2. Reads file from storage
3. Gets dimensions using `getimagesize()`
4. Calculates orientation (landscape vs portrait)
5. Saves updated image record

## Dependencies

### External Dependencies

- `intervention/image` (^2.6) - Image manipulation library
- `illuminate/contracts` (^10.0) - Laravel contracts
- PHP GD or Imagick extension (required by Intervention Image)

### Internal Dependencies

- `bongo/framework` (^3.0) - Provides:
  - `AbstractServiceProvider` - Auto-bootstrapping
  - `AbstractModel` - Base model class
  - `AbstractController` - Base controller class
  - `File` helper - Filename generation
  - `HasUUID` trait - UUID support

## Testing Strategy

### Unit Tests

- Model methods (getFileName, getCachedFileName, etc.)
- Service class methods
- Trait methods (HasImages)

### Feature Tests

- Image upload endpoints
- Image manipulation and caching
- Polymorphic relationship attachment
- BuilderService HTML processing
- Placeholder generation

### Integration Tests

- Full upload-to-retrieval flow
- Multi-image uploads
- Base64 extraction from content
- Cache invalidation

## Performance Considerations

1. **Eager Loading**: Always eager load images to avoid N+1 queries
2. **Caching**: Generated images are cached indefinitely until manually cleared
3. **Memory Limit**: ImageManipulator respects `settings.memory_limit` config
4. **Browser Caching**: 24-hour cache headers on frontend delivery
5. **Lazy Generation**: Images only manipulated on first request

## Security Considerations

1. **Validation**: All uploads validated for file type and size
2. **Authorization**: Backend/API routes protected by auth middleware
3. **Path Traversal**: Paths cleaned with ltrim/rtrim
4. **Soft Deletes**: Images soft-deleted to prevent data loss
5. **UUID Support**: Images have UUIDs for non-sequential IDs
