# Bongo Image Package

[![Latest Version](https://img.shields.io/badge/version-3.0-blue.svg)](https://github.com/designtec/image)
[![PHP Version](https://img.shields.io/badge/php-8.2%2B-purple.svg)](https://php.net)
[![Laravel Version](https://img.shields.io/badge/laravel-10%2B-red.svg)](https://laravel.com)

Comprehensive image management package for Laravel applications, providing uploads, on-the-fly manipulation with caching, and polymorphic relationships.

## Features

- **Polymorphic Image Relationships** - Attach images to any model via `HasImages` trait
- **On-the-Fly Manipulation** - Resize, crop, and fit images dynamically using Intervention Image
- **Intelligent Caching** - Generated images cached indefinitely with browser cache headers
- **Multiple Upload Types** - Avatar, cover images, standard uploads, and WYSIWYG editor uploads
- **Base64 Extraction** - Automatically extract and store base64 images from HTML content
- **Placeholder Generation** - Automatic placeholder images when no image exists
- **Configurable Presets** - Named size presets (thumb, small, medium, large, full)
- **RESTful APIs** - Frontend, backend, and API endpoints included
- **Soft Deletes** - Images soft-deleted to prevent data loss
- **UUID Support** - Non-sequential image IDs

## Requirements

- PHP 8.2+
- Laravel 10+
- `bongo/framework` ^3.0
- `intervention/image` ^2.6
- PHP GD or Imagick extension

## Installation

### Via Composer

```bash
composer require bongo/image
```

### Laravel Auto-Discovery

The package uses Laravel's auto-discovery feature, so the service provider will be automatically registered.

### Manual Registration (if needed)

If auto-discovery is disabled, add to `config/app.php`:

```php
'providers' => [
    // ...
    Bongo\Image\ImageServiceProvider::class,
],
```

### Publish Configuration

```bash
php artisan vendor:publish --provider="Bongo\Image\ImageServiceProvider"
```

### Run Migrations

```bash
php artisan migrate
```

## Quick Start

### 1. Add Trait to Models

```php
use Bongo\Image\Traits\HasImages;
use Illuminate\Database\Eloquent\Model;

class Post extends Model
{
    use HasImages;
}
```

### 2. Upload Images

```php
use Bongo\Image\Services\WysiwygImage;

// In your controller
$wysiwyg = new WysiwygImage($post, $request->file('image'));
$image = $wysiwyg->save();
```

### 3. Display Images

```blade
{{-- In your Blade views --}}
<img src="{{ $post->getPrimaryImage(['preset' => 'large']) }}" alt="{{ $post->title }}">

{{-- With custom dimensions --}}
<img src="{{ $post->getPrimaryImage(['w' => 800, 'h' => 600, 'mode' => 'crop']) }}">
```

## Usage

### Attaching Images to Models

Use the `HasImages` trait on any model:

```php
class Product extends Model
{
    use HasImages;
}
```

This provides access to:

```php
$product->images();                          // MorphToMany relationship
$product->primaryImage();                    // First image
$product->secondaryImage();                  // Second image
$product->thumbnails($limit);                // Skip first, take $limit
$product->hasImages();                       // Check if images exist
$product->getPrimaryImage(['preset' => 'large']);  // Get image URL
$product->getPlaceholder(['w' => 800]);      // Get placeholder URL
```

### Upload Services

#### Avatar Upload (Single, Replaces Existing)

```php
use Bongo\Image\Services\AvatarImage;

$avatar = new AvatarImage(auth()->user(), $request->file('avatar'));
$user = $avatar->save();
```

#### Cover Image Upload (Single, Replaces Existing)

```php
use Bongo\Image\Services\CoverImage;

$cover = new CoverImage($post, $request->file('cover'));
$post = $cover->save();
```

#### WYSIWYG Upload (Multiple Allowed)

```php
use Bongo\Image\Services\WysiwygImage;

$wysiwyg = new WysiwygImage($post, $request->file('file'));
$image = $wysiwyg->save();
```

#### Extract Base64 Images from HTML

```php
use Bongo\Image\Services\BuilderService;

$post = Post::create($request->validated());

$builder = new BuilderService($post, 'posts/images');
$post->content = $builder->process();  // Extracts base64, updates paths
$post->save();
```

### Image Manipulation

Images are manipulated on-the-fly via URL parameters:

```php
// Using presets
route('frontend.image.show', ['image' => $image, 'preset' => 'large'])

// Custom dimensions
route('frontend.image.show', ['image' => $image, 'w' => 800, 'h' => 600])

// With quality and mode
route('frontend.image.show', [
    'image' => $image,
    'w' => 800,
    'h' => 600,
    'q' => 80,
    'mode' => 'crop'
])
```

**Available Modes:**
- `resize` - Maintain aspect ratio, constrain to dimensions (default)
- `crop` - Crop to exact dimensions
- `fit` - Fit to dimensions, may add padding
- `resizeCrop` - Resize then crop
- `resizeFit` - Resize then fit

**Query Parameters:**
- `preset` - Use named preset (thumb, small, medium, large, full)
- `w` - Width in pixels (1-3000)
- `h` - Height in pixels (1-3000)
- `q` - Quality (1-100)
- `mode` - Manipulation mode

### Blade Templates

```blade
{{-- Primary image with preset --}}
<img src="{{ $post->getPrimaryImage(['preset' => 'large']) }}" alt="{{ $post->title }}">

{{-- Check before displaying --}}
@if($post->hasPrimaryImage())
    <img src="{{ $post->getPrimaryImage(['preset' => 'medium']) }}">
@endif

{{-- Placeholder fallback --}}
<img src="{{ $post->getPlaceholder(['preset' => 'medium']) }}">

{{-- Custom dimensions with crop --}}
<img src="{{ $post->getPrimaryImage(['w' => 800, 'h' => 600, 'mode' => 'crop', 'q' => 80]) }}">

{{-- Gallery/thumbnails --}}
@foreach($post->thumbnails(12) as $thumbnail)
    <img src="{{ route('frontend.image.show', ['image' => $thumbnail, 'preset' => 'thumb']) }}">
@endforeach
```

### Direct Image Model Usage

```php
use Bongo\Image\Models\Image;

// Query by type
$avatars = Image::ofType(Image::AVATAR)->get();

// Get file operations
$image->getFileName();         // Filename
$image->getFilePath();         // Relative path
$image->getFileSrc();          // Absolute filesystem path
$image->getFileUrl();          // Public URL
$image->fileExists();          // Check if file exists
$image->deleteFile();          // Delete physical file

// Cached file paths
$cachedFile = $image->getCachedFile(['preset' => 'large']);
```

## Configuration

The package publishes a config file at `config/image.php`:

```php
return [
    // URL prefix for frontend image routes
    'prefix' => 'photos',

    // Storage paths
    'public_path' => 'public/',
    'tmp_path' => 'public/tmp/',
    'cache_path' => 'public/cache/',

    // Default JPEG quality (1-100)
    'quality' => 100,

    // Named size presets (in pixels)
    'presets' => [
        'thumb' => 150,
        'small' => 480,
        'medium' => 720,
        'large' => 1600,
        'full' => 1920,
    ],

    // Browser cache duration (seconds)
    'browser_cache' => 86400,

    // Filename suffix separator
    'suffix' => '__',
];
```

## Routes

### Frontend (Public Image Delivery)

```
GET /photos/{image}?preset=large
```

Route name: `frontend.image.show`

### Backend (Admin Panel)

All routes prefixed with `/admin/images`, protected by `auth` and `employee` middleware:

```
GET    /admin/images                 - Image browser
POST   /admin/images/store           - Upload image
POST   /admin/images/copy            - Upload base64 image
POST   /admin/images/{image}/update  - Update image
DELETE /admin/images/{image}/delete  - Delete image
```

### API (Authenticated)

All routes prefixed with `/api/images`, protected by `auth:sanctum` middleware:

```
GET    /api/images        - List images
POST   /api/images/store  - Upload image
POST   /api/images/update - Update image
POST   /api/images/delete - Delete image
```

## Artisan Commands

### Update Image Dimensions

Updates width, height, and orientation for images missing these values:

```bash
php artisan image:update_dimensions
```

## Database Structure

### images Table

Stores image metadata and file information.

**Columns:**
- `id`, `uuid` - Identifiers
- `name`, `title` - Filename and display title
- `width`, `height`, `orientation` - Dimensions and orientation (landscape/portrait)
- `path`, `ext`, `size` - File location, extension, and size
- `type` - Image type (avatar, cover_image, upload, wysiwyg)
- `created_by`, `updated_by` - User tracking
- Timestamps and soft deletes

### imageables Table

Polymorphic pivot table for attaching images to any model.

**Columns:**
- `image_id` - Foreign key to images
- `imageable_id`, `imageable_type` - Polymorphic relationship
- `sort_order` - Order for multiple images
- Timestamps

## Image Types

The package defines four image types:

| Type | Constant | Usage | Behaviour |
|------|----------|-------|-----------|
| Avatar | `Image::AVATAR` | User avatars | Single image (replaces existing) |
| Cover Image | `Image::COVER_IMAGE` | Cover/featured images | Single image (replaces existing) |
| Upload | `Image::UPLOAD` | Standard uploads | Multiple images allowed |
| WYSIWYG | `Image::WYSIWYG` | Editor images | Multiple images allowed |

## Performance

- **Lazy Generation**: Images manipulated only on first request
- **Caching**: Generated images cached indefinitely
- **Browser Caching**: 24-hour cache headers on image delivery
- **Memory Limit**: Respects `settings.memory_limit` config for large images
- **Eager Loading**: Use `->with('images')` to avoid N+1 queries

## Advanced Usage

### Custom Image Methods

Extend functionality by adding methods to your models:

```php
class Post extends Model
{
    use HasImages;

    public function featuredImage(): ?Image
    {
        return $this->images()
            ->where('type', Image::COVER_IMAGE)
            ->first();
    }

    public function galleryImages()
    {
        return $this->images()
            ->where('type', Image::UPLOAD)
            ->orderBy('sort_order')
            ->limit(10);
    }
}
```

### Eager Loading

Prevent N+1 queries by eager loading images:

```php
// Basic eager loading
$posts = Post::with('images')->get();

// With constraints
$posts = Post::with(['images' => function($query) {
    $query->where('type', Image::UPLOAD)
          ->orderBy('sort_order')
          ->limit(5);
}])->get();
```

### Custom Presets

Add your own presets to the config:

```php
'presets' => [
    'thumb' => 150,
    'small' => 480,
    'xlarge' => 2400,     // Custom preset
    'hero' => 3000,       // Custom preset
],
```

## Testing

```bash
# Install dependencies
composer install

# Run tests
vendor/bin/phpunit

# Check code style
vendor/bin/pint --test

# Fix code style
vendor/bin/pint
```

## Documentation

- [ARCHITECTURE.md](ARCHITECTURE.md) - Detailed architecture, diagrams, and data flows
- [CLAUDE.md](CLAUDE.md) - Claude Code quick reference
- [.cursorrules](.cursorrules) - Cursor AI development guide
- [.github/copilot-instructions.md](.github/copilot-instructions.md) - GitHub Copilot templates

## Contributing

This package is part of the Bongo monorepo. Contributions should follow:

1. PSR-12 coding standards
2. Laravel best practices
3. Comprehensive test coverage
4. Clear commit messages

Run code style fixes before committing:

```bash
vendor/bin/pint
```

## Security

- All uploads validated for file type and size
- Backend/API routes protected by authentication middleware
- Path traversal prevention via path cleaning
- Soft deletes prevent accidental data loss
- UUID support for non-sequential IDs

If you discover any security issues, please email stuart.elliott@bespokeuk.com.

## Credits

- **Author**: Stuart Elliott
- **Company**: [Bespoke UK](https://bespokeuk.com)
- **Framework**: Built on [Bongo Framework](https://github.com/designtec/framework)

## Licence

The MIT Licence (MIT). Please see [Licence File](LICENCE) for more information.

## Related Packages

Part of the Bongo package ecosystem:

- `bongo/framework` - Base framework and service providers
- `cms/builder` - Content builder (uses BuilderService)
- `cms/page` - Page management (uses HasImages)
- `cms/post` - Post management (uses HasImages)
