<?php

namespace Bongo\Image\Http\Controllers\Backend;

use Bongo\Framework\Http\Controllers\AbstractController;
use Bongo\Image\Http\Requests\UpdateImageRequest;
use Bongo\Image\Models\Image;
use Exception;
use Illuminate\Http\JsonResponse;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Log;
use Illuminate\Support\Facades\Storage;
use Illuminate\Support\Str;
use Illuminate\View\View;

class ImageController extends AbstractController
{
    public function __construct()
    {
        // If the tmp path does not exist then create it
        if (! Storage::exists(config('image.tmp_path'))) {
            Storage::makeDirectory(config('image.tmp_path'));
        }
    }

    public function index(): View
    {
        return view('image::backend.index');
    }

    public function store(): JsonResponse
    {
        $file = request()->file('file');
        $extension = $file->getClientOriginalExtension();
        $filename = uniqid(Str::slug($file->getFilename())).'.'.$extension;
        $upload_success = $file->storePubliclyAs(config('image.tmp_path'), $filename);

        if ($upload_success) {
            $id = $filename;
            $url = Storage::url(config('image.tmp_path').$filename);
            $fileLink = Storage::url(config('image.tmp_path').$filename);

            return response()->json([
                'id' => $id,
                'url' => $url,
                'filelink' => $fileLink,
            ]);
        }

        return response()->json('error', 400);
    }

    /**
     * This method allows a user to paste and save a base 64 image
     * into the wysiwyg editor.
     */
    public function copy(): JsonResponse
    {
        $filename = uniqid('image').'.png';
        $base64string = request()->input()->get('data');

        try {
            // Try to move the file
            Storage::put(config('image.tmp_path').$filename, base64_decode($base64string));

            // Return the response
            $id = $filename;
            $url = Storage::url(config('image.tmp_path').$filename);
            $fileLink = Storage::url(config('image.tmp_path').$filename);

            return response()->json([
                'id' => $id,
                'url' => $url,
                'filelink' => $fileLink,
            ]);

        } catch (Exception $e) {
            $error = [
                'message' => $e->getMessage(),
                'file' => $e->getFile(),
                'line_number' => $e->getLine(),
            ];
            Log::info(json_encode($error));

            return response()->json('Error: an error occurred whilst uploading file '.$filename, 400);
        }
    }

    public function update(UpdateImageRequest $request, Image $image): JsonResponse
    {
        $image->update($request->all());

        return response()->json('Image updated');
    }

    public function destroy(Image $image): JsonResponse
    {
        // If the file exists in storage then delete it
        if ($image->fileExists()) {
            $image->deleteFile();
        }

        // Delete model relationship and then the image
        DB::table('imageables')->where('image_id', $image->id)->delete();
        $image->delete();

        return response()->json('Image deleted');
    }
}
