<?php

namespace Bongo\Image\Http\Controllers\Frontend;

use Bongo\Framework\Http\Controllers\AbstractController;
use Bongo\Image\Models\Image;
use Bongo\Image\Services\ImageManipulator;
use Bongo\Image\Services\ImagePlaceholder;
use Exception;
use Illuminate\Http\File;
use Illuminate\Support\Facades\Log;
use Illuminate\Support\Facades\Storage;

class ImageController extends AbstractController
{
    /**
     * @throws Exception
     */
    public function show(Image $image)
    {
        // Get the arguments
        $args = request()->query();

        // Validate the args
        if ($this->has('preset', $args) && ! config("image.presets.{$args['preset']}")) {
            throw new Exception("Preset {$args['preset']} could not be found");
        }
        if ($this->has('w', $args) && ($args['w'] < 1 || $args['w'] > 3000)) {
            throw new Exception("Width {$args['w']} must be between 1-3000");
        }
        if ($this->has('h', $args) && ($args['h'] < 1 || $args['h'] > 3000)) {
            throw new Exception("Height {$args['h']} must be between 1-3000");
        }
        if ($this->has('q', $args) && ($args['q'] < 1 || $args['q'] > 100)) {
            throw new Exception("Quality {$args['q']} must be between 1-100");
        }

        // Get the cached file path/name
        $cachedFile = $image->getCachedFile($args);

        // If the cached file does not exist in the cached folder then create it
        if (! Storage::exists($cachedFile)) {
            try {
                if (config('settings.memory_limit') && ! empty(config('settings.memory_limit'))) {
                    ini_set('memory_limit', config('settings.memory_limit'));
                }

                // Create a new manipulator instance
                $imageManipulator = new ImageManipulator();
                $imageManipulator->setImage($image->getFileData());

                // Do we have a preset?
                if ($this->has('preset', $args)) {
                    $imageManipulator->setWidth(config("image.presets.{$args['preset']}"));

                } elseif ($this->has('w', $args) && $this->has('h', $args)) { // If not do we have both width & height
                    $imageManipulator->setWidth($args['w'])->setHeight($args['h']);

                } elseif ($this->has('w', $args)) { // If not do we have just a width
                    $imageManipulator->setWidth($args['w']);

                } elseif ($this->has('h', $args)) { // If not do we have just a height
                    $imageManipulator->setHeight($args['h']);

                } else { // Well this is awkward, guess we should just use a default then
                    $imageManipulator->setWidth(config("image.presets.large"));
                }

                // Do we need to adjust the quality
                if ($this->has('q', $args)) {
                    $imageManipulator->setQuality($args['q']);
                }

                // Is this a crop, resize or both
                if ($this->has('mode', $args) && $args['mode'] === 'resizeFit') {
                    $imageManipulator->resize()->fit();

                } elseif ($this->has('mode', $args) && $args['mode'] === 'resizeCrop') { // Just a crop?
                    $imageManipulator->crop();

                } elseif ($this->has('mode', $args) && $args['mode'] === 'crop') { // Just a crop?
                    $imageManipulator->crop();

                } else { // Standard resize, how fun
                    $imageManipulator->resize();
                }

                // Save the modified image to cache
                Storage::put($cachedFile, $imageManipulator->stream(), 'public');

            } catch (Exception $e) {
                Log::info($e->getMessage());

                return (new ImagePlaceholder($args))->get();
            }
        }

        // Set the headers
        $file = new File(Storage::path($cachedFile));
        $headers = [
            'Content-Type' => $file->getMimeType(),
            'Content-Length' => $file->getSize(),
            'Last-Modified' => $file->getMTime(),
            'Cache-Control' => 'max-age='.config('image.browser_cache'),
            'Expires' => gmdate('D, d M Y H:i:s \G\M\T', time() + config('image.browser_cache')),
        ];

        // Return a streamed response, be careful changing this, as the headers have to be just right
        return response()->stream(function () use ($cachedFile) {
            echo Storage::get($cachedFile);
        }, 200, $headers);
    }

    private function has(string $key, array $args): bool
    {
        return isset($args[$key]) && ! empty($args[$key]);
    }
}
